/*-------------------------------------------------------------------------
 Copyright 2009 Olivier Berlanger

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 -------------------------------------------------------------------------*/
package net.sf.sfac.file;


import java.io.File;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;


/**
 * Utilities to manipulate file paths as strings. <br>
 * In particular it can convert absolute to relative paths (and vice-versa). This utility class is quite platform-independent
 * beacuse it handles indifferently slash or backslash as path separator. The only platform-specific thing is that it ignores
 * case in path comparison (wich is mandatory on Windows system), this is detected fom the file system or can be passed as a
 * constructor parameter.
 * 
 * @author Olivier Berlanger
 */
public class FilePathUtils {


    private static Log log = LogFactory.getLog(FilePathUtils.class);

    /** Flag saying if file comparison should not be caseSensitive */
    private boolean ignoreCase;


    /** The base drive (null on non-windows system) used a default drive for relative paths */
    private String appBaseDrive;

    /**
     * The base path used a starting point for relative paths. <br>
     * Note that this path also includes the drive (if applicable).
     */
    private String appBasePath;

    private char separatorChar;


    /**
     * Create a FilePathUtils bounded to the given local base directory. The FilePathUtils will use the separator char of the
     * current system.
     * 
     * @param baseDir
     *            The base directory used a starting point for relative paths.
     * @exception IllegalArgumentException
     *                if the given base directory is not a directory of cannot be canonicalised.
     */
    public FilePathUtils(File baseDir) {
        ignoreCase = checkFileSystemIgnoreCase();
        separatorChar = File.separatorChar;
        try {
            appBasePath = baseDir.getCanonicalPath();
            appBaseDrive = getDrive(appBasePath);
        } catch (Exception e) {
            throw new IllegalArgumentException("The base directory '" + baseDir + "' is invalid !", e);
        }
    }


    /**
     * Create a FilePathUtils bounded to the given base directory. You can use this constructor to pass other parameters than
     * the defaults of the current system, for example, if you want to calculate path for a remote system.
     * 
     * @param basePath
     *            The canonicalized base path (with the drive if present). This class will use (when creating paths) the path
     *            separator given in this base reference path.
     * @param ignoreCase
     *            true if the target file system is not case sensitive.
     * @exception IllegalArgumentException
     *                if the given base directory cannot be canonicalised.
     */
    public FilePathUtils(String basePath, boolean ignoreCaseInPaths) {
        ignoreCase = ignoreCaseInPaths;
        separatorChar = (basePath.indexOf('\\') >= 0) ? '\\' : '/';
        try {
            appBaseDrive = getDrive(basePath);
            appBasePath = canonicalize(basePath);
        } catch (Exception e) {
            throw new IllegalArgumentException("The base directory '" + basePath + "' is invalid !", e);
        }
    }


    /**
     * Get the preferred file separator char ('/' or '\') of this FilePathUtils. Though there is one preferred separator char,
     * both '\' and '/' are always accepted in input paths.
     * 
     * @return the preferred file separator char ('/' or '\')
     */
    public char getSparatorChar() {
        return separatorChar;
    }


    /**
     * Check if the local file system is case sensitive.
     */
    private boolean checkFileSystemIgnoreCase() {
        boolean fileSystemIgnoresCase = true;
        boolean checked = false;
        try {
            String tempDir = System.getProperty("java.io.tmpdir");
            File fil = new File(tempDir);
            String filePath = fil.getAbsolutePath();
            String uppercaseFilePath = filePath.toUpperCase();
            String lowercaseFilePath = filePath.toLowerCase();
            if (!lowercaseFilePath.equals(uppercaseFilePath)) {
                File uppercaseFile = new File(uppercaseFilePath);
                File lowercaseFile = new File(lowercaseFilePath);
                fileSystemIgnoresCase = uppercaseFile.exists() && lowercaseFile.exists();
                checked = true;
            }
            if (!checked) {
                log.warn("Cannot check if file system is case-senstive (assume it is not)");
            }
        } catch (Exception e) {
            log.error("Cannot check if file system is case-senstive (assume it is not)", e);
        }
        return fileSystemIgnoresCase;
    }


    /**
     * Get the base path used a starting point for relative paths. <br>
     * Note: this path also includes the drive (if applicable).
     * 
     * @return the base path used a starting point for relative paths.
     */
    public String getBasePath() {
        return appBasePath;
    }


    /**
     * Get an absolute file path for the given path. If the given path is a relative one, it will be added to the base path. The
     * returned path will be canonicalized. Returns null if the given path is null.
     * 
     * @param filePath
     *            the file path to make absolute.
     * @return an absolute file path corresponding to the given one.
     * @exception InvalidPathException
     *                if the given path can not be combined with the base path to produce a canonicalised path.
     */
    public String getAbsoluteFilePath(String filePath) throws InvalidPathException {
        if (filePath == null) return null;
        if (isWhite(filePath) || isThisPath(filePath)) return appBasePath;
        if (!isAbsolute(filePath)) filePath = concatPaths(appBasePath, filePath);
        // be sure that the absolute path starts with a drive name (on Win32)
        if ((appBaseDrive != null) && (getDrive(filePath) == null)) {
            filePath = appBaseDrive + filePath;
        }
        return canonicalize(filePath);
    }


    /**
     * Canonicalize the given absolute path.
     * 
     * @param originalPath
     *            an absolute path to canonicalize.
     * @return the given path canonicalized.
     * @exception InvalidPathException
     *                If the path cannot be canonicalized for any reason.
     */
    private String canonicalize(String originalPath) throws InvalidPathException {
        if (originalPath == null) return null;
        // tokenize the absolutePath
        String absolutePath = originalPath;
        if (separatorChar == '/') absolutePath = absolutePath.replace('\\', '/');
        else absolutePath = absolutePath.replace('/', '\\');
        List<String> items = new ArrayList<String>();
        absolutePath = absolutePath.trim();
        if (startsWithSeparator(absolutePath)) absolutePath = absolutePath.substring(1);
        if (endsWithSeparator(absolutePath)) absolutePath = absolutePath.substring(0, absolutePath.length() - 1);
        int firstPathItem = (appBaseDrive == null) ? 0 : 1;
        int nbrItems;
        int lastIndex = 0;
        int slashIndex = 0;
        String pathItem;
        while (slashIndex >= 0) {
            slashIndex = absolutePath.indexOf(separatorChar, lastIndex);
            if (slashIndex >= 0) pathItem = absolutePath.substring(lastIndex, slashIndex);
            else pathItem = absolutePath.substring(lastIndex);
            // remove '.' and '..'
            if (pathItem.equals(".")) {
                // ignore
            } else if (pathItem.equals("..")) {
                nbrItems = items.size();
                if (nbrItems <= firstPathItem) throw new InvalidPathException("Too many '..' in path '" + originalPath + "'");
                items.remove(nbrItems - 1);
            } else {
                if (items.size() >= firstPathItem) checkPathItem(originalPath, pathItem);
                items.add(pathItem);
            }
            lastIndex = slashIndex + 1;
        }
        // rebuild the absolutePath.
        StringBuffer sb = new StringBuffer();
        if (appBaseDrive == null) sb.append(separatorChar);
        nbrItems = items.size();
        for (int i = 0; i < nbrItems; i++) {
            if (i > 0) sb.append(separatorChar);
            sb.append(items.get(i));
        }
        if ((appBaseDrive != null) && (nbrItems == 1)) sb.append(separatorChar);
        return sb.toString();
    }


    /**
     * Check if a path item is valid. <br>
     * Checked rules:
     * <ul>
     * <li>The path item length cannot be 0
     * <li>The path item cannot contain following chars: ':', '?', '*', '|', '"', '&lt;', '&gt;', '/', '\'
     * </ul>
     * 
     * @param path
     *            The path containing the item to check (only used for exception reporting)
     * @param str
     *            The path item to check.
     * @exception InvalidPathException
     *                If the path item is invalid.
     */
    private void checkPathItem(String path, String str) throws InvalidPathException {
        int len = str.length();
        if (len == 0) throw new InvalidPathException("Empty path item in path '" + path + "'");
        for (int i = 0; i < len; i++) {
            switch (str.charAt(i)) {
                case ':':
                case '?':
                case '*':
                case '|':
                case '"':
                case '<':
                case '>':
                case '/':
                case '\\':
                    throw new InvalidPathException("Illegal character '" + str.charAt(i) + "' in path '" + path + "'");
                default:
                    // valid char
            }
        }
    }


    /**
     * Get the given path expressed as relative to this class base directory (if possible). <br>
     * If the given path is already relative, it's returned without changes. Returns null if the given path is null. <br>
     * Note: the result can be an absolute path, if the given path cannot be expressed as a relative to the base directory (ex:
     * if the given path specify another drive than the base drive).
     * 
     * @param filePath
     *            The path to express as relative to this class base directory.
     * @return The given path expressed as relative to this class base directory.
     */
    public String getRelativeFilePath(String filePath) throws InvalidPathException {
        if (filePath == null) return null;
        if (isWhite(filePath) || isThisPath(filePath)) return ".";
        if (!isAbsolute(filePath)) return filePath;
        // add any missing drive to the absolute file name
        String drive = getDrive(filePath);
        if ((appBaseDrive != null) && (drive == null)) {
            filePath = appBaseDrive + filePath;
        }
        filePath = canonicalize(filePath);
        // start building relative path
        String relativePath;
        int fileLen = filePath.length();
        int baseLen = appBasePath.length();
        if (startsWithCheckCase(filePath, appBasePath) && ((fileLen == baseLen) || isSeparatorChar(filePath.charAt(baseLen)))) {
            // simpler case : the two starts exactly match
            relativePath = filePath.substring(appBasePath.length());
        } else if ((drive == null) || equalsCheckCase(drive, appBaseDrive)) {
            // complex type : whe have to compute diff
            // --> get the divergence index
            int divergenceIndex = 0;
            int len = (fileLen < baseLen) ? fileLen : baseLen;
            int i;
            for (i = 0; i < len; i++) {
                char ch = filePath.charAt(i);
                if (!equalsCheckCase(ch, appBasePath.charAt(i))) break;
                if (isSeparatorChar(ch)) divergenceIndex = i + 1;
            }
            if ((i == fileLen) && (i < baseLen) && isSeparatorChar(appBasePath.charAt(len))) divergenceIndex = len + 1;
            // create the ../../ part
            StringBuffer sb = new StringBuffer(".." + separatorChar);
            for (i = divergenceIndex; i < baseLen; i++) {
                if (isSeparatorChar(appBasePath.charAt(i))) sb.append(".." + separatorChar);
            }
            // add the diverging file part
            if (divergenceIndex < fileLen) sb.append(filePath.substring(divergenceIndex));
            relativePath = sb.toString();
        } else {
            // cannot compute relative path because an other drive is used.
            return filePath;
        }
        if (startsWithSeparator(relativePath)) relativePath = relativePath.substring(1);
        if (endsWithSeparator(relativePath)) relativePath = relativePath.substring(0, relativePath.length() - 1);
        if (relativePath.length() == 0) relativePath = ".";
        return relativePath;
    }


    private static boolean isSeparatorChar(char ch) {
        return (ch == '/') || (ch == '\\');
    }


    private static boolean startsWithSeparator(String str) {
        return str.startsWith("/") || str.startsWith("\\");
    }


    private static boolean endsWithSeparator(String str) {
        return str.endsWith("/") || str.endsWith("\\");
    }


    private static boolean isThisPath(String str) {
        return "./".equals(str) || ".\\".equals(str) || ".".equals(str);
    }


    /**
     * Check if the given file path is absolute.
     * 
     * @param filePath
     *            the file path to check
     * @return true iff the given file path is absolute.
     */
    public static boolean isAbsolute(String filePath) {
        if (filePath == null) return false;
        if (filePath.indexOf(':') > 0) return true;
        return startsWithSeparator(filePath);
    }


    /**
     * Get the drive of a path or null if no drive is specified in the path. <br>
     * The returned drive string contains the drive letter and the colon (like "C:").
     * 
     * @param filePath
     *            The file path possibly starting with a drive letter.
     * @return The drive of a path or null if no drive is specified in the path.
     */
    public static String getDrive(String filePath) {
        if (filePath == null) return null;
        int colonIndex = filePath.indexOf(':');
        return (colonIndex > 0) ? filePath.substring(0, colonIndex + 1) : null;
    }


    /**
     * Concatenate two paths to one path. <br>
     * The first path can be any path (relative or absolute) or can even be null. If the second path is absolute, this path is
     * returned as is. If it is a relative path, this method concatenate the two taking care if there is a file separator to add
     * between the two concatenated paths (and if the first path is not empty).
     * 
     * @param firstPath
     *            first path to concatenate.
     * @param secondPath
     *            second path to concatenate.
     * @return the two paths concatenated to one path.
     */
    public static String concatPaths(String firstPath, String secondPath) {
        if (isWhite(secondPath)) return firstPath;
        if (isAbsolute(secondPath)) return secondPath;
        if (isWhite(firstPath) || isThisPath(firstPath)) return secondPath;
        if (endsWithSeparator(firstPath)) return firstPath + secondPath;
        return firstPath + File.separator + secondPath;
    }


    /**
     * Check if the given string starts with the given start string. The check will or will not be case sensitive depending of
     * the <code>IGNORE_CASE</code> flag.
     * 
     * @param str
     *            String to check.
     * @param startStr
     *            Start string to compare.
     * @return true iff the given string starts with the givent start string.
     */
    private boolean startsWithCheckCase(String str, String startStr) {
        if (startStr == null) return true;
        if (str == null) return false;
        int startLen = startStr.length();
        if (str.length() >= startLen) return str.regionMatches(ignoreCase, 0, startStr, 0, startLen);
        return false;
    }


    /**
     * Check if the given chars are the same. The check will or will not be case sensitive depending of the
     * <code>IGNORE_CASE</code> flag.
     */
    private boolean equalsCheckCase(char ch1, char ch2) {
        if (ignoreCase) return Character.toLowerCase(ch1) == Character.toLowerCase(ch2);
        return ch1 == ch2;
    }


    /**
     * Check if the given strings are the same. The check will or will not be case sensitive depending of the
     * <code>IGNORE_CASE</code> flag.
     * 
     * @param str1
     *            String to check.
     * @param str2
     *            String to check.
     */
    private boolean equalsCheckCase(String str1, String str2) {
        if (str1 == null) return str2 == null;
        if (ignoreCase) return str1.equalsIgnoreCase(str2);
        return str1.equals(str2);
    }


    /**
     * Check if a string is null or contains only whitespaces.
     * 
     * @param str
     *            string to check
     * @return true iff the given string is null or contains only whitespaces.
     */
    private static boolean isWhite(String str) {
        if (str == null) return true;
        int len = str.length();
        for (int i = 0; i < len; i++) {
            if (str.charAt(i) > ' ') return false;
        }
        return true;
    }


    /**
     * Get the extension of the given file (ex: extension of "myIcon.gif" returns "gif"). <br>
     * This method simply returns the text after the last dot, or null if there is no dot.
     * 
     * @param fileName
     *            a file name
     * @return the extension of the given file.
     */
    public static String getExtension(String fileName) {
        if (fileName == null) return null;
        int lastDotIndex = fileName.lastIndexOf('.');
        if (lastDotIndex < 0) return null;
        if (fileName.lastIndexOf('/') > lastDotIndex) return null;
        if (fileName.lastIndexOf('\\') > lastDotIndex) return null;
        return fileName.substring(lastDotIndex + 1);
    }


    /**
     * Get the directory part of a full file path. <br>
     * Example: directory for "\Temp\test\MyFile.txt" is "\Temp\test". <br>
     * If there is no directory, null is returned. <br>
     * Example: directory for "MyFile.txt" is null.
     * <p>
     * Note: This method does not try to interpret the path, the last item of the path is assumed to be a file name (even if it
     * is a directory). So directory for "\Temp\test" is "\Temp", and directory for "\Temp\" is "\Temp".
     * </p>
     * 
     * @param filePath
     *            a full file path
     */
    public static String getDirectoryPath(String filePath) {
        if (filePath == null) return null;
        int lastSlashIndex = filePath.lastIndexOf('/');
        int lastBackSlashIndex = filePath.lastIndexOf('\\');
        int index = Math.max(lastSlashIndex, lastBackSlashIndex);
        if (index >= 0) return filePath.substring(0, index);
        return null;
    }


    /**
     * Get the file name (and extension) from the gievn path. <br>
     * Example: file name for "\Temp\test\MyFile.txt" is "MyFile.txt".
     * <p>
     * Note: This method does not try to interpret the path, the last item of the path is assumed to be a file name (even if it
     * is a directory). So file name for "\Temp\test" is "test", and file name for "\Temp\" is "".
     * </p>
     * 
     * @param filePath
     *            a full file path
     */
    public static String getFileName(String filePath) {
        if (filePath == null) return null;
        int lastSlashIndex = filePath.lastIndexOf('/');
        int lastBackSlashIndex = filePath.lastIndexOf('\\');
        int index = Math.max(lastSlashIndex, lastBackSlashIndex);
        if (index >= 0) return filePath.substring(index + 1);
        return filePath;
    }

}
