/*
 * Seam-Perf4j - Perf4j integration for Seam Framework
 * Copyright (C) 2010 Marcin Zajaczkowski
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package net.sf.seam.perf4j;

import org.jboss.seam.log.Log;
import org.perf4j.LoggingStopWatch;
import org.perf4j.aop.AgnosticTimingAspect;
import org.perf4j.aop.Profiled;

/**
 * An implementation of Perf4j's AgnosticTimingAspect using Seam Logger to log results.
 *
 * @author Marcin Zajączkowski, 2010-01-30
 */
public class SeamTimingAspect extends AgnosticTimingAspect {

    private Log logger;

    /**
     * Causes default logger and default message generator to be used.
     */
    public SeamTimingAspect() {
        this((Log)null);
    }

    /**
     * Allows to specify a logger to be used.
     *
     * Messages will be generated using default message generator defined in Perf4j library.
     *
     * @param logger logger to be used
     */
    //TODO: MZA: is there sense to pass logger?
    public SeamTimingAspect(Log logger) {
        super();
        this.logger = logger;
    }

    /**
     * Executes the real profiling method from Perf4j.
     *
     * @param sjp seam join point
     * @param profiled profiled annotation from profiled method
     * @return return value from executed methof
     * @throws Throwable exception thrown by executing method
     */
    public Object doPerfLogging(SeamJoinPoint sjp, Profiled profiled) throws Throwable {
        return runProfiledMethod(sjp, profiled, newStopWatch(profiled.logger(), profiled.level()));
    }

    /**
     * Returns an implementation of StopWatch using Seam logger.
     *
     * The logger passed through constructor is used if available. Otherwise default Seam logger is obtained.
     *
     * @param loggerName logger name passed to StopWatch
     * @param levelName logging level name passed to StopWatch
     * @return LoggingStopWatch implementation
     *
     * TODO: MZA: level names for Seam have to be standarized.
     */
    protected LoggingStopWatch newStopWatch(String loggerName, String levelName) {
        if (logger == null) {
            return new SeamLoggerStopWatch(org.jboss.seam.log.Logging.getLog(loggerName), levelName);
        } else {
            return new SeamLoggerStopWatch(logger, levelName);
        }
    }
}
