/*
 * Seam-Perf4j - Perf4j integration for Seam Framework
 * Copyright (C) 2010 Marcin Zajaczkowski
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package net.sf.seam.perf4j;

import org.jboss.seam.log.Log;
import org.perf4j.LoggingStopWatch;

/**
 * An implementation of Perf4j's LoggingStopWatch using Seam Logger to log messages.
 *
 * @author Marcin Zajączkowski, 2010-01-30
 */
public class SeamLoggerStopWatch extends LoggingStopWatch {

    /**
     * Levels (priorites) to log messages on.
     */
    public enum LoggingLevel {
        FATAL,
        ERROR,
        WARN,
        INFO,
        DEBUG,
        TRACE;

        /**
         * Safe version of valueOf(). Returns matching logging level enum or that given as parameter if none matches.
         *
         * @param name name of loging level as String
         * @param defaultLevel logging level to be returned if none matches
         * @return matching logging level enum or given default if none matches
         */
        public static LoggingLevel safeValueOf(String name, LoggingLevel defaultLevel) {
            try {
                return LoggingLevel.valueOf(name);
            } catch (IllegalArgumentException e) {
                return defaultLevel;
            }
        }
    }

    public static final LoggingLevel DEFAULT_NORMAL_PRIORITY = LoggingLevel.INFO;
    public static final LoggingLevel DEFAULT_EXCEPTION_PRIORITY = LoggingLevel.WARN;

    private transient Log logger;
    private LoggingLevel normalPriority;
    private LoggingLevel exceptionPriority;

    /**
     * Default constructor.
     */
    public SeamLoggerStopWatch() {
        this(DEFAULT_NORMAL_PRIORITY);
    }

    public SeamLoggerStopWatch(LoggingLevel normalPriority) {
        this(normalPriority, DEFAULT_EXCEPTION_PRIORITY);
    }

    public SeamLoggerStopWatch(LoggingLevel normalPriority, LoggingLevel exceptionPriority) {
        this(org.jboss.seam.log.Logging.getLog(DEFAULT_LOGGER_NAME), normalPriority, exceptionPriority);
    }

    public SeamLoggerStopWatch(String normalPriority) {
        this(LoggingLevel.safeValueOf(normalPriority, DEFAULT_NORMAL_PRIORITY));
    }

    public SeamLoggerStopWatch(String normalPriority, String exceptionPriority) {
        this(LoggingLevel.safeValueOf(normalPriority, DEFAULT_NORMAL_PRIORITY),
             LoggingLevel.safeValueOf(exceptionPriority, DEFAULT_EXCEPTION_PRIORITY));
    }

    public SeamLoggerStopWatch(Log logger) {
        this(logger, DEFAULT_NORMAL_PRIORITY);
    }

    public SeamLoggerStopWatch(Log logger, String normalPriority) {
        this(logger, LoggingLevel.safeValueOf(normalPriority, DEFAULT_NORMAL_PRIORITY));
    }

    public SeamLoggerStopWatch(Log logger, String normalPriority, String exceptionPriority) {
        this(logger, LoggingLevel.safeValueOf(normalPriority, DEFAULT_NORMAL_PRIORITY),
                LoggingLevel.safeValueOf(exceptionPriority, DEFAULT_EXCEPTION_PRIORITY));
    }

    public SeamLoggerStopWatch(Log logger, LoggingLevel normalPriority) {
        this(logger, normalPriority, DEFAULT_EXCEPTION_PRIORITY);
    }

    /**
     * Constructs StopWatch with given logger and given logging levels (priorities).
     *
     * @param logger logger to use
     * @param normalPriority default logging level to use
     * @param exceptionPriority logging level to use when exception occurred
     */
    public SeamLoggerStopWatch(Log logger, LoggingLevel normalPriority, LoggingLevel exceptionPriority) {
        super();
        if (logger == null) {
            throw new IllegalArgumentException("logger cannot be null!");
        }
        this.logger = logger;
        this.normalPriority = normalPriority;
        this.exceptionPriority = exceptionPriority;
    }

    public Log getLogger() {
        return logger;
    }

    public LoggingLevel getNormalPriority() {
        return normalPriority;
    }

    public LoggingLevel getExceptionPriority() {
        return exceptionPriority;
    }

    /**
     * Determines if logging is currently enabled for normal log messages printed by StopWatch.
     *
     * In implementation isXXXEnabled() from org.jboss.seam.log.Log interface is used.
     *
     * @return true if calls of stop() method that do not take an exception will be logged using logger  
     */
    @Override
    public boolean isLogging() {
        switch (normalPriority) {
            case FATAL:
                return logger.isFatalEnabled();
            case ERROR:
                return logger.isErrorEnabled();
            case WARN:
                return logger.isWarnEnabled();
            case INFO:
                return logger.isInfoEnabled();
            case DEBUG:
                return logger.isDebugEnabled();
            case TRACE:
                return logger.isTraceEnabled();
            default:
                //hmm, shouldn't happen
                return true;
        }
    }

    /**
     * Logs StopWatch message using Seam's logger.
     *
     * @param stopWatchAsString serialized StopWatch string
     * @param exception optional exception passed to be logged
     */
    @Override
    protected void log(String stopWatchAsString, Throwable exception) {
        final LoggingLevel requiredPriority = (exception == null ? normalPriority : exceptionPriority);
        switch (requiredPriority) {
            case FATAL:
                logger.fatal(stopWatchAsString, exception);
                break;
            case ERROR:
                logger.error(stopWatchAsString, exception);
                break;
            case WARN:
                logger.warn(stopWatchAsString, exception);
                break;
            case INFO:
            default:
                logger.info(stopWatchAsString, exception);
                break;
            case DEBUG:
                logger.debug(stopWatchAsString, exception);
                break;
            case TRACE:
                logger.trace(stopWatchAsString, exception);
                break;
        }
    }

    //TODO: MZA: Take a look at serialization aspect 
}
