/*
 * Seam-Perf4j - Perf4j integration for Seam Framework
 * Copyright (C) 2010 Marcin Zajaczkowski
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package net.sf.seam.perf4j;

import org.jboss.seam.annotations.intercept.AroundInvoke;
import org.jboss.seam.annotations.intercept.Interceptor;
import org.jboss.seam.intercept.AbstractInterceptor;
import org.jboss.seam.intercept.InvocationContext;
import org.perf4j.aop.DefaultProfiled;
import org.perf4j.aop.Profiled;

import java.lang.reflect.Method;

/**
 * Perf4j interceptor for Seam.
 *
 * Note: it's required to turn on that interceptor in components.xml core:init -> core:interceptors.
 *
 * @author Marcin Zajączkowski, 2010-01-17
 *
 * @see Perf4jProfiled
 */
@Interceptor
public class Perf4jProfiledInterceptor extends AbstractInterceptor {

    private static final long serialVersionUID = -8581318944020712128L;

    /**
     * Delegates performance logging to specific TimingAspect class for annotated methods.
     *
     * @param ic invocation context
     * @return method execution return value
     * @throws Exception exception thrown during excecution
     */
    @AroundInvoke
    public Object aroundInvoke(InvocationContext ic) throws Exception {

        //Perf4jProfiled can be used only with method or constructor, but interceptor could be used directly
        //so it's better to verify that there is a method to prevent NPE.
        Method method = ic.getMethod();
        //no method execution or method not annotated no need to measure
        if (method == null || method.getAnnotation(Perf4jProfiled.class) == null) {
            return ic.proceed();
        }

        Profiled profiled = method.getAnnotation(Profiled.class);
        //when no Profiled annotation was used default values are used
        if (profiled == null) {
            profiled = DefaultProfiled.INSTANCE;
        }

        SeamTimingAspect ata = newTimingAspect();
        try {
            return ata.doPerfLogging(new SeamJoinPoint(ic), profiled);
        } catch (Exception e) {
            throw e;
        } catch (Throwable throwable) {
            //construction forced by Throwable in AgnosticTimingAspect.runProfiledMethod() vs.
            //Exception in OptimizedInterceptor.aroundInvoke
            throw new Exception("Throwable in AgnosticTimingAspect.runProfiledMethod()", throwable);
        }
    }

    /**
     * Returns new TimingAspect.
     *
     * Made mostly for tests.
     *
     * @return new instance of SeamTimingAspect
     */
    protected SeamTimingAspect newTimingAspect() {
        return new SeamTimingAspect();
    }

    /**
     * Tells when interceptor is enabled.
     *
     * In current implementation interceptor is enabled only for components with any method annotated with
     * {@link Perf4jProfiled}. There is not handler to currently called method.
     *
     * @return <code>true</code> if processing method is annotated with @Perf4jProfiled
     *         <code>false</code> for other methods
     */
    public boolean isInterceptorEnabled() {
        return getComponent().beanClassHasAnnotation(Perf4jProfiled.class);
    }
}
