package net.sf.seaf.util.convert.collection;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import net.sf.seaf.exception.SeafConversionException;

public class SequenceConverterUtils {

	/**
	 * Convert an Object array or a Collection to an array or a List of a given
	 * component type.
	 * <p>
	 * Each object is converted using a given object converter.
	 * 
	 * @param sourceValue
	 *            Object to be converted
	 * @param targetType
	 *            The required target type
	 * @param targetComponentType
	 *            Component type of the target array
	 * @param objectConverter
	 *            Object converter to be used to convert each object
	 * @return Converted array
	 * @throws SeafConversionException
	 *             When source object is of unsupported type
	 */
	public Object convert(Object sourceValue, Class<?> targetType,
			Class<?> targetComponentType,
			ObjectConverter<Object, Object> objectConverter) {
		if (targetType.isArray())
			return convertToArray(sourceValue, targetComponentType,
					objectConverter);
		return convertToList(sourceValue, targetComponentType, objectConverter);
	}

	/**
	 * Convert an Object array or a Collection to an array of a given component
	 * type.
	 * <p>
	 * Each object is converted using a given object converter.
	 * 
	 * @param sourceValue
	 *            Object to be converted
	 * @param targetComponentType
	 *            Component type of the target array
	 * @param objectConverter
	 *            Object converter to be used to convert each object
	 * @return Converted array
	 * @throws SeafConversionException
	 *             When source object is of unsupported type
	 */
	public Object[] convertToArray(Object sourceValue,
			Class<?> targetComponentType,
			ObjectConverter<Object, Object> objectConverter) {
		Collection<Object> targetCollection = new ArrayList<Object>();
		Class<?> sourceType = sourceValue.getClass();
		Collection<Object> sourceCollection;
		if (sourceType.isArray()
				&& Object.class.isAssignableFrom(sourceType.getComponentType())) {
			sourceCollection = Arrays.asList((Object[]) sourceValue);
		} else if (Collection.class.isAssignableFrom(sourceType)) {
			sourceCollection = castToCollectionObject(sourceValue);
		} else {
			throw new SeafConversionException(
					"Source value is of unsupported type: " + sourceType);
		}
		new CollectionConverter<Object, Object>().convert(sourceCollection,
				targetCollection, objectConverter);
		Object[] targetArray = targetCollection.toArray((Object[]) Array
				.newInstance(targetComponentType, 0));
		return targetArray;
	}

	/**
	 * Convert an Object array or a Collection to a List of a given component
	 * type.
	 * <p>
	 * Each object is converted using a given object converter.
	 * 
	 * @param sourceValue
	 *            Object to be converted
	 * @param targetComponentType
	 *            Component type of the target List
	 * @param objectConverter
	 *            Object converter to be used to convert each object
	 * @return Converted List
	 * @throws SeafConversionException
	 *             When source object is of unsupported type
	 */
	public List<Object> convertToList(Object sourceValue,
			Class<?> targetComponentType,
			ObjectConverter<Object, Object> objectConverter) {
		Object[] targetArray = convertToArray(sourceValue, targetComponentType,
				objectConverter);
		return Arrays.asList(targetArray);
	}

	@SuppressWarnings("unchecked")
	private Collection<Object> castToCollectionObject(Object sourceValue) {
		return (Collection<Object>) sourceValue;
	}

}
