package net.sf.seaf.util;

import java.util.Collection;

/**
 * Builder of toString() return value for a JavaBean.
 */
public class ToStringUtils {

	private final StringBuilder b;
	private boolean firstProperty = true;

	/**
	 * Initialize the builder.
	 * 
	 * @param clazz
	 *            The class of the bean
	 * @param superToString
	 *            The result of toString() on the superclass
	 * @throws NullPointerException
	 *             When clazz is null
	 */
	public ToStringUtils(Class<?> clazz, String superToString) {
		b = new StringBuilder(clazz.getName());
		b.append('[');
		Class<?> superClass = clazz.getSuperclass();
		if (null != superClass
				&& !"java.lang.Object".equals(superClass.getName())) {
			b.append('[');
			b.append(superToString);
			b.append(']');
		}
	}

	/**
	 * Add a property to the builder.
	 * 
	 * @param name
	 *            Name of the property
	 * @param value
	 *            Value of the property
	 */
	public void addProperty(String name, Object value) {
		if (firstProperty)
			firstProperty = false;
		else
			b.append(',');
		b.append(name);
		b.append('=');
		appendValue(value);
	}

	/**
	 * Append value to the builder, expanding sequences appropriately.
	 * <p>
	 * A sequence can be array or Collection.
	 * 
	 * @param value
	 *            Value to append
	 */
	private void appendValue(Object value) {
		if (value instanceof byte[]) {
			appendArray(value.getClass().getComponentType(),
					((byte[]) value).length, new Object[0]);
		} else if (value instanceof int[]) {
			appendArray(value.getClass().getComponentType(),
					((int[]) value).length, new Object[0]);
		} else if (value instanceof long[]) {
			appendArray(value.getClass().getComponentType(),
					((long[]) value).length, new Object[0]);
		} else if (value instanceof char[]) {
			appendArray(value.getClass().getComponentType(),
					((char[]) value).length, new Object[0]);
		} else if (value instanceof float[]) {
			appendArray(value.getClass().getComponentType(),
					((float[]) value).length, new Object[0]);
		} else if (value instanceof double[]) {
			appendArray(value.getClass().getComponentType(),
					((double[]) value).length, new Object[0]);
		} else if (value instanceof Object[]) {
			appendArray(value.getClass().getComponentType(), (Object[]) value);
		} else if (value instanceof Collection) {
			appendArray(value.getClass(), ((Collection<?>) value).toArray());
		} else
			b.append(String.valueOf(value));
	}

	/**
	 * Append array of items to the builder.
	 * 
	 * @param clazz
	 *            Type of the sequence
	 * @param items
	 *            Sequence items to output
	 */
	private void appendArray(Class<?> clazz, Object[] items) {
		appendArray(clazz, items.length, items);
	}

	/**
	 * Append array of items to the builder.
	 * 
	 * @param clazz
	 *            Type of the sequence
	 * @param displayLength
	 *            Length to display
	 * @param items
	 *            Sequence items to output
	 */
	private void appendArray(Class<?> clazz, int displayLength, Object[] items) {
		b.append(clazz.getName());
		b.append('#');
		b.append(displayLength);
		b.append('[');
		for (int i = 0; i < items.length; i++) {
			b.append(items[i]);
			if (i < items.length - 1)
				b.append(',');
		}
		b.append(']');
	}

	/**
	 * Returns the value of the buffered toString output.
	 * 
	 * @return String
	 */
	@Override
	public String toString() {
		b.append(']');
		return b.toString();
	}

}
