package net.sf.seaf.util;

import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

/**
 * Utility methods for converting temporal types.
 */
public class TemporalConverterUtil {

	/**
	 * Convert an object to a {@link java.util.Calendar}.
	 * <p>
	 * Passes <code>null</code> unchanged. Passes {@link java.util.Calendar}
	 * unchanged. Converts {@link java.util.Date}. Otherwise, parses the passed
	 * object as String using the format supplied.
	 * 
	 * @param o
	 *            The object to be converted
	 * @param format
	 *            The Java date format pattern, see {@link java.text.DateFormat}
	 * @return Calendar The object converted to {@link java.util.Calendar}
	 * @throws IllegalArgumentException
	 *             Thrown when the supplied object is not a valid temporal value
	 *             given the format specified
	 */
	public static Calendar convert(Object o, String format)
			throws IllegalArgumentException {
		if (null == o)
			return null;
		if (o instanceof Date) {
			GregorianCalendar cal = new GregorianCalendar();
			cal.setTime(((Date) o));
			return cal;
		}
		if (o instanceof Calendar)
			return (Calendar) o;
		return TemporalConverterUtil.parseDate(o.toString(), format);
	}

	/**
	 * Parse a Calendar from String value using a Java date format pattern.
	 * 
	 * @param text
	 *            The String containing temporal value representation
	 * @param format
	 *            The Java date format pattern, see {@link java.text.DateFormat}
	 * @return Calendar The text converted to {@link java.util.Calendar}
	 * @throws IllegalArgumentException
	 *             Thrown when the supplied text is not a valid temporal value
	 *             given the format specified
	 */
	public static Calendar parseDate(String text, String format)
			throws IllegalArgumentException {
		SimpleDateFormat formatter = new SimpleDateFormat(format);
		ParsePosition pos = new ParsePosition(0);
		Date date = formatter.parse(text, pos);
		if (null == date)
			throw new IllegalArgumentException("Value " + text
					+ " cannot be parsed as " + format);
		GregorianCalendar cal = new GregorianCalendar();
		cal.setTime(date);
		backwardValidationOfStringValue(text, cal, format);
		return cal;
	}

	/**
	 * Throw IllegalArgumentException when the parsed Calendar value does not
	 * match the original String value, from which it has been parsed.
	 * 
	 * @param originalValue
	 *            The original String value
	 * @param parsedValue
	 *            The parsed Calendar value
	 * @param format
	 *            The format used for parsing
	 * @throws IllegalArgumentException
	 *             Thrown when the parsed value does not match the original
	 *             value
	 */
	public static void backwardValidationOfStringValue(String originalValue,
			Calendar parsedValue, String format)
			throws IllegalArgumentException {
		String result = formatDate(parsedValue, format);
		if (!originalValue.equals(result))
			throw new IllegalArgumentException("Value " + originalValue
					+ " is not valid.");
	}

	/**
	 * Format a Calendar value as a String using the given format.
	 * 
	 * @param value
	 *            Calendar value
	 * @param format
	 *            Format to use
	 * @return String Formatted value
	 */
	public static String formatDate(Calendar value, String format) {
		SimpleDateFormat formatter = new SimpleDateFormat(format);
		return formatter.format(value.getTime());
	}

}
