package net.sf.seaf.util;

import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Map;
import java.util.Map.Entry;

import net.sf.seaf.util.convert.simple.ConverterToString;
import net.sf.seaf.util.convert.simple.TemporalType;

import org.apache.commons.lang.builder.ReflectionToStringBuilder;
import org.apache.commons.lang.builder.ToStringStyle;

/**
 * Converts Object to String recursively using {@link ReflectionToStringBuilder}
 * .
 * <p>
 * Base type objects such as String, Number, Character or Boolean are passed to
 * ToStringBuilder and directly converted to String, objects Date and Calendar
 * are converted to String with a timestamp format YYYY-MM-DD'T'HH:MM:SS.sss.
 * <p>
 * Each object in Array, Collection or Map is recursively converted as well. In
 * case of Map, keys are not handled recursively.
 * <p>
 * Unicode null character is treated. In case of String, it is removed. In case
 * of Character, it is replaced by a null string.
 * 
 * @author pspicka
 * 
 */
public class RecursiveReflectionToStringStyle extends ToStringStyle {

	private static final long serialVersionUID = 1L;

	/**
	 * The default style.
	 */
	public static final RecursiveReflectionToStringStyle DEFAULT_RR_STYLE = new RecursiveReflectionToStringStyle();

	/**
	 * Simplified style. Changes comprise: no ID hash code.
	 */
	public static final RecursiveReflectionToStringStyle SIMPLE_RR_STYLE = new SimpleRecursiveReflectionToStringStyle();

	private static class SimpleRecursiveReflectionToStringStyle extends
			RecursiveReflectionToStringStyle {
		private static final long serialVersionUID = 1L;

		public SimpleRecursiveReflectionToStringStyle() {
			setUseIdentityHashCode(false);
		}
	}

	/**
	 * Default constructor.
	 */
	private RecursiveReflectionToStringStyle() {
		super();
		setSizeStartText("{#");
		setSizeEndText("}");
		setUseShortClassName(false);
		setUseIdentityHashCode(true);
	}

	@Override
	protected void appendDetail(StringBuffer buffer, String fieldName,
			byte[] array) {
		appendSummary(buffer, fieldName, array);
	}

	@Override
	protected void appendSummarySize(StringBuffer buffer, String fieldName,
			int size) {
		buffer.append(getSizeStartText());
		buffer.append(size);
		buffer.append(getSizeEndText());
	}

	@Override
	protected void appendDetail(StringBuffer buffer, String fieldName,
			@SuppressWarnings("rawtypes") Map map) {
		buffer.append(Map.class.getName());
		appendDetail(buffer, fieldName, map.entrySet().toArray());
	}

	@Override
	protected void appendDetail(StringBuffer buffer, String fieldName,
			@SuppressWarnings("rawtypes") Collection coll) {
		buffer.append(Collection.class.getName());
		if (isArrayContentDetail()) {
			appendDetail(buffer, fieldName, coll.toArray());
		} else {
			appendSummary(buffer, fieldName, coll.toArray());
		}
	}

	@Override
	@SuppressWarnings("rawtypes")
	protected void appendDetail(StringBuffer buffer, String fieldName,
			Object value) {
		if ((isBaseType(value))) {
			appendBaseType(buffer, fieldName, value);
		} else if (isHashMapEntry(value)) {
			appendDetail(buffer, fieldName, ((Entry) value));
		} else if (isDate(value)) {
			super.appendDetail(buffer, fieldName,
					ConverterToString.convert(value, TemporalType.TIMESTAMP));
		} else {
			new ReflectionToStringBuilder(value, this, buffer).toString();
		}
	}

	/**
	 * Returns true, if the Object can be directly converted to String.
	 * 
	 * @param buffer
	 * @param fieldName
	 * @param value
	 */
	private void appendBaseType(StringBuffer buffer, String fieldName,
			Object value) {
		if (isString(value)) {
			appendDetail(buffer, fieldName, (String) value);
		} else if (isChar(value)) {
			appendDetail(buffer, fieldName, (Character) value);
		} else {
			super.appendDetail(buffer, fieldName, value);
		}
	}

	/**
	 * Appender for String, filters unicode null character and appends String to
	 * Buffer.
	 * 
	 * @param buffer
	 * @param fieldName
	 * @param value
	 */
	protected void appendDetail(StringBuffer buffer, String fieldName,
			String value) {
		value = (value).replaceAll('\u0000' + "", "");
		super.appendDetail(buffer, fieldName, value);
	}

	/**
	 * Appender for Character. If it is unicode null value replaces it with null
	 * text, otherwise appends the value.
	 * 
	 * @param buffer
	 * @param fieldName
	 * @param value
	 */
	protected void appendDetail(StringBuffer buffer, String fieldName,
			Character value) {
		if (value.equals('\u0000')) {
			super.appendDetail(buffer, fieldName, getNullText());
		} else {
			super.appendDetail(buffer, fieldName, value);
		}
	}

	/**
	 * Appender for Map.Entry, key is appended directly, value is appended
	 * recursively.
	 * 
	 * @param buffer
	 * @param fieldName
	 * @param hashMapEntry
	 */
	protected void appendDetail(StringBuffer buffer, String fieldName,
			@SuppressWarnings("rawtypes") Entry hashMapEntry) {
		super.appendDetail(buffer, fieldName, hashMapEntry.getKey());
		buffer.append(getFieldNameValueSeparator());
		appendInternal(buffer, fieldName, hashMapEntry.getValue(), true);
	}

	/**
	 * Returns true, is value is Character.
	 * 
	 * @param value
	 * @return
	 */
	private boolean isChar(Object value) {
		return (value instanceof Character);
	}

	/**
	 * Returns true if value is String.
	 * 
	 * @param value
	 * @return
	 */
	private boolean isString(Object value) {
		return (value instanceof String);
	}

	/**
	 * Returns true if value is Date or Calendar.
	 * 
	 * @param value
	 * @return
	 */
	private boolean isDate(Object value) {
		return (value instanceof Date || value instanceof Calendar);
	}

	/**
	 * Returns true if value is Map.Entry.
	 * 
	 * @param value
	 * @return
	 */
	private boolean isHashMapEntry(Object value) {
		return (value instanceof Entry);
	}

	/**
	 * Returns true if value is directly convertible to String.
	 * 
	 * @param value
	 * @return
	 */
	private boolean isBaseType(Object value) {
		return value.getClass().isPrimitive() || value instanceof String
				|| value instanceof Number || value instanceof Boolean
				|| value instanceof Character;
	}
}
