package net.sf.seaf.factory.impl;

import net.sf.seaf.exception.SeafTypeResolvingException;
import net.sf.seaf.factory.Factory;
import net.sf.seaf.factory.TypeResolver;
import net.sf.seaf.factory.impl.support.DelegatingFactoryBase;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Factory that resolves the requested type before delegating instantiation to
 * an instantiating factory.
 * <p>
 * When the type resolver returns null, throws
 * {@link SeafTypeResolvingException}.
 */
public class TypeResolvingFactory extends DelegatingFactoryBase implements
		Factory {

	private TypeResolver<?> typeResolver;
	private final Logger log = LoggerFactory
			.getLogger(TypeResolvingFactory.class);

	/**
	 * Default empty constructor. The type resolver and instantiating factory
	 * must be supplied via its setter methods.
	 */
	public TypeResolvingFactory() {
	}

	/**
	 * Full constructor.
	 * 
	 * @param typeResolver
	 *            The type resolver
	 * @param instantiatingFactory
	 *            The instantiating factory
	 */
	public TypeResolvingFactory(TypeResolver<?> typeResolver,
			Factory instantiatingFactory) {
		super(instantiatingFactory);
		this.typeResolver = typeResolver;
	}

	public <Type> Type getInstanceOf(Class<Type> type)
			throws SeafTypeResolvingException {
		Class<? extends Type> resolvedType = getTypeResolver(type)
				.resolve(type);
		log.trace("Resolved type {} as {}", type, resolvedType);

		if (null == resolvedType) {
			String message = "Cannot resolve type " + type
					+ " in type resolver " + typeResolver.getClass().getName()
					+ ", resolver returned null";
			log.error(message);
			throw new SeafTypeResolvingException(message);
		}

		return getInstantiatingFactory().getInstanceOf(resolvedType);
	}

	@SuppressWarnings("unchecked")
	private <Type> TypeResolver<Type> getTypeResolver(Class<Type> type) {
		return (TypeResolver<Type>) typeResolver;
	}

	/**
	 * Set the type resolver.
	 * 
	 * @param typeResolver
	 *            The type resolver
	 */
	public final void setTypeResolver(TypeResolver<?> typeResolver) {
		this.typeResolver = typeResolver;
	}

}
