package net.sf.seaf.factory.impl;

import net.sf.seaf.exception.SeafInitializationException;
import net.sf.seaf.exception.SeafRuntimeException;
import net.sf.seaf.factory.Factory;
import net.sf.seaf.factory.Initializer;
import net.sf.seaf.factory.impl.support.DelegatingFactoryBase;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Factory that initializes the returned instances using an {@link Initializer}.
 * <p>
 * The initializer set on the factory must match the type of the instance
 * returned, otherwise {@link SeafInitializationException} is thrown.
 */
public class InitializingFactory extends DelegatingFactoryBase implements
		Factory {

	private Initializer<?> initializer;
	private final Logger log = LoggerFactory
			.getLogger(InitializingFactory.class);

	/**
	 * Default empty constructor. The initializer and instantiating factory must
	 * be supplied via its setter methods.
	 */
	public InitializingFactory() {
	}

	/**
	 * Full constructor.
	 * 
	 * @param initializer
	 *            The initializer
	 * @param instantiatingFactory
	 *            The instantiating factory
	 */
	public InitializingFactory(Initializer<?> initializer,
			Factory instantiatingFactory) {
		super(instantiatingFactory);
		this.initializer = initializer;
	}

	public <Type> Type getInstanceOf(Class<Type> type)
			throws SeafRuntimeException {
		Type instance = getInstantiatingFactory().getInstanceOf(type);
		log.trace("Initializing instance of type {}", type);
		try {
			getInitializer(type).initialize(instance);
		} catch (ClassCastException e) {
			String message = "Cannot initialize instance of type " + type
					+ " in initializer " + initializer.getClass().getName();
			log.error(message, e);
			throw new SeafInitializationException(message, e);
		}
		log.trace("Initialized instance {}", instance);
		return instance;
	}

	@SuppressWarnings("unchecked")
	private <Type> Initializer<Type> getInitializer(Class<Type> type) {
		return (Initializer<Type>) initializer;
	}

	/**
	 * Set the initializer.
	 * 
	 * @param initializer
	 *            The initializer
	 */
	public final void setInitializer(Initializer<?> initializer) {
		this.initializer = initializer;
	}

}
