package net.sf.seaf.util;

/**
 * A utility which allows easy implementation of <code>hashCode()</code>.
 * <p>
 * Just pass in the values of the properties which compose the hash code:
 * 
 * <pre>
 * public class Person {
 *   String name;
 *   int age;
 *   boolean smoker;
 *   ...
 *   public int hashCode() {
 *     return HashCodeUtils.hashCodeFor(name, age, smoker);
 *   }
 * }
 * </pre>
 */
public class HashCodeUtils {

	/**
	 * Returns a hash code for an array of values.
	 * 
	 * @param values
	 *            Object[] of values to be hashed
	 * @return int hash code
	 */
	public static int hashCodeFor(Object... values) {
		return hashObject(SEED, values);
	}

	/**
	 * An initial value for a hash code, to which is added contributions from
	 * fields. Using a non-zero value decreases collisions of hash code values.
	 */
	private static final int SEED = 23;

	/**
	 * Hash code calculation for an Object.
	 * <p>
	 * <code>anObject</code> is a possibly-null object field, and possibly an
	 * array.
	 * <p>
	 * If <code>anObject</code> is an array, then each element is a
	 * possibly-null object.
	 */
	private static int hashObject(int aSeed, Object anObject) {
		int result = aSeed;
		if (anObject == null) {
			result = hashOtherHashCode(result, 0);
		} else if (isArray(anObject)) {
			Object[] array = (Object[]) anObject;
			for (int i = 0; i < array.length; ++i) {
				Object item = array[i];
				result = hashObject(result, item);
			}
		} else {
			result = hashOtherHashCode(result, anObject.hashCode());
		}
		return result;
	}

	private static final int ODD_PRIME_NUMBER = 37;

	/**
	 * Core hash calculation from other hash code.
	 */
	private static int hashOtherHashCode(int aSeed, int otherHashCode) {
		return (ODD_PRIME_NUMBER * aSeed) + otherHashCode;
	}

	private static boolean isArray(Object anObject) {
		return anObject.getClass().isArray();
	}

}
