package net.sf.seaf.util;

import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

/**
 * Utility class for random values generation.
 */
public class Generator {

	/**
	 * Default constructor.
	 */
	public Generator() {
	}

	// base utility methods:

	/**
	 * Generate random number with value between minimum and maximum
	 * (exclusive).
	 * 
	 * @param min
	 *            minimum value, inclusive
	 * @param max
	 *            maximum value, exclusive
	 * @return random number
	 */
	public int numberExcl(int min, int max) {
		double r = Math.random();
		return (int) (r * (max - min)) + min;
	}

	/**
	 * Generate random number with value between minimum and maximum
	 * (inclusive).
	 * 
	 * @param min
	 *            minimum value, inclusive
	 * @param max
	 *            maximum value, inclusive
	 * @return random number
	 */
	public int numberIncl(int min, int max) {
		return numberExcl(min, max + 1);
	}

	// number values generation methods:

	/**
	 * Generate random number with the specified length.
	 * 
	 * @param length
	 *            length of number to be generated in digits
	 * @return random number
	 */
	public long number(int length) {
		int from = 1;
		StringBuilder sb = new StringBuilder();
		for (int i = 0; i < length; i++) {
			int code = numberIncl(from, 9);
			if (1 == from)
				from = 0;
			sb.append((char) (code + '0'));
		}
		return Long.valueOf(sb.toString());
	}

	/**
	 * Generate random number with length in the specified interval, inclusive.
	 * 
	 * @param min
	 *            minimum length in digits, inclusive
	 * @param max
	 *            maximum length in digits, inclusive
	 * @return random number
	 */
	public long number(int min, int max) {
		return number(numberIncl(min, max));
	}

	/**
	 * Generate random phone number.
	 * 
	 * @return random phone number
	 */
	public String phone() {
		return "" + number(3, 4) + " " + number(3, 4) + " " + number(3, 4);
	}

	// temporal values generation methods

	/**
	 * Generate random date with the given year and month.
	 * 
	 * @param year
	 *            full year
	 * @param month
	 *            month from 0 to 11
	 * @return random date
	 */
	public Calendar date(int year, int month) {
		int day = numberIncl(1, 28);
		return new GregorianCalendar(year, month, day);
	}

	/**
	 * Generate random date with the given year.
	 * 
	 * @param year
	 *            full year
	 * @return random date
	 */
	public Calendar date(int year) {
		int month = numberIncl(0, 11);
		return date(year, month);
	}

	/**
	 * Generate random date with the given year.
	 * 
	 * @return random date
	 */
	public Calendar date() {
		int currentYear = GregorianCalendar.getInstance().get(
				GregorianCalendar.YEAR);
		int year = numberIncl(1950, currentYear);
		return date(year);
	}

	/**
	 * Generate random time.
	 * 
	 * @return random time
	 */
	public Calendar time() {
		String text = "" + numberIncl(0, 23) + ":" + numberIncl(0, 59);
		SimpleDateFormat formatter = new SimpleDateFormat("H:mm");
		ParsePosition pos = new ParsePosition(0);
		Date timeValue = formatter.parse(text, pos);
		Calendar cal = GregorianCalendar.getInstance();
		cal.setTime(timeValue);
		return cal;
	}

	// word values generation methods

	/**
	 * Frequencies of letters in English language.
	 */
	public static final double[] LETTER_FREQS = { 0.08167, 0.09659, 0.12441,
			0.16694, 0.29396, 0.31624, 0.33639, 0.39733, 0.46699, 0.46852,
			0.47624, 0.51649, 0.54055, 0.60804, 0.68311, 0.70240, 0.70335,
			0.76322, 0.82649, 0.91705, 0.94463, 0.95441, 0.97801, 0.97951,
			0.99925 };

	/**
	 * Generate random letter 0-based index using the letter frequencies of
	 * English.
	 * 
	 * @return random letter 0-based index
	 */
	public int englishLetterCode() {
		double r = Math.random();
		for (int i = 0; i < LETTER_FREQS.length; i++)
			if (r < LETTER_FREQS[i])
				return i;
		return LETTER_FREQS.length;
	}

	/**
	 * Generate random word with the specified length.
	 * 
	 * @param length
	 *            length of word to be generated in characters
	 * @return random word
	 */
	public String word(int length) {
		StringBuilder sb = new StringBuilder();
		for (int i = 0; i < length; i++) {
			char ch = (char) (englishLetterCode() + 'a');
			if (0 == i)
				ch = Character.toUpperCase(ch);
			sb.append(ch);
		}
		return sb.toString();
	}

	/**
	 * Generate random word with length in the specified interval, inclusive.
	 * 
	 * @param min
	 *            minimum length in characters, inclusive
	 * @param max
	 *            maximum length in characters, inclusive
	 * @return random word
	 */
	public String word(int min, int max) {
		return word(numberIncl(min, max));
	}

	/**
	 * Generate random sentence of the specified length in characters.
	 * 
	 * @param min
	 *            minimum length in characters
	 * @param max
	 *            maximum length in characters
	 * @return random sentence
	 */
	public String sentence(int min, int max) {
		StringBuilder sb = new StringBuilder();
		int wordLengthMin = 3, wordLengthMax = 8;
		int len = numberIncl(min, max);
		while (sb.length() < len) {
			String word = word(wordLengthMin, wordLengthMax);
			if (0 != sb.length())
				word = word.toLowerCase();
			sb.append(word);
			sb.append(' ');
		}
		if (sb.length() > max)
			sb.setLength(max);
		return sb.toString();
	}

}