package net.sf.seaf.factory.impl;

import java.util.Iterator;
import java.util.Map;

import net.sf.seaf.exception.SeafRuntimeException;
import net.sf.seaf.exception.SeafTypeResolvingException;
import net.sf.seaf.factory.Factory;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.ListableBeanFactory;

/**
 * Gets instances from a Spring {@link ListableBeanFactory}.
 * <p>
 * A bean is retrieved from the Spring bean factory when there is exactly one
 * instance defined for the requested type.
 */
public class SpringFactory implements Factory {

	private final ListableBeanFactory springFactory;
	private final Logger log = LoggerFactory.getLogger(SpringFactory.class);

	/**
	 * Initialize the SpringFactory with an instance of Spring
	 * {@link ListableBeanFactory}.
	 * 
	 * @param springFactory
	 */
	public SpringFactory(ListableBeanFactory springFactory) {
		this.springFactory = springFactory;
	}

	/**
	 * Return a bean of the specified type from Spring bean factory.
	 * 
	 * @param type
	 *            The type of the requested bean
	 * @return The bean instance
	 * @throws SeafTypeResolvingException
	 *             Thrown when there are multiple instances or no instance of
	 *             the requested type defined
	 */
	public <Type> Type getInstanceOf(Class<Type> type)
			throws SeafRuntimeException {
		Iterator<Type> iterator = getBeansOfType(type).values().iterator();
		if (!iterator.hasNext())
			throw new SeafTypeResolvingException(
					"No bean of the type defined: " + type);
		Type result = iterator.next();
		if (iterator.hasNext())
			throw new SeafTypeResolvingException(
					"Multiple beans of the type defined: " + type);
		log.trace("Retrieved bean from Spring context {}", result);
		return result;
	}

	private <Type> Map<String, Type> getBeansOfType(Class<Type> type) {
		return springFactory.getBeansOfType(type);
	}

}
