package net.sf.seaf.factory.impl;

import net.sf.seaf.exception.SeafInstantiationException;
import net.sf.seaf.factory.Factory;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Factory that always returns a new instance of the requested type created via
 * {@link Class#newInstance()}. The requested type must therefore have a public
 * default constructor.
 * <p>
 * Checked exceptions are re-thrown as {@link SeafInstantiationException} with
 * the original exception wrapped.
 * 
 * @see Class#newInstance()
 */
public class DefaultInstantiatingFactory implements Factory {

	private final Logger log = LoggerFactory
			.getLogger(DefaultInstantiatingFactory.class);

	public final <Type> Type getInstanceOf(Class<Type> type)
			throws SeafInstantiationException {
		log.trace("Instantiating type {}", type);
		try {
			return type.newInstance();
		} catch (InstantiationException e) {
			throw processException(type, e);
		} catch (IllegalAccessException e) {
			throw processException(type, e);
		}
	}

	private <Type> SeafInstantiationException processException(
			Class<Type> type, Exception e) {
		String msg = "Cannot instantiate " + type
				+ ", does it have a public default constuctor?";
		log.error(msg, e);
		return new SeafInstantiationException(msg, e);
	}

}
