package net.sf.seaf.factory.impl;

import java.util.HashMap;
import java.util.Map;

import net.sf.seaf.factory.Factory;
import net.sf.seaf.factory.impl.support.DelegatingFactoryBase;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Factory that caches object instances.
 * <p>
 * On each request, if an instance of the requested type has already been
 * cached, returns the cached instance as is. Otherwise delegates its
 * instantiation to a supplied instantiating factory, caches the newly created
 * instance for further requests and returns it.
 */
public class CachingFactory extends DelegatingFactoryBase implements Factory {

	private final Map<Class<?>, Object> cache = new HashMap<Class<?>, Object>();
	private final Logger log = LoggerFactory.getLogger(CachingFactory.class);

	/**
	 * Default empty constructor. The instantiating factory must be supplied via
	 * its setter method.
	 */
	public CachingFactory() {
	}

	/**
	 * Full constructor.
	 * 
	 * @param instantiatingFactory
	 *            The instantiating factory
	 */
	public CachingFactory(Factory instantiatingFactory) {
		super(instantiatingFactory);
	}

	public synchronized final <Type> Type getInstanceOf(Class<Type> type) {
		if (getCache().containsKey(type)) {
			@SuppressWarnings("unchecked")
			Type cachedInstance = (Type) getCache().get(type);
			log.trace("Returning cached instance {}", cachedInstance);
			return cachedInstance;
		}

		log.trace("Delegating instantiation of type {} "
				+ "to instantiating factory", type);
		Type newInstance = getInstantiatingFactory().getInstanceOf(type);
		getCache().put(type, newInstance);
		log.trace("Returning newly created instance {}", newInstance);
		return newInstance;
	}

	private Map<Class<?>, Object> getCache() {
		return cache;
	}

}
