package net.sf.seaf.mojo.util;

import java.io.File;
import java.util.HashMap;
import java.util.Map;

import net.sf.xsltmp.FromManyBase;
import net.sf.xsltmp.ManyToDynamicMojo;
import net.sf.xsltmp.ManyToManyMojo;
import net.sf.xsltmp.ManyToOneMojo;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;
import org.codehaus.plexus.archiver.manager.ArchiverManager;

public class ServiceGenerator extends GeneratorBase {

	private final String serviceSrcIncludes;
	private final File mergedServiceFile;
	private final File serviceReportDir;
	private final String applicationName;
	private final String applicationVersion;
	private final String mergedPersistenceFile;
	private final File generatedXmlDir;
	private final File generatedJavaDir;

	public ServiceGenerator(MavenProject project,
			ArchiverManager archiverManager, Log log, File configurationDir,
			String sourceEncoding, String serviceSrcIncludes,
			File mergedServiceFile, File serviceReportDir,
			String applicationName, String applicationVersion,
			String mergedPersistenceFile, File generatedXmlDir,
			File generatedJavaDir) {
		super(project, archiverManager, log, configurationDir, sourceEncoding);
		this.serviceSrcIncludes = serviceSrcIncludes;
		this.mergedServiceFile = mergedServiceFile;
		this.serviceReportDir = serviceReportDir;
		this.applicationName = applicationName;
		this.applicationVersion = applicationVersion;
		this.mergedPersistenceFile = mergedPersistenceFile;
		this.generatedXmlDir = generatedXmlDir;
		this.generatedJavaDir = generatedJavaDir;
	}

	public void regenerateService() throws MojoExecutionException,
			MojoFailureException {
		mergeService();
		generateServiceReport();
		generateServiceEnums();
		generateServiceBeans();
		generateServiceInterface();
		generateServiceFactory();
		generateServiceControllerInterface();
	}

	private void mergeService() throws MojoExecutionException,
			MojoFailureException {
		ManyToOneMojo mojo = new ManyToOneMojo();
		initMojoBaseProperties(mojo);
		mojo.setSrcIncludes(serviceSrcIncludes);
		mojo.setXslTemplate("generator/seaf-commons/Merge.xsl");
		mojo.setDestFile(mergedServiceFile);
		mojo.setForce(true);
		mojo.execute();
	}

	private void generateServiceReport() throws MojoExecutionException,
			MojoFailureException {
		ManyToOneMojo mojo = new ManyToOneMojo();
		initServiceMojo(mojo, "Report");
		mojo.setSrcFile(mergedServiceFile);
		mojo.setDestFile(new File(serviceReportDir, "service-report.html"));
		Map parameters = new HashMap();
		parameters.put("application-name", applicationName);
		parameters.put("version", applicationVersion);
		parameters.put("persistence-merged-file-name", mergedPersistenceFile);
		mojo.setParameters(parameters);
		mojo.execute();
	}

	private void generateServiceEnums() throws MojoExecutionException,
			MojoFailureException {
		ManyToDynamicMojo mojo = new ManyToDynamicMojo();
		initServiceMojo(mojo, "Enum");
		mojo.setDestDir(generatedJavaDir);
		Map parameters = new HashMap();
		parameters.put("persistence-merged-file-name", mergedPersistenceFile);
		mojo.setParameters(parameters);
		mojo.execute();
	}

	private void generateServiceBeans() throws MojoExecutionException,
			MojoFailureException {
		ManyToManyMojo mojo = new ManyToManyMojo();
		initServiceMojo(mojo, "Beans");
		mojo.setDestDir(generatedXmlDir);
		mojo.setFileNameRegex("\\.xml");
		mojo.setFileNameReplacement("-beans\\.xml");
		Map parameters = new HashMap();
		parameters.put("persistence-merged-file-name", mergedPersistenceFile);
		parameters.put("service-merged-file-name", mergedServiceFile);
		mojo.setParameters(parameters);
		mojo.execute();
	}

	private void generateServiceInterface() throws MojoExecutionException,
			MojoFailureException {
		ManyToDynamicMojo mojo = new ManyToDynamicMojo();
		initServiceMojo(mojo, "ServiceInterface");
		mojo.setDestDir(generatedJavaDir);
		Map parameters = new HashMap();
		parameters.put("service-merged-file-name", mergedServiceFile);
		mojo.setParameters(parameters);
		mojo.execute();
	}

	private void generateServiceFactory() throws MojoExecutionException,
			MojoFailureException {
		ManyToDynamicMojo mojo = new ManyToDynamicMojo();
		initServiceMojo(mojo, "ServiceFactory");
		mojo.setDestDir(generatedJavaDir);
		Map parameters = new HashMap();
		parameters.put("service-merged-file-name", mergedServiceFile);
		mojo.setParameters(parameters);
		mojo.execute();
	}

	private void generateServiceControllerInterface()
			throws MojoExecutionException, MojoFailureException {
		ManyToDynamicMojo mojo = new ManyToDynamicMojo();
		initServiceMojo(mojo, "ControllerInterface");
		mojo.setDestDir(generatedJavaDir);
		Map parameters = new HashMap();
		parameters.put("persistence-merged-file-name", mergedPersistenceFile);
		parameters.put("service-merged-file-name", mergedServiceFile);
		mojo.setParameters(parameters);
		mojo.execute();
	}

	private void initServiceMojo(FromManyBase mojo, String template) {
		initMojoBaseProperties(mojo);
		mojo.setSrcIncludes(serviceSrcIncludes);
		mojo.setXslTemplate("generator/seaf-service/" + template + ".xsl");
	}

}
