package net.sf.seaf.mojo;

import java.io.File;

import net.sf.seaf.mojo.util.BeansGenerator;
import net.sf.seaf.mojo.util.FilesChangedUtils;
import net.sf.seaf.mojo.util.PersistenceGenerator;
import net.sf.seaf.mojo.util.ServiceGenerator;
import net.sf.seaf.mojo.util.WebGenerator;
import net.sf.xsltmp.FinalizeMojo;
import net.sf.xsltmp.InitializeMojo;
import net.sf.xsltmp.util.AddSourcesUtils;
import net.sf.xsltmp.util.TimestampUtils;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.project.MavenProject;
import org.codehaus.plexus.archiver.manager.ArchiverManager;

/**
 * Generate all SEAF artifacts.
 * 
 * @goal generate-all
 * @phase generate-sources
 * @requiresDependencyResolution compile
 */
public class GenerateAllMojo extends AbstractMojo {

	// user setup properties:

	/**
	 * Set to <code>true</code> to generate the persistence layer.
	 * 
	 * @parameter default-value="false"
	 */
	private boolean generatePersistence;

	/**
	 * Set to <code>true</code> to generate the service layer.
	 * 
	 * @parameter default-value="false"
	 */
	private boolean generateService;

	/**
	 * Set to <code>true</code> to generate the web layer.
	 * 
	 * @parameter default-value="false"
	 */
	private boolean generateWeb;

	/**
	 * Which files from the source directory tree should be included in the
	 * generation of the persistence layer.
	 * <p>
	 * Specifies a fileset of source files in the comma- or space-separated list
	 * of patterns of files.
	 * 
	 * @parameter default-value="**\/*-persistence.xml"
	 * @required
	 */
	private String persistenceSrcIncludes;

	/**
	 * Which files from the source directory tree should be included in the
	 * generation of the service layer.
	 * <p>
	 * Specifies a fileset of source files in the comma- or space-separated list
	 * of patterns of files.
	 * 
	 * @parameter default-value="**\/*-service.xml"
	 * @required
	 */
	private String serviceSrcIncludes;

	/**
	 * Which files from the source directory tree should be included in the
	 * generation of the web layer.
	 * <p>
	 * Specifies a fileset of source files in the comma- or space-separated list
	 * of patterns of files.
	 * 
	 * @parameter default-value="**\/*-web.xml"
	 * @required
	 */
	private String webSrcIncludes;

	/**
	 * Directory containing the generator XML configuration files.
	 * 
	 * @parameter default-value="${project.basedir}/src/main/gen-config"
	 * @required
	 */
	private File configurationDir;

	/**
	 * Read the source files (XSL templates and XML transformation sources)
	 * using this encoding.
	 * 
	 * @parameter expression="${project.build.sourceEncoding}"
	 */
	private String sourceEncoding;

	/**
	 * The directory where to generate the persistence report.
	 * 
	 * @parameter 
	 *            default-value="${project.build.directory}/site/seaf-persistence"
	 * @required
	 */
	private File persistenceReportDir;

	/**
	 * The directory where to generate the service report.
	 * 
	 * @parameter default-value="${project.build.directory}/site/seaf-service"
	 * @required
	 */
	private File serviceReportDir;

	/**
	 * Application name displayed to user.
	 * 
	 * @parameter default-value= "${project.name}"
	 */
	private String applicationName;

	/**
	 * Application version displayed to user.
	 * 
	 * @parameter default-value= "${project.version}"
	 */
	private String applicationVersion;

	/**
	 * Generated file with SQL "insert" statements to initialize enumeration
	 * entities in database.
	 * 
	 * @parameter 
	 *            default-value="src/main/resources/${project.artifactId}-enum-init.sql"
	 * @required
	 */
	private File enumInitFile;

	/**
	 * Hibernate.cfg.xml source file.
	 * 
	 * @parameter default-value="src/main/resources/hibernate.cfg.xml"
	 */
	private File hibernateCfgXmlSrcFile;

	/**
	 * Hibernate.cfg.xml destination file.
	 * 
	 * @parameter 
	 *            default-value="${project.build.outputDirectory}/hibernate.cfg.xml"
	 */
	private File hibernateCfgXmlDestFile;

	/**
	 * Session factory context source file.
	 * 
	 * @parameter 
	 *            default-value="src/main/resources/data-access-context-local.xml"
	 */
	private File sessionFactoryContextSrcFile;

	/**
	 * Session factory context destination file.
	 * 
	 * @parameter default-value=
	 *            "${project.build.outputDirectory}/data-access-context-local.xml"
	 */
	private File sessionFactoryContextDestFile;

	/**
	 * Persistence context destination file.
	 * 
	 * @parameter default-value=
	 *            "${project.build.outputDirectory}/persistence-context.xml"
	 */
	private File persistenceContextFile;

	/**
	 * Code table init source file.
	 * 
	 * @parameter default-value= "src/main/gen-config/setup/code-table-init.xml"
	 */
	private File codeTableInitSrcFile;

	/**
	 * Code table init destination file.
	 * 
	 * @parameter default-value=
	 *            "src/main/resources/${project.artifactId}-code-table-init.sql"
	 */
	private File codeTableInitDestFile;

	/**
	 * The mode of generating the persistence layer. Default value is
	 * <code>full</code>, which generates all persistence layer artifacts. Set
	 * to <code>entities-only</code> to only generate entities (and related
	 * artifacts).
	 * 
	 * @parameter default-value="full"
	 * @required
	 */
	private String generatePersistenceMode;

	// internal properties:

	/**
	 * Location of file to hold merged persistence XML.
	 * 
	 * @parameter default-value=
	 *            "${project.build.outputDirectory}/net/sf/seaf/persistence/merged-persistence.xml"
	 * @required
	 */
	private String mergedPersistenceFile;

	/**
	 * Location of file to hold merged service XML.
	 * 
	 * @parameter default-value=
	 *            "${project.build.outputDirectory}/net/sf/seaf/service/merged-service.xml"
	 * @required
	 */
	private File mergedServiceFile;

	/**
	 * Location of file to hold merged web XML.
	 * 
	 * @parameter default-value=
	 *            "${project.build.outputDirectory}/net/sf/seaf/web/merged-web.xml"
	 * @required
	 */
	private File mergedWebFile;

	/**
	 * Directory to hold intermediary XML files.
	 * 
	 * @parameter default-value="${project.build.directory}/generated-xml/seaf"
	 * @required
	 */
	private File generatedXmlDir;

	/**
	 * Directory to hold generated Java source files.
	 * 
	 * @parameter 
	 *            default-value="${project.build.directory}/generated-sources/seaf"
	 * @required
	 */
	private File generatedJavaDir;

	/**
	 * Where should the generated sources directory be added. Valid values are:
	 * sources, test-sources.
	 * <p>
	 * This is a shortcut to avoid configuring the build-helper-maven-plugin.
	 * 
	 * @parameter default-value="sources"
	 */
	private String addGeneratedSourcesTo;

	/**
	 * Where should the generated resources directory be added. Valid values
	 * are: resources, test-resources.
	 * <p>
	 * This is a shortcut to avoid configuring the build-helper-maven-plugin.
	 * 
	 * @parameter default-value="resources"
	 */
	private String addGeneratedResourcesTo;

	// Plexus infrastructure properties:

	/**
	 * The Maven project model.
	 * 
	 * @parameter expression="${project}"
	 * @required
	 * @readonly
	 */
	private MavenProject project;

	/**
	 * To look up Archiver/UnArchiver implementations.
	 * 
	 * @component
	 */
	private ArchiverManager archiverManager;

	public void execute() throws MojoExecutionException, MojoFailureException {
		TimestampUtils timestampUtils = new TimestampUtils(project);
		FilesChangedUtils filesChangedUtils = new FilesChangedUtils();
		File timestampFile = timestampUtils.getTimestampFile();

		if (!generatePersistence && !generateWeb && !generateService) {
			getLog().warn(
					"No layer set for generation. (Did you forget to set one?)");
			return;
		}

		AddSourcesUtils addSourcesUtils = new AddSourcesUtils(project, getLog());
		addSourcesUtils.addSources(addGeneratedSourcesTo, generatedJavaDir);
		addSourcesUtils.addResources(addGeneratedResourcesTo, generatedJavaDir);

		// find what's changed:
		boolean persistenceChanged = false;
		boolean hibernateCfgChanged = false;
		boolean sessionFactoryContextChanged = false;
		boolean codeTableInitChanged = false;
		boolean serviceChanged = false;
		boolean webChanged = false;
		if (generatePersistence) {
			if (filesChangedUtils.filesChanged(configurationDir,
					persistenceSrcIncludes, timestampFile)) {
				persistenceChanged = true;
				hibernateCfgChanged = true;
				sessionFactoryContextChanged = true;
			}
			if (!hibernateCfgChanged
					&& hibernateCfgXmlSrcFile.exists()
					&& hibernateCfgXmlSrcFile.lastModified() > timestampFile
							.lastModified()) {
				hibernateCfgChanged = true;
			}
			if (!sessionFactoryContextChanged
					&& sessionFactoryContextSrcFile.exists()
					&& sessionFactoryContextSrcFile.lastModified() > timestampFile
							.lastModified()) {
				sessionFactoryContextChanged = true;
			}
			if (codeTableInitSrcFile.exists()
					&& codeTableInitSrcFile.lastModified() > timestampFile
							.lastModified()) {
				codeTableInitChanged = true;
			}
		}
		if (generateService
				&& filesChangedUtils.filesChanged(configurationDir,
						serviceSrcIncludes, timestampFile)) {
			serviceChanged = true;
		}
		if (generateWeb
				&& filesChangedUtils.filesChanged(configurationDir,
						webSrcIncludes, timestampFile)) {
			webChanged = true;
		}
		boolean changed = persistenceChanged || hibernateCfgChanged
				|| sessionFactoryContextChanged || codeTableInitChanged
				|| serviceChanged || webChanged;
		boolean beansChanged = persistenceChanged || serviceChanged
				|| webChanged;

		// perform actions:
		if (changed) {
			initialize();
		}
		PersistenceGenerator persistenceGenerator = createPersistenceGenerator();
		if (persistenceChanged) {
			getLog().info(
					"Regenerating persistence layer from configuration in mode "
							+ generatePersistenceMode + "...");
			persistenceGenerator.regeneratePersistence();
		}
		if (hibernateCfgChanged) {
			if (persistenceGenerator.isFull()) {
				getLog().info("Regenerating hibernate.cfg.xml file...");
				persistenceGenerator.generatePersistenceHibernateCfg();
			}
		}
		if (sessionFactoryContextChanged) {
			if (persistenceGenerator.isFull()) {
				getLog().info("Regenerating session factory context file...");
				persistenceGenerator.generatePersistenceSessionFactoryContext();
			}
		}
		if (codeTableInitChanged) {
			if (persistenceGenerator.isFull()) {
				getLog().info("Regenerating code table init file...");
				persistenceGenerator.generatePersistenceCodeTableInit();
			}
		}
		if (serviceChanged) {
			getLog().info("Regenerating service layer from configuration...");
			ServiceGenerator serviceGenerator = createServiceGenerator();
			serviceGenerator.regenerateService();
		}
		if (webChanged) {
			getLog().info("Regenerating web layer from configuration...");
			WebGenerator webGenerator = createWebGenerator();
			webGenerator.regenerateWeb();
		}
		if (beansChanged) {
			getLog().info("Regenerating beans...");
			BeansGenerator beansGenerator = createBeansGenerator();
			beansGenerator.generateBeans();
		}
		if (changed) {
			finalize_();
		}
		if (!changed) {
			getLog().info("Up to date.");
		}
	}

	private void initialize() throws MojoExecutionException,
			MojoFailureException {
		InitializeMojo mojo = new InitializeMojo();
		mojo.setProject(project);
		mojo.setLog(getLog());
		mojo.execute();
	}

	private void finalize_() throws MojoExecutionException,
			MojoFailureException {
		FinalizeMojo mojo = new FinalizeMojo();
		mojo.setProject(project);
		mojo.setLog(getLog());
		mojo.execute();
	}

	private BeansGenerator createBeansGenerator() {
		return new BeansGenerator(project, archiverManager, getLog(),
				configurationDir, sourceEncoding, generatedXmlDir,
				generatedJavaDir);
	}

	private WebGenerator createWebGenerator() {
		return new WebGenerator(project, archiverManager, getLog(),
				configurationDir, sourceEncoding, webSrcIncludes,
				mergedWebFile, generatedXmlDir, generatedJavaDir);
	}

	private ServiceGenerator createServiceGenerator() {
		return new ServiceGenerator(project, archiverManager, getLog(),
				configurationDir, sourceEncoding, serviceSrcIncludes,
				mergedServiceFile, serviceReportDir, applicationName,
				applicationVersion, mergedPersistenceFile, generatedXmlDir,
				generatedJavaDir);
	}

	private PersistenceGenerator createPersistenceGenerator() {
		return new PersistenceGenerator(project, archiverManager, getLog(),
				configurationDir, sourceEncoding, persistenceSrcIncludes,
				new File(mergedPersistenceFile), persistenceReportDir,
				applicationName, applicationVersion, generatedXmlDir,
				generatedJavaDir, enumInitFile, persistenceContextFile,
				hibernateCfgXmlSrcFile, hibernateCfgXmlDestFile,
				sessionFactoryContextSrcFile, sessionFactoryContextDestFile,
				codeTableInitSrcFile, codeTableInitDestFile,
				generatePersistenceMode);
	}

}
