// Copyright (c) 2006 - 2008, Markus Strauch.
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
// 
// * Redistributions of source code must retain the above copyright notice, 
// this list of conditions and the following disclaimer.
// * Redistributions in binary form must reproduce the above copyright notice, 
// this list of conditions and the following disclaimer in the documentation 
// and/or other materials provided with the distribution.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
// THE POSSIBILITY OF SUCH DAMAGE.

package net.sf.sdedit.editor.apple;

import java.io.File;

import net.sf.sdedit.editor.Editor;

/**
 * Proxy class for installing an <tt>ApplicationListener</tt> that receives
 * events generated by MacOS and sends them to the editor.
 * <p>
 * If the application runs on MacOS, there should be a class that actually
 * performs the installation, with the name <tt>ApplicationListenerImpl</tt>,
 * located in the same package as this class.
 * 
 * @author Markus Strauch
 * 
 */
public abstract class AppInstaller {

	private static AppInstaller instance = null;

	/**
	 * Constructor.
	 */
	protected AppInstaller() {

	}

	/**
	 * Installs an <tt>ApplicationListener</tt> that sends events generated by
	 * MacOS to the given editor.
	 * 
	 * @param editor
	 *            the editor
	 */
	abstract void install(Editor editor);

	/**
	 * Installs an <tt>ApplicationListener</tt> that receives application
	 * events generated by MacOS or does nothing if not running MacOS.
	 * 
	 * @param editor
	 *            the <tt>Editor</tt> instance
	 */
	@SuppressWarnings("unchecked")
	public static void installApplication(Editor editor) {
		if (instance != null) {
			throw new IllegalStateException("Application already installed");
		}
		try {
			Class<? extends AppInstaller> implClass = (Class<? extends AppInstaller>) Class
					.forName(AppInstaller.class.getName() + "Impl");
			instance = implClass.newInstance();
			instance.install(editor);
		} catch (Throwable t) {
			t.printStackTrace();
		}
	}

	/**
	 * Returns a file to be loaded, if a MacOS requests a file to be opened at
	 * startup.
	 * 
	 * @return a file to be loaded
	 */
	protected abstract File fileToLoad();

	/**
	 * If an <tt>ApplicationListener</tt> has been successfully installed and
	 * if it has received an event saying that a file is to be loaded at startup
	 * time, this file is returned, otherwise <tt>null</tt>
	 * 
	 * @return a file to be loaded on startup running MacOS
	 */
	public static File getFileToLoad() {
		if (instance == null) {
			return null;
		}
		return instance.fileToLoad();
	}
}
