/*
 * $Id: AlarmClock.java,v 1.7 2009/12/20 17:30:26 oboehm Exp $
 *
 * Copyright (c) 2008 by Oliver Boehm
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express orimplied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * (c)reated 22.02.2009 by oliver (ob@oasd.de)
 */
package patterntesting.sample.jmx;

import java.lang.management.ManagementFactory;
import java.util.*;
import java.util.concurrent.TimeUnit;

import javax.management.*;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.impl.LogFactoryImpl;

import patterntesting.runtime.jmx.MBeanRegistry;
import patterntesting.runtime.util.ThreadUtil;

/**
 * This is a simple example how you can to write an MBean and how you can use
 * the MBeanRegistry interface to register the MBean at the MBean server.
 *
 * @author <a href="boehm@javatux.de">oliver</a>
 * @since 22.02.2009
 * @version $Revision: 1.7 $
 */
public class AlarmClock extends TimerTask implements AlarmClockMBean,
        MBeanRegistry {

    private static final Log log = LogFactoryImpl.getLog(AlarmClock.class);
    private Date alarmTime;
    private Timer timer;

    /**
     * Instantiates a new alarm clock.
     */
    public AlarmClock() {
        this.alarmTime = new Date();
    }

    /**
     * Instantiates a new alarm clock.
     *
     * @param alarmTime the alarm time
     */
    public AlarmClock(Date alarmTime) {
        this.alarmTime = alarmTime;
    }

    /* (non-Javadoc)
     * @see patterntesting.sample.jmx.AlarmClockMBean#getTime()
     */
    public Date getTime() {
        return new Date();
    }

    /* (non-Javadoc)
     * @see patterntesting.sample.jmx.AlarmClockMBean#getAlarmTime()
     */
    public Date getAlarmTime() {
        return alarmTime;
    }

    /* (non-Javadoc)
     * @see patterntesting.sample.jmx.AlarmClockMBean#setAlarmTime(java.util.Date)
     */
    public synchronized void setAlarmTime(final Date alarmTime) {
        this.alarmTime = (Date) alarmTime.clone();
        this.activateAlarm();
    }

    /* (non-Javadoc)
     * @see patterntesting.sample.jmx.AlarmClockMBean#getTimeToWaitInMillis()
     */
    public synchronized long getTimeToWaitInMillis() {
        return this.alarmTime.getTime() - System.currentTimeMillis();
    }

    /* (non-Javadoc)
     * @see patterntesting.sample.jmx.AlarmClockMBean#setTimeToWaitInMillis(long)
     */
    public synchronized void setTimeToWaitInMillis(final long t) {
        setAlarmTime(new Date(System.currentTimeMillis() + t));
    }

    /**
     * We can use the timer only once so first a new timer is created.
     * This is also true for the AlarmClock (which implements the TimerTask).
     * We need a new AlarmClock for the newly created timer here.
     */
    private void activateAlarm() {
        timer = new Timer("AlarmClock");
        timer.schedule(new AlarmClock(this.alarmTime), this.alarmTime);
    }

    /* (non-Javadoc)
     * @see patterntesting.sample.jmx.AlarmClockMBean#deactivateAlarm()
     */
    public synchronized void deactivateAlarm() {
        if (timer == null) {
            log.debug("alarm already deactivated");
        } else {
            timer.cancel();
            this.alarmTime = new Date();
            log.debug("alarm deactivated");
        }
    }

    /**
     * Checks if is activated.
     *
     * @return true, if is activated
     */
    public boolean isActivated() {
        return this.alarmTime.after(new Date());
    }

    /* (non-Javadoc)
     * @see patterntesting.sample.jmx.AlarmClockMBean#ring()
     */
    public void ring() {
        log.info("rrrrring");
        this.alarmTime = null;
    }

    /* (non-Javadoc)
     * @see java.util.TimerTask#run()
     */
    @Override
    public void run() {
        this.ring();
    }

    /**
     * The main method.
     *
     * @param args the arguments
     *
     * @throws JMException the JM exception
     */
    public static void main(final String[] args) throws JMException {
        registerAlarmClockManual();
        registerAlarmClockSimple();
        ThreadUtil.sleep(300, TimeUnit.SECONDS);
    }

    /**
     * This is an example how to register an AlarmClock at the MBeanServer.
     *
     * @throws JMException the JM exception
     */
    public static void registerAlarmClockManual() throws JMException {
        ObjectName name = new ObjectName(
                "patterntesting.sample.jmx:type=ManualRegisteredClock");
        MBeanServer server = ManagementFactory.getPlatformMBeanServer();
        server.registerMBean(new AlarmClock(), name);
        log.info(name + " registered at " + server);
    }

    /**
     * This is an example how to register an AlarmClock provided by the
     * default imolmentation of the MBeanRegistry interface.
     *
     * @throws JMException the JM exception
     *
     * @{@link MBeanRegistry}
     */
    public static void registerAlarmClockSimple() throws JMException {
        AlarmClock clock = new AlarmClock();
        clock.registerAsMBean();
    }

}

/*
 * $Log: AlarmClock.java,v $
 * Revision 1.7  2009/12/20 17:30:26  oboehm
 * trailing spaces removed
 *
 * Revision 1.6  2009/12/12 16:54:34  oboehm
 * nearly all args are now final
 *
 * Revision 1.5  2009/12/12 15:30:02  oboehm
 * some checkstyle warnings fixed
 *
 * Revision 1.4  2009/09/21 08:19:25  oboehm
 * javadoc warnings removed
 *
 * Revision 1.3  2009/03/09 21:33:29  oboehm
 * Bomb class as example for the @TestException added
 *
 * Revision 1.2  2009/02/27 19:24:04  oboehm
 * NPE fixed after initial deactiveAlarm() call
 *
 * Revision 1.1  2009/02/23 21:58:54  oboehm
 * example for MBean registration added
 *
 * $Source: /cvsroot/patterntesting/PatternTesting08/patterntesting-samples/src/main/java/patterntesting/sample/jmx/AlarmClock.java,v $
 */
