/**
 * OpenDSE is free software: you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 * 
 * OpenDSE is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with OpenDSE. If not, see http://www.gnu.org/licenses/.
 */
package net.sf.opendse.generator;

import static java.lang.Math.min;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import java.util.Set;

import net.sf.opendse.model.Application;
import net.sf.opendse.model.Communication;
import net.sf.opendse.model.Dependency;
import net.sf.opendse.model.ICommunication;
import net.sf.opendse.model.Task;

/**
 * The {@code ApplicationGenerator} contains several methods to generate and
 * transform applications.
 * 
 * @author lukasiewycz
 * 
 */
public class ApplicationGenerator extends Generator {

	protected IdProvider taskId;
	protected IdProvider commId;
	protected IdProvider dependencyId;

	/**
	 * Constructs an {@code ApplicationGenerator} with a random seed.
	 */
	public ApplicationGenerator() {
		this(System.currentTimeMillis());
	}

	/**
	 * Constructs an {@code ApplicationGenerator} with a given seed.
	 * 
	 * @param seed
	 *            the seed
	 */
	public ApplicationGenerator(long seed) {
		this(seed, new IdProvider("p"), new IdProvider("c"), new IdProvider("d"));
	}

	/**
	 * Constructs an {@code ApplicationGenerator}.
	 * 
	 * @param seed
	 *            the seed
	 * @param taskId
	 *            the provider for the task ids
	 * @param commId
	 *            the provider for the communication ids
	 * @param dependencyId
	 *            the proider for the dependency ids
	 */
	public ApplicationGenerator(long seed, IdProvider taskId, IdProvider commId, IdProvider dependencyId) {
		super(new Random(seed));
		this.taskId = taskId;
		this.commId = commId;
		this.dependencyId = dependencyId;
	}

	/**
	 * Returns an application generated by the TGFF algorithm (old).
	 * 
	 * @param x
	 *            the minimal number of tasks
	 * @param id
	 *            the maximum in-degree
	 * @param od
	 *            the maximum out-degree
	 * @return the application
	 */
	public Application<Task, Dependency> generate(int x, int id, int od) {
		Application<Task, Dependency> application = new Application<Task, Dependency>();
		{
			Task task = new Task(taskId.next());
			application.addVertex(task);
		}

		while (application.getVertexCount() < x) {
			if (random.nextBoolean()) {
				// fan-in
				List<Task> candidates = new ArrayList<Task>();
				for (Task task : application) {
					if (application.getSuccessorCount(task) < od) {
						candidates.add(task);
					}
				}

				int q = candidates.size();
				assert (q > 0);

				int o = min(candidates.size(), min(application.getVertexCount(), id));
				int z = rand(0, o);

				if (z > 0) {
					while (candidates.size() > z) {
						candidates.remove(rand(candidates));
					}

					Task task = new Task(taskId.next());
					for (Task c : candidates) {
						Dependency dependency = new Dependency(dependencyId.next());
						application.addEdge(dependency, c, task);
					}
				}

			} else {
				// fan-out
				int minDegree = Integer.MAX_VALUE;
				List<Task> candidates = new ArrayList<Task>();
				for (Task task : application) {
					int d = application.getOutEdges(task).size();
					if (d < minDegree) {
						minDegree = d;
						candidates.clear();
					}
					if (d == minDegree) {
						candidates.add(task);
					}
				}

				if (minDegree < od) {
					Task p = rand(candidates);
					int r = od - application.getSuccessorCount(p);
					int y = rand(0, r);

					for (int i = 0; i < y; i++) {
						Task task = new Task(taskId.next());
						Dependency dependency = new Dependency(dependencyId.next());
						application.addEdge(dependency, p, task);
					}
				}
			}
		}

		return application;
	}

	/**
	 * Adds communication tasks between process tasks.
	 * 
	 * @param application
	 *            the application
	 * @param min
	 *            the minimal number of receivers of a communication
	 * @param max
	 *            the maximal number of receivers of a communicatoin
	 */
	public void insertCommunication(Application<Task, Dependency> application, int min, int max) {

		Set<Task> tasks = new HashSet<Task>(application.getVertices());

		for (Task task : tasks) {
			List<Dependency> edges = new ArrayList<Dependency>(application.getOutEdges(task));

			for (Dependency edge : new ArrayList<Dependency>(edges)) {
				if (application.getDest(edge) instanceof ICommunication) {
					edges.remove(edge);
				}
			}

			while (!edges.isEmpty()) {
				int x = rand(min, max);
				List<Dependency> targets = new ArrayList<Dependency>();

				while (!edges.isEmpty() && targets.size() < x) {
					Dependency edge = rand(edges);
					edges.remove(edge);
					targets.add(edge);
				}

				Task comm = new Communication(commId.next());
				Dependency e = new Dependency(dependencyId.next());
				application.addEdge(e, task, comm);

				for (Dependency edge : targets) {
					Task t = application.getDest(edge);
					application.removeEdge(edge);
					application.addEdge(edge, comm, t);
				}
			}
		}
	}
	
	public Application<Task, Dependency> merge(Collection<Application<Task, Dependency>> applications) {
		Application<Task, Dependency> application = new Application<Task, Dependency>();
		for (Application<Task, Dependency> appl : applications) {
			for (Task task : appl.getVertices()) {
				application.addVertex(task);
			}
			for (Dependency dependency : appl.getEdges()) {
				application.addEdge(dependency, appl.getEndpoints(dependency), appl.getEdgeType(dependency));
			}
		}
		return application;
	}

	public Application<Task, Dependency> merge(Application<Task, Dependency>... applications) {
		Collection<Application<Task, Dependency>> apps = Arrays.asList(applications);
		return merge(apps);
	}

}
