/*===========================================================================
  Copyright (C) 2008-2013 by the Okapi Framework contributors
-----------------------------------------------------------------------------
  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
===========================================================================*/

package net.sf.okapi.lib.preprocessing.filters.common;

import java.security.InvalidParameterException;
import java.util.LinkedList;
import java.util.List;

import net.sf.okapi.common.Event;
import net.sf.okapi.common.IParameters;
import net.sf.okapi.common.ListUtil;
import net.sf.okapi.common.encoder.EncoderManager;
import net.sf.okapi.common.filters.FilterConfiguration;
import net.sf.okapi.common.filters.IFilter;
import net.sf.okapi.common.filters.IFilterConfigurationMapper;
import net.sf.okapi.common.filterwriter.IFilterWriter;
import net.sf.okapi.common.pipeline.IPipelineStep;
import net.sf.okapi.common.resource.MultiEvent;
import net.sf.okapi.common.resource.RawDocument;
import net.sf.okapi.common.resource.StartDocument;
import net.sf.okapi.common.skeleton.ISkeletonWriter;

/**
 * This filter is a wrapper around another filter, used when events generated by the internal 
 * filter should be modified (transformed, pre-processed) for the pipeline steps consuming the filter events.
 * The constructor takes an instance of the internal filter and an array of the needed transformation steps.
 */
public class PreprocessingFilter implements IFilter {

	private IFilter filter;
	private List<IPipelineStep> steps;
	private LinkedList<Event> queue;

	public PreprocessingFilter(IFilter filter, IPipelineStep... steps) {
		if (filter == null) 
			throw new InvalidParameterException("Internal filter is not specified");
		
		if (steps.length == 0) 
			throw new InvalidParameterException("Steps are not specified");
		
		this.queue = new LinkedList<>();
		this.filter = filter;
		this.steps = ListUtil.arrayAsList(steps);		
	}
	
	@Override
	public String getName() {
		return "okf_preprocessing";
	}

	@Override
	public String getDisplayName() {
		return "Preprocesssing Filter";
	}

	@Override
	public void open(RawDocument input) {
		this.open(input, true);
	}

	@Override
	public void open(RawDocument input, boolean generateSkeleton) {
		filter.open(input, generateSkeleton);
		processFilterEvents();
	}

	@Override
	public void close() {
		for (IPipelineStep step : this.steps) {
			step.cancel();
		}
		filter.close();		
		queue.clear();
	}

	@Override
	public boolean hasNext() {
		if (filter.hasNext()) 
			return true;
		else
			return queue.size() > 0;
	}

	/**
	 * Read events from the filter until one output event is formed.
	 * Place that in the queue.
	 */
	private void processFilterEvents() {
		while (filter.hasNext()) {
			Event e = filter.next();
			if (e == null) return;
			if (e.isNoop()) continue; // Get next filter event
			if (e.isStartDocument()) {
				StartDocument sd = e.getStartDocument();
				sd.setFilterId(getName());
				sd.setFilterParameters(this.getParameters()); // Replace internal filter's parameters with this one's
			}
			if (steps != null) {
				for (IPipelineStep step : steps) {
					e = step.handleEvent(e);
				}
				if (e.isNoop()) continue; // Get next filter event
			}
			if (e.isMultiEvent()) {
				MultiEvent me = e.getMultiEvent();
				for (Event ev : me) {
					queue.add(ev);
				}				
			}
			else
				queue.add(e);
			break;
		}
	}
	
	@Override
	public Event next() {
		if (queue.size() > 0) 
			return queue.poll();
		else {
			processFilterEvents();
			if (queue.size() > 0) 
				return queue.poll();
		}
		return null;
	}

	@Override
	public void cancel() {
		close();
	}

	@Override
	public IParameters getParameters() {
		return filter.getParameters();
	}

	@Override
	public void setParameters(IParameters params) {
		filter.setParameters(params);
	}

	@Override
	public void setFilterConfigurationMapper(IFilterConfigurationMapper fcMapper) {
		filter.setFilterConfigurationMapper(fcMapper);
	}

	@Override
	public ISkeletonWriter createSkeletonWriter() {
		return filter.createSkeletonWriter();
	}

	@Override
	public IFilterWriter createFilterWriter() {
		return filter.createFilterWriter();
	}

	@Override
	public EncoderManager getEncoderManager() {
		return filter.getEncoderManager();
	}

	@Override
	public String getMimeType() {
		return filter.getMimeType();
	}

	@Override
	public List<FilterConfiguration> getConfigurations() {
		return filter.getConfigurations();
	}
			
	protected IFilter getFilter() {
		return filter;
	}

	protected void setFilter(IFilter filter) {
		this.filter = filter;
	}

	protected List<IPipelineStep> getSteps() {
		return steps;
	}

}
