/** This file is part of nervalreports.
 *
 * nervalreports is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * nervalreports is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with nervalreports.  If not, see <http://www.gnu.org/licenses/>. */
package net.sf.nervalreports.core.paper;

/** Basic class for creating paper formats. A paper format is defined by paper's page size (width and height), and 
 * respective name definitions for HTML and LaTeX. <br>
 * <b>Note:</b> All children classes must be implemented as singletons.
 * @author farrer */
public abstract class ReportPaperFormat {
	
	/** Default DPI used on PDFReportGenerator. */
	private static final int DEFAULT_USER_SPACE_UNIT_DPI = 72;

	/** Conversion factor from Millimeters to PDF units. */
    private static final float MM_TO_UNITS = 1 / (10 * 2.54f) * DEFAULT_USER_SPACE_UNIT_DPI;
    
    /** Paper width in PDF units. */
    private final float widthInPDFUnits;
    
    /** Paper height in PDF units. */
    private final float heightInPDFUnits;

	/** @return width (in millimeters) of the paper. */
	protected abstract int getWidthInMillimeters();
	/** @return height (in millimeters) of the paper. */
	protected abstract int getHeightInMillimeters();
	
	/** Name of the format used on LaTeX <code>\documentclass</code> */
	public abstract String getTeXName();
	/** Name of the format used on HTML <code>@page size</code> */
	public abstract String getHtmlName();
	
	/** Default constructor */
	protected ReportPaperFormat() {
		widthInPDFUnits = getWidthInMillimeters() * MM_TO_UNITS;
		heightInPDFUnits = getHeightInMillimeters() * MM_TO_UNITS;
	}
	
	/** @return the paper width in PDF units. */
	public float getWidth() {
		return widthInPDFUnits;
	}
	
	/** @return the paper height in PDF units. */
	public float getHeight() {
		return heightInPDFUnits;
	}
	
	
}
