/** This file is part of nervalreports.
 *
 * nervalreports is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * nervalreports is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with nervalreports.  If not, see <http://www.gnu.org/licenses/>. */
package net.sf.nervalreports.core;

import java.nio.charset.Charset;

/** Basic class for converters from a textual file format to another file format.
 * @author farrer */
public abstract class TextualFileConverter {
	
	/** Convert the textual file loaded at the {@link String} to another format, represented by a byte array, using the default machine charset.
	 * @param textualFileAsString loaded file contents.
	 * @return the conversion result.
	 * @throws Exception in case of error at conversion. */
	public byte[] convert(String textualFileAsString) throws Exception {
		return convert(textualFileAsString, null);
	}
	
	/** Convert the textual file loaded at the {@link String} to another format, represented by a byte array, using an specific charset.
	 * @param textualFileAsString loaded file contents.
	 * @param charset charset to use. If <code>null</code> should use VM's default.
	 * @return the conversion result.
	 * @throws Exception in case of error at conversion. */
	public byte[] convert(String textualFileAsString, Charset charset) throws Exception {
		if (charset == null) {
			charset = Charset.defaultCharset();
		}
		
		return doConvert(textualFileAsString, charset);
	}
	
	/** Convert the textual file loaded at the {@link String} to another format, represented by a byte array, using an specific charset.
	 * @param textualFileAsString loaded file contents.
	 * @param charset charset to use. Never <code>null</code>.
	 * @return the conversion result.
	 * @throws Exception in case of error at conversion. */
	public abstract byte[] doConvert(String textualFileAsString, Charset charset) throws Exception;
	
	/** @return {@link String} with the converted ContentType, usually used for Servlet responses. */
	public abstract String getContentType();
	
	/** @return {@link String} with the file extension of the converted. */
   public abstract String getFileExtension();
}
