/** This file is part of nervalreports.
 *
 * nervalreports is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * nervalreports is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with nervalreports.  If not, see <http://www.gnu.org/licenses/>. */
package net.sf.nervalreports.core;

import java.io.File;
import java.io.IOException;
import java.util.EmptyStackException;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Stack;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;

import net.sf.nervalreports.core.charsets.ReportCharset;
import net.sf.nervalreports.core.paper.ReportPaperFormat;
import net.sf.nervalreports.core.paper.formats.A4;

//TODO: add a hint to use memory report creation or direct-on-file report creation. 

/** Basic class for all generator's implementations.<br>
 * <br>
 * A generator is a state machine of styles and commands, where each style affects all command calls after its definition (for those
 * familiarized with fixed-pipelined OpenGL, it should be easy to get the general idea).<br>
 * <br>
 * For example, when defined {@link #setBold} with <code>true</code> value, all texts added thereafter to the report will be rendered bold,
 * until another call to {@link #setBold} is made with <code>false</code> value.<br>
 * <br>
 * Also, each report element usually is added with sequential begin-end calls. For example, to add a table, one should call {@link #beginTable},
 * add any valid elements to it (with each specific command call) and then call {@link endTable}.<br>
 * <br>
 * Although it's somewhat verbose, it's the best way to keep the creation easily portable to any kind of report format (html, tex, csv, ods)
 * without any changes to the report code and, as much important, feasible to only iterate through the data once for most of the use cases 
 * (ideally, one should only iterate a single time through the report data. For example, when using a relational database, it should iterate
 * to generate the report exactly at the same time it iterate through the <code>ResultSet</code>).<br>
 * <br>
 * Page Footer and Header are defined with three elements: left, center and right, representing each one a corner of the page or its element
 * center.<br>
 * <br>
 * After a report is generated and no longer necessary, a call to {@link #release()} is <b>highly recommended</b>.<br>
 * <br>
 * <i>To generator creators:</i> All sanity validations to the state machine are made here with {@link EmptyStackException} and 
 * {@link IllegalStateException}.<br>
 * <br>
 * The {@link ReportGenerator} class <b>isn't thread-safe</b>, but obviously each instance of a {@link ReportGenerator} should be (and in fact
 * they are) independent, allowing the existence of multiple instances without interference between them.
 * @author farrer */
public abstract class ReportGenerator {
	
	/** Conversion factor from pixels to PDF units. */
	protected static final float PIXELS_TO_UNITS_FACTOR = 0.812f;
	
	/** Conversion factor from PDF units to pixels */
	protected final float UNITS_TO_PIXELS = 1.0f / PIXELS_TO_UNITS_FACTOR;
	
	/** Initial size to allocate for {@link #stringBuilder}. */
	private static final int BUFFER_INITIAL_SIZE = 4096;
	/** {@link StringBuilder} to use when generating a report directly to the memory. */
	private StringBuilder stringBuilder;

	/** All possible command types for begin-end. */
	private enum ReportBlockToken {
		DOCUMENT,
		DOCUMENT_HEAD,
		DOCUMENT_BODY,
		TABLE,
		TABLE_HEADER,
		TABLE_ROW,
		TABLE_ELEMENT,
		GROUP,
		PAGE_HEADER_LEFT,
		PAGE_HEADER_CENTER,
		PAGE_HEADER_RIGHT,
		PAGE_FOOTER_LEFT,
		PAGE_FOOTER_CENTER,
		PAGE_FOOTER_RIGHT
	}

	/** Background color for odd table rows. */
	private ReportColor oddRowColor = ReportColors.DEFAULT_ODD_ROW_COLOR;
	
	/** Background color for odd table rows. */
	private ReportColor evenRowColor = ReportColors.DEFAULT_EVEN_ROW_COLOR;
	
	/** Background color for all table rows (when defined, will ignore {@link #oddRowColor} and {@link #evenRowColor}. */
	private ReportColor rowColor = null;
	
	//TODO: ReportColor columnColor (if defined ignore rowColors).
	
	/** Background color for table headers. */
	private ReportColor headerBackgroundColor = ReportColors.DEFAULT_HEADER_BACKGROUND_COLOR;
	
	/** Color to use when table border is defined. */
	private ReportColor borderColor = ReportColors.DEFAULT_TEXT_COLOR;
	
	/** Width to use for table border (<code>null</code> for none). */
	private Float borderWidth = null;
		
	/** Current defined text color. */
	private ReportColor textColor = ReportColors.DEFAULT_TEXT_COLOR;
	
	/** Paper format used for the whole report document. */
	private ReportPaperFormat paperFormat = A4.getSingleton();
	
	/** ReportCharset to use. If null will be at VM's default, but support for LaTeX would be ambiguous. */
	private ReportCharset charset;
	
	/** Defined margins for report.<br>
	 * [0] - left<br>
	 * [1] - top<br>
	 * [2] - right<br>
	 * [3] - bottom<br> */
	private int[] margins = {12, 12, 12, 12};

	/** If bold text. */
	private boolean bold;

	/** If italic text. */
	private boolean italic;
	
	/** Current font size. */
	private ReportFontSize fontSize = ReportFontSize.NORMAL;
	
	/** Current font alignment. */
	private ReportTextAlignment textAlignment = ReportTextAlignment.LEFT;

	/** If document already started. */
	private boolean documentBeginned;

	/** If document ended. */
	private boolean documentEnded;

	/** If document have its head defined. */
	private boolean documentHasHead;

	/** If the report already has a body or not. */
	private boolean documentHasBody;

	/** Command stack of current opened command blocks. */
	private Stack<ReportBlockToken> documentStack = new Stack<ReportBlockToken>();;

	/** Stack with types of each currently opened group. */
	private Stack<ReportGroupType> lastGroupTypes = new Stack<ReportGroupType>();
	
	/** Current defined colorNames */
	private HashSet<String> colorNames = new HashSet<String>();
	
	/** Current defined image names */
	private HashSet<String> imageNames = new HashSet<String>();

	/** @return {@link String} to be used as extension to the files generated by the generator. */
	public abstract String getFileExtension();
	/**@return if the implemented generator supports images (if not supported, should avoid some image - graphs and plots, for example - creation).*/
	public abstract boolean supportImages();
	/** @return if the implemented generator supports both images inside jars or remote web images. */
	public abstract boolean supportImagesOfJarFilesOrURLs();
	/** @return if header and footer must be declared at document head (<code>true</code>) or body (<code>false</code>). */
	public abstract boolean isHeaderAndFooterDeclarationAtDocumentHead();

	/** The format specific implementation of {@link #beginDocument}.
    * @throws Exception in case of failure. */
	protected abstract void doBeginDocument() throws Exception;
	/** The format specific implementation of {@link #endDocument}.
    * @throws Exception in case of failure. */
	protected abstract void doEndDocument() throws Exception;

	/** The format specific implementation of {@link #beginDocumentHead}. 
    * @throws Exception in case of failure. */
	protected abstract void doBeginDocumentHead() throws Exception;
	/** Add a color to the document.
	 * @param color color to add. 
    * @throws Exception in case of failure. */
	protected abstract void doAddColor(ReportColor color) throws Exception;
	/** Implementation of the hint to use a serifed or non-serifed font.
	 * @param serifedFont <code>true</code> for serifed fonts, false for non-serifed. 
    * @throws Exception in case of failure. */
	protected abstract void doHintFont(boolean serifedFont) throws Exception;
	/** The format specific implementation of {@link #hintLanguage(String)}.  
    * @throws Exception in case of failure. */
	protected abstract void doHintLanguage(String language) throws Exception;
	/** The format specific implementation of {@link #setMargins(int, int, int, int)}.  
    * @throws Exception in case of failure. */
	protected abstract void doDefineMargins(int left, int top, int right, int bottom) throws Exception;
	/** The format specific implementation of {@link #endDocumentHead}. 
    * @throws Exception in case of failure. */
	protected abstract void doEndDocumentHead() throws Exception;

	/** The format specific implementation of {@link #beginDocumentBody}. 
    * @throws Exception in case of failure. */
	protected abstract void doBeginDocumentBody() throws Exception;
	/** The format specific implementation of {@link #endDocumentBody}. 
    * @throws Exception in case of failure. */
	protected abstract void doEndDocumentBody() throws Exception;

	/** The format specific implementation of {@link #beginTable}. 
    * @throws Exception in case of failure. */
	protected abstract void doBeginTable(int totalColumns, int[] eachColumnWidth) throws Exception;
	/** The format specific implementation of {@link #endTable}. 
    * @throws Exception in case of failure. */
	protected abstract void doEndTable() throws Exception;

	/** The format specific implementation of {@link #beginTableHeaderRow}. 
    * @throws Exception in case of failure. */
	protected abstract void doBeginTableHeaderRow() throws Exception;
	/** The format specific implementation of {@link #addTableHeaderCell}. 
    * @throws Exception in case of failure. */
	protected abstract void doAddTableHeaderCell(String columnName, int colspan) throws Exception;
	/** The format specific implementation of {@link #endTableHeaderRow}. 
    * @throws Exception in case of failure. */
	protected abstract void doEndTableHeaderRow() throws Exception;

	/** The format specific implementation of {@link #beginTableRow}. 
    * @throws Exception in case of failure. */
	protected abstract void doBeginTableRow() throws Exception;
	/** The format specific implementation of {@link #beginTableCell(int, int)}.
	 * {@link #beginTableCell()}. 
    * @throws Exception in case of failure. */
	protected abstract void doBeginTableCell(int colspan, int rowspan) throws Exception;
	/** The format specific implementation of {@link #endTableCell}. 
    * @throws Exception in case of failure. */
	protected abstract void doEndTableCell() throws Exception;
	/** The format specific implementation of {@link #endTableRow}. 
    * @throws Exception in case of failure. */
	protected abstract void doEndTableRow() throws Exception;

	/** The format specific implementation of {@link #beginGroup}. 
    * @throws Exception in case of failure. */
	protected abstract void doBeginGroup(ReportGroupType type, String name) throws Exception;
	/** The format specific implementation of {@link #endGroup}. 
    * @throws Exception in case of failure. */
	protected abstract void doEndGroup(ReportGroupType type) throws Exception;

	/** The format specific implementation of {@link #addImage}. 
    * @throws Exception in case of failure. */
	protected abstract void doAddImage(byte[] img, String imgName, int width) throws Exception;
	/** The format specific implementation of {@link #addLinkedImage}. 
    * @throws Exception in case of failure. */
	protected abstract void doAddLinkedImage(String imgLink, int width) throws Exception;
	/** The format specific implementation of {@link #addLineSpaces}. 
    * @throws Exception in case of failure. */
	protected abstract void doCreateLineSpaces(int n) throws Exception;
	/** The format specific implementation of {@link #addLineBreak}. 
    * @throws Exception in case of failure. */
	protected abstract void doLineBreak() throws Exception;
	/** The format specific implementation of {@link #addText}. 
    * @throws Exception in case of failure. */
	protected abstract void doAddText(String text) throws Exception;
	/** The format specific implementation of {@link #addSeparatorLine}. 
    * @throws Exception in case of failure. */
	protected abstract void doAddSeparatorLine() throws Exception;

	/** The format specific implementation of {@link #addCurrentPageNumber}. 
    * @throws Exception in case of failure. */
	protected abstract void doAddCurrentPageNumber() throws Exception;
	/** The format specific implementation of {@link #addTotalPagesCount}. 
    * @throws Exception in case of failure. */
	protected abstract void doAddTotalPagesCount() throws Exception;

	/** The format specific implementation of {@link #beginPageHeaderLeft}. 
    * @throws Exception in case of failure. */
	protected abstract void doBeginPageHeaderLeft() throws Exception;
	/** The format specific implementation of {@link #endPageHeaderLeft}. 
    * @throws Exception in case of failure. */
	protected abstract void doEndPageHeaderLeft() throws Exception;

	/** The format specific implementation of {@link #beginPageHeaderCenter}. 
    * @throws Exception in case of failure. */
	protected abstract void doBeginPageHeaderCenter() throws Exception;
	/** The format specific implementation of {@link #endPageHeaderCenter}. 
    * @throws Exception in case of failure. */
	protected abstract void doEndPageHeaderCenter() throws Exception;

	/** The format specific implementation of {@link #beginPageHeaderRight}. 
    * @throws Exception in case of failure. */
	protected abstract void doBeginPageHeaderRight() throws Exception;
	/** The format specific implementation of {@link #endPageHeaderRight}. 
    * @throws Exception in case of failure. */
	protected abstract void doEndPageHeaderRight() throws Exception;

	/** The format specific implementation of {@link #beginPageFooterLeft}. 
    * @throws Exception in case of failure. */
	protected abstract void doBeginPageFooterLeft() throws Exception;
	/** The format specific implementation of {@link #endPageFooterLeft}. 
    * @throws Exception in case of failure. */
	protected abstract void doEndPageFooterLeft() throws Exception;

	/** The format specific implementation of {@link #beginPageFooterCenter}. 
    * @throws Exception in case of failure. */
	protected abstract void doBeginPageFooterCenter() throws Exception;
	/** The format specific implementation of {@link #endPageFooterCenter}. 
    * @throws Exception in case of failure. */
	protected abstract void doEndPageFooterCenter() throws Exception;

	/** The format specific implementation of {@link #beginPageFooterRight}. 
    * @throws Exception in case of failure. */
	protected abstract void doBeginPageFooterRight() throws Exception;
	/** The format specific implementation of {@link #endPageFooterRight}. 
    * @throws Exception in case of failure. */
	protected abstract void doEndPageFooterRight() throws Exception;

	/** Save the content of the report to a file.
	 * @param filename path name of the file to save.
	 * @throws ReportGenerationException on file errors. */
	public void saveToFile(String filename) throws ReportGenerationException {
		if (stringBuilder == null) {
			throw new IllegalStateException("The document declaration didn't start or it was freed.");
		}
		if (documentStack.size() > 0) {
			throw new IllegalStateException("The document declaration didn't finished.");
		}
		File file = new File(filename);
		try {
			if (charset == null) {
				FileUtils.writeStringToFile(file, getStringBuilder().toString());
			} else {
				FileUtils.writeStringToFile(file, getStringBuilder().toString(), charset.getCharset());
			}
		} catch (IOException e) {
			throw new ReportGenerationException(e);
		}
	}
	
	/** Get the byte array representation of the report.
	 * @return Byte array with report.
	 * @throws ReportGenerationException on conversion errors. */
	public byte[] toByteArray() throws ReportGenerationException {
		if (stringBuilder == null) {
			throw new IllegalStateException("The document declaration didn't start or it was freed.");
		}
		if (documentStack.size() > 0) {
			throw new IllegalStateException("The document declaration didn't finished.");
		}
		return getStringBuilder().toString().getBytes();
	}
	
	/** {@inheritDoc} */
	@Override
	public String toString() {
		if (stringBuilder == null) {
			throw new IllegalStateException("The document declaration didn't start or it was freed.");
		}
		if (documentStack.size() > 0) {
			throw new IllegalStateException("The document declaration didn't finished.");
		}
		return stringBuilder.toString();
	}

	/** Clear all references of the report generation (including files created) when it's no more necessary
	 * (for example, on reports generated for web applications, after it is sent to the client). It's highly
	 * recommended to call this function as soon as possible, to avoid keeping unnecessary references to 
	 * no more used memory spaces, or bunch of temporary files. */
	public void release() {
		stringBuilder = null;
		clear();
	}

	/** Clear the generation state of the report. Must be called before a sequential generation of the report
	 * with the same instance. */
	private void clear() {
		/* Clear states */
		documentBeginned = false;
		documentEnded = false;
		documentHasHead = false;
		documentHasBody = false;
		
		/* Empty stacks */
		documentStack.clear();
		lastGroupTypes.clear();
		colorNames.clear();
		imageNames.clear();
		
		/* Reset default styles */
		bold = false;
		italic = false;
		textColor = ReportColors.DEFAULT_TEXT_COLOR;
		fontSize = ReportFontSize.NORMAL;
		textAlignment = ReportTextAlignment.LEFT;
		oddRowColor = ReportColors.DEFAULT_ODD_ROW_COLOR;
		evenRowColor = ReportColors.DEFAULT_EVEN_ROW_COLOR;
		headerBackgroundColor = ReportColors.DEFAULT_HEADER_BACKGROUND_COLOR;
		rowColor = null;
		borderColor = null;
		borderWidth = null;
		int[] defaultMargin = {12, 12, 12, 12};
		margins = defaultMargin;
		charset = null;
	}

	/** @return {@link #bold}. */
	public boolean isBold() {
		return bold;
	}

	/** Define {@link #bold}.
	 * @param bold new value to {@link #bold}. */
	public void setBold(boolean bold) {
		this.bold = bold;
	}

	/** @return {@link #italic}. */
	public boolean isItalic() {
		return italic;
	}

	/** Define {@link #italic}.
	 * @param italic new value to {@link #italic}. */
	public void setItalic(boolean italic) {
		this.italic = italic;
	}

	/** @return {@link #fontSize}. */
	public ReportFontSize getFontSize() {
		return fontSize;
	}
	
	/** Set {@link #fontSize}.
	 * @param fontSize font size to use.*/
	public void setFontSize(ReportFontSize fontSize) {
		this.fontSize = fontSize;
	}
	
	/** return {{@link #textAlignment}. */
	public ReportTextAlignment getTextAlignment() {
		return textAlignment;
	}
	
	/** Set {@link #textAlignment}. It is usually applied for the next begin* block.
	 * @param textAlignment text alignment to use. */
	public void setTextAlignment(ReportTextAlignment textAlignment) {
		this.textAlignment = textAlignment;
	}
	
	/** @return {@link #oddRowColor}. */
	public ReportColor getOddRowColor() {
		return oddRowColor;
	}
	
	/** Set {@link #oddRowColor}.
	 * @param oddRowColor new value to {@link #oddRowColor}. */
	public void setOddRowColor(ReportColor oddRowColor) {
		if (oddRowColor == null) {
			throw new IllegalArgumentException("Color should not be null. Use ReportColors.WHITE instead.");
		}
		this.oddRowColor = oddRowColor;
	}
	
	/** @return {@link #evenRowColor}. */
	public ReportColor getEvenRowColor() {
		return evenRowColor;
	}
	
	/** Set {@link #evenRowColor}.
	 * @param evenRowColor new value to {@link #evenRowColor}. */
	public void setEvenRowColor(ReportColor evenRowColor) {
		if (evenRowColor == null) {
			throw new IllegalArgumentException("Color should not be null. Use ReportColors.WHITE instead.");
		}
		this.evenRowColor = evenRowColor;
	}
	
	/** @return {@link #rowColor}. Could be <code>null</code>. */
	public ReportColor getRowColor() {
		return rowColor;
	}
	/** Set {@link #rowColor}. 
	 * @param rowColor color to use for all rows. If <code>null</code> will use {@link #evenRowColor} and {@link #oddRowColor} instead. */
	public void setRowColor(ReportColor rowColor) {
		this.rowColor = rowColor;
	}
	/** @return {@link #headerBackgroundColor}. */
	public ReportColor getHeaderBackgroundColor() {
		return headerBackgroundColor;
	}
	
	/** Set {@link #headerBackgroundColor}.
	 * @param headerBackgroundColor new value to {@link #headerBackgroundColor}. */
	public void setHeaderBackgroundColor(ReportColor headerBackgroundColor) {
		if (headerBackgroundColor == null) {
			throw new IllegalArgumentException("Color should not be null. Use ReportColors.WHITE instead.");
		}
		this.headerBackgroundColor = headerBackgroundColor;
	}
	
	/** @return {@link #borderColor}. */
	public ReportColor getBorderColor() {
		return borderColor;
	}
	
	/** @return {@link #borderWidth}. */
	public Float getBorderWidth() {
		return borderWidth;
	}
	
	/** @return {@link #textColor}. */
	public ReportColor getTextColor() {
		return textColor;
	}

	/** Define {@link #textColor}.
	 * @param color new value to {@link #textColor}. */
	public void setTextColor(ReportColor color) {
		if (color == null) {
			throw new IllegalArgumentException("Color should not be null. Use ReportColors.WHITE instead.");
		}
		this.textColor = color;
	}

	/** @return {@link #getStringBuilder()}. */
	protected StringBuilder getStringBuilder() {
		return stringBuilder;
	}

	/** Init the document. Must be called only once per report generation. 
    * @throws ReportGenerationException in case of failure. */
	public void beginDocument() throws ReportGenerationException {
		if (documentBeginned) {
			throw new IllegalStateException("You can't begin a document for more than once.");
		}
		/* Create the new buffer. */
		stringBuilder = new StringBuilder(BUFFER_INITIAL_SIZE);
		/* Do the document begin specific code. */
		try {
			doBeginDocument();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
		/* Define it started, and to the stack!. */
		documentBeginned = true;
		documentStack.push(ReportBlockToken.DOCUMENT);
		/* Add all current defined colors */
		colorNames.clear();
		for (ReportColor color : ReportColors.DEFINED_COLORS) {
			colorNames.add(color.getName());
		}
		/* Clear current defined image names */
		imageNames.clear();
	}

	/** End the document. 
    * @throws ReportGenerationException in case of failure. */
	public void endDocument() throws ReportGenerationException {
		checkStackState(true, ReportBlockToken.DOCUMENT);
		try {
			doEndDocument();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
		documentEnded = true;
	}

	/** Begin the document's header, where should lies all specific implementation declarations. 
    * @throws ReportGenerationException in case of failure. */
	public void beginDocumentHead() throws ReportGenerationException {
		checkDocumentNotFinished();
		if (documentHasHead) {
			throw new IllegalStateException("It's only possible a head per document.");
		}
		try {
			doBeginDocumentHead();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
		documentHasHead = true;
		documentStack.push(ReportBlockToken.DOCUMENT_HEAD);
	}
	
	/** Declare a color in RGB mode.<br>
	 * <br>
	 * <b>Note:</b> Should be called inner document's head.
	 * @param red red component of the color [0, 255].
	 * @param green green component of the color [0, 255].
	 * @param blue blue component of the color [0, 255].
	 * @param name color name (must be unique per report and without spaces).
	 * @return {@link ReportColor} declared and created. 
    * @throws ReportGenerationException in case of failure. */
	public ReportColor addColor(int red, int green, int blue, String name) throws ReportGenerationException {
		checkStackState(false, ReportBlockToken.DOCUMENT_HEAD);
		if (StringUtils.isEmpty(name)) {
			throw new IllegalArgumentException("The color name should not be empty.");
		}
		if ((StringUtils.isBlank(name)) || (name.contains(" "))) {
			throw new IllegalArgumentException("The color name is mandatory and shouldn't contain spaces.");
		}
		if (colorNames.contains(name)) {
			throw new IllegalArgumentException("The color name should be unique per report.");
		}
		
		ReportColor color = new ReportColor(red, green, blue, name);
		try {
			doAddColor(color);
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
		colorNames.add(name);
		return color;
	}
	
	/** Set the paper format to use for the whole document (default is {@link A4}).<br>
	 * <br>
	 * <b>Note:</b> Should be called inner document's head.
	 * @param paperFormat to use. */
	public void setPaper(ReportPaperFormat paperFormat) {
		if (paperFormat == null) {
			throw new IllegalArgumentException("Paper should not be null.");
		}
		checkStackState(false, ReportBlockToken.DOCUMENT_HEAD);
		this.paperFormat = paperFormat;
	}
	
	/** @return {@link #paperFormat}. */
	public ReportPaperFormat getPaperFormat() {
		return this.paperFormat;
	}
	
	/** Define the document page margins.<br>
	 * <br>
	 * <b>Note:</b> Must be called immediately after {@link #beginDocument()} and before {@link #beginDocumentHead()} / {@link #beginDocumentBody()}. 
	 * @param left left margin in PDF units.
	 * @param top top margin in PDF units.
	 * @param right right margin in PDF units.
	 * @param bottom bottom margin in PDF units. */
	public void setMargins(int left, int top, int right, int bottom) throws ReportGenerationException {
		checkStackState(false, ReportBlockToken.DOCUMENT);
		if ((documentHasHead) || (documentHasBody)) {
			throw new IllegalArgumentException("Margin definition must be set after document begin, but before document head or body.");
		}
		try {
			margins[0] = left;
			margins[1] = top;
			margins[2] = right;
			margins[3] = bottom;
			doDefineMargins(left, top, right, bottom);
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}
	
	/** @return {@link #margins}. */
	public int[] getMargins() {
		return margins;
	}
	
	/** Define the language to use.
	 * @param language String with the language as defined by LaTeX babel package {<b>see:</b> http://www.ctan.org/pkg/babel}. 
    * @throws ReportGenerationException in case of failure. */
	public void hintLanguage(String language) throws ReportGenerationException {
		checkStackState(false, ReportBlockToken.DOCUMENT_HEAD);
		try {
			doHintLanguage(language);
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}
	
	/** Define the character set to use. If not defined will use VM's default, but LaTeX support could be incomplete.
	 * @param charset to define. 
    * @throws ReportGenerationException in case of failure. */
	public void setReportCharset(ReportCharset charset) throws ReportGenerationException {
		checkStackState(false, ReportBlockToken.DOCUMENT_HEAD);
		this.charset = charset;
	}
	
	/** @return {#charset}. If <code>null</code> should be at VM's default.  */
	public ReportCharset getReportCharset() {
		return charset;
	}
	
	/** Hint to use a serifed font. 
    * @throws ReportGenerationException in case of failure. */
	public void hintSerifFont() throws ReportGenerationException {
		checkStackState(false, ReportBlockToken.DOCUMENT_HEAD);
		try {
			doHintFont(true);
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}
	
	/** Hint to use a non-serifed font. 
    * @throws ReportGenerationException in case of failure. */
	public void hintNonSerifFont() throws ReportGenerationException {
		checkStackState(false, ReportBlockToken.DOCUMENT_HEAD);
		try {
			doHintFont(false);
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** End the header of the document. 
    * @throws ReportGenerationException in case of failure. */
	public void endDocumentHead() throws ReportGenerationException {
		checkStackState(true, ReportBlockToken.DOCUMENT_HEAD);
		try {
			doEndDocumentHead();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** Begin the report document body, where usually are all report data.<br>
	 * <b>Each report must have a maximum of one body</b>. 
    * @throws ReportGenerationException in case of failure. */
	public void beginDocumentBody() throws ReportGenerationException {
		checkDocumentNotFinished();
		if (documentHasBody) {
			throw new IllegalStateException("A report can't have multiple bodies.");
		}
		try {
			doBeginDocumentBody();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
		documentHasBody = true;
		documentStack.push(ReportBlockToken.DOCUMENT_BODY);
	}

	/** End the currently opened report body. 
    * @throws ReportGenerationException in case of failure. */
	public void endDocumentBody() throws ReportGenerationException {
		checkStackState(true, ReportBlockToken.DOCUMENT_BODY);
		try {
			doEndDocumentBody();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}
	
	/** Set table border style that will take effect for next {@link #beginTable} call.
	 * @param borderWidth width to use for table border (<code>null</code> for no border).
	 * @param borderColor color to use for table border (if its width is not <code>null</code>). */
	public void setTableBorderStyle(Float borderWidth, ReportColor borderColor) {
		checkDocumentNotFinished();
		this.borderWidth = borderWidth;
		this.borderColor = borderColor;
	}

	/** Begin a table, defining or not the width of all its columns. To not define any column width, simple call the method only
	 * with the totalColumns defined.<br>
	 * <b>Note:</b> If not defined, most ReportGenerators (but not all) will make the table with full page width.
	 * @param totalColumns total number of columns the table has.
	 * @param eachColumnWidth width, in PDF units, for each table column. If defined, the total number of eachColumnWidth
	 *                        <b>must be equal</b> to the number of columns. 
    * @throws ReportGenerationException in case of failure. */
	public void beginTable(int totalColumns, int ... eachColumnWidth) throws ReportGenerationException {
		
		/* Sanity check. */
		if ((eachColumnWidth != null) && (eachColumnWidth.length != 0) && (totalColumns != eachColumnWidth.length)) {
			throw new IllegalArgumentException("One must define all column widths or none at all.");
		}

		checkDocumentNotFinished();
		try {
			if ((eachColumnWidth != null) && (eachColumnWidth.length > 0)) {
				doBeginTable(totalColumns, eachColumnWidth);
			} else {
				doBeginTable(totalColumns, null);
			}
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
		documentStack.push(ReportBlockToken.TABLE);
	}

	/** Begin a table definition, setting all column's width to the same fixed value.
	 * @see ReportGenerator#beginTable(int, int...)
	 * @param totalColumns total number of columns the table has.
	 * @param fixedWidth width, in PDF units, for each table columns. 
    * @throws ReportGenerationException in case of failure. */
	public void beginTableWithFixedDefinedWidth(int totalColumns, int fixedWidth) throws ReportGenerationException {
		/* Convert to the 'index, width ...' call */
		int[] colWidths = new int[totalColumns];
		for (int i = 0; i < totalColumns; i++) {
			colWidths[i] = fixedWidth;
		}
		beginTable(totalColumns, colWidths);
	}

	/** End the current table. */
	public void endTable() throws ReportGenerationException {
		checkStackState(true, ReportBlockToken.TABLE);
		try {
			doEndTable();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** Create a table header row with a predefined list of table column's titles.
	 * @param header list with each column title. */
	public void addTableHeaderRow(List<String> header) throws ReportGenerationException {
		if ((header == null) || (header.size() <= 0)) {
			throw new IllegalArgumentException("The list with column titles should not be empty.");
		}
		beginTableHeaderRow();
		for (String column : header) {
			addTableHeaderCell(column);
		}
		endTableHeaderRow();
	}

	/** Begin the definition of a table header row.<br>
	 * <br>
	 * <b>Note:</b> Must be inner a table. */
	public void beginTableHeaderRow() throws ReportGenerationException {
		checkStackState(false, ReportBlockToken.TABLE, ReportBlockToken.TABLE_HEADER);
		try {
			doBeginTableHeaderRow();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
		documentStack.push(ReportBlockToken.TABLE_HEADER);
	}

	/** Add a table header title text for a single table column.<br>
	 * <br>
	 * <b>Note:</b> Must be inner a table. 
	 * @param columnName column title. */
	public void addTableHeaderCell(String columnName) throws ReportGenerationException {
		addTableHeaderCell(columnName, 1);
	}

	/** Add a table header title text.<br>
	 * <br>
	 * <b>Note:</b> Must be inner a table. 
	 * @param columnName title of the column.
	 * @param colspan number of columns the title should occupy. Equals to <code>1</code> to represent a single column title. */
	public void addTableHeaderCell(String columnName, int colspan) throws ReportGenerationException {
		if (StringUtils.isEmpty(columnName)) {
			throw new IllegalArgumentException("Column's name shouldn't empty.");
		}
		if (colspan <= 0) {
			throw new IllegalArgumentException("The cell can't occupy a space lesser then 1.");
		}
		checkStackState(false, ReportBlockToken.TABLE_HEADER);
		try {
			doAddTableHeaderCell(columnName, colspan);
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** End the declaration of current table header row. */
	public void endTableHeaderRow() throws ReportGenerationException {
		checkStackState(true, ReportBlockToken.TABLE_HEADER);
		try {
			doEndTableHeaderRow();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** Generate a table line<br>
	 * <br>
	 * <b>Note:</b> only use this function if <i>a priori</i> (ie: at the moment of ResultSet iteration) it isn't possible
	 * to add the columns. This method reiterates through the elements, which is really awful when using large volumes of data.
	 * When possible, use a sequence of {@link #beginTableRow()}, {@link #addTableCell(String)}'s {@link #endTableRow()} instead
	 * of this.<br>
	 * <br>
	 * <b>Note:</b> Must be inner a table.
	 * @param cells list with all cells of the line (according to their columns). */
	public void addTableRow(List<String> cells) throws ReportGenerationException {
		if ((cells == null) || (cells.size() <= 0)) {
			throw new IllegalArgumentException("The row cells must be defined.");
		}
		beginTableRow();
		for (String column : cells) {
			addTableCell(column, 1, 1);
		}
		endTableRow();
	}

	/** Init a table row.<br>
	 * <br>
	 * <b>Note:</b> Must be inner a table. */
	public void beginTableRow() throws ReportGenerationException {
		checkStackState(false, ReportBlockToken.TABLE);
		try {
			doBeginTableRow();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
		documentStack.push(ReportBlockToken.TABLE_ROW);
	}

	/** Begin the creation of a table element (cell), of a single column in a single row.<br>
	 * <b>Note:</b> Must be inner a table row. */
	public void beginTableCell() throws ReportGenerationException {
		beginTableCell(1, 1);
	}

	/** Begin the creation of a table element (cell).<br>
	 * <br>
	 * <b>Note:</b> Must be inner a table row.
	 * @param colspan number of columns occupied by the cell.
	 * @param rowspan number of rows occupied by the cell. */
	public void beginTableCell(int colspan, int rowspan) throws ReportGenerationException {
		checkStackState(false, ReportBlockToken.TABLE_ROW);
		if ((colspan <= 0) || (rowspan <= 0)) {
			throw new IllegalArgumentException("The cell can't occupy a space lesser then 1.");
		}
		try {
			doBeginTableCell(colspan, rowspan);
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
		documentStack.push(ReportBlockToken.TABLE_ELEMENT);
	}

	/** Same of {@link #addTableCell(String)}, but defining its spans.<br>
	 * <br>
	 * <b>Note:</b> Must be inner a table row.
	 * @param element cell's text.
	 * @param colspan number of columns occupied by the cell.
	 * @param rowspan number of rows occupied by the cell. */
	public void addTableCell(String element, int colspan, int rowspan) throws ReportGenerationException {
		beginTableCell(colspan, rowspan);
		addText(element);
		endTableCell();
	}

	/** Adds a textual table cell element to the page. This function is a quick alternative for
	 * {@link #beginTableCell} - {@link #addText} - {@link #endTableCell} sequence, of adding
	 * single text to a table cell.<br>
	 * <br>
	 * <b>Note:</b> Must be inner a table row.
	 * @param element cell's text. */
	public void addTableCell(String element) throws ReportGenerationException {
		addTableCell(element, 1, 1);
	}

	/** Ends the definition of a table cell element. */
	public void endTableCell() throws ReportGenerationException {
		checkStackState(true, ReportBlockToken.TABLE_ELEMENT);
		try {
			doEndTableCell();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** Ends the definition of a table row line. */
	public void endTableRow() throws ReportGenerationException {
		checkStackState(true, ReportBlockToken.TABLE_ROW);
		try {
			doEndTableRow();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** Begin a group of report elements. 
	 * @param type type of the group to begin. */
	public void beginGroup(ReportGroupType type) throws ReportGenerationException {
		beginGroup(type, null);
	}

	/** Begin a named group of report elements.
	 * @param type type of the group to begin.
	 * @param name optional name of the group (could be <code>null</code>). */
	public void beginGroup(ReportGroupType type, String name) throws ReportGenerationException {
		checkStackState(false, ReportBlockToken.DOCUMENT_BODY, ReportBlockToken.GROUP,
				ReportBlockToken.TABLE_ELEMENT);
		if (type == null) {
			throw new IllegalArgumentException("The report group must be defined.");
		}
		lastGroupTypes.push(type);
		documentStack.push(ReportBlockToken.GROUP);
		try {
			doBeginGroup(type, name);
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** End the current opened group of report elements. */
	public void endGroup() throws ReportGenerationException {
		checkStackState(true, ReportBlockToken.GROUP);
		ReportGroupType type = lastGroupTypes.pop();
		try {
			doEndGroup(type);
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** Add the current page number to the report. */
	public void addCurrentPageNumber() throws ReportGenerationException {
		checkDocumentNotFinished();
		try {
			doAddCurrentPageNumber();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}
	
	/** @return if is currently inside a header or a footer. */
	protected boolean isInsideHeaderOrFooter() {
		for (ReportBlockToken token : documentStack) {
			if ((ReportBlockToken.PAGE_FOOTER_LEFT.equals(token)) ||
				(ReportBlockToken.PAGE_FOOTER_CENTER.equals(token)) ||
				(ReportBlockToken.PAGE_FOOTER_RIGHT.equals(token)) ||
				(ReportBlockToken.PAGE_HEADER_LEFT.equals(token)) ||
				(ReportBlockToken.PAGE_HEADER_CENTER.equals(token)) ||
				(ReportBlockToken.PAGE_HEADER_RIGHT.equals(token)))
			{
				return true;
			}
		}
		return false;
	}

	/** Add the number of total report pages to the report.<br><br>
	 * <b>Note: </b> Should only be used inside page's headers or footers.<br><br>
	 * <b>Note: </b> Beware that using this could make the report generation slower for some generators, because this value will only be
	 * available after the whole document is defined (making the need to go back through it to replace places where totalPages 
	 * references are made). */
	public void addTotalPagesCount() throws ReportGenerationException {
		checkDocumentNotFinished();
		if (!isInsideHeaderOrFooter()) {
			throw new IllegalStateException("Total pages count is only valid inside header or footer.");
		}
		try {
			doAddTotalPagesCount();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** Begin the right page header definition. */
	public void beginPageHeaderLeft() throws ReportGenerationException {
		checkDocumentNotFinished();
		documentStack.push(ReportBlockToken.PAGE_HEADER_LEFT);
		try {
			doBeginPageHeaderLeft();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** End the center page header definition. */
	public void endPageHeaderLeft() throws ReportGenerationException {
		checkStackState(true, ReportBlockToken.PAGE_HEADER_LEFT);
		try {
			doEndPageHeaderLeft();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** Begin the center page header definition. */
	public void beginPageHeaderCenter() throws ReportGenerationException {
		checkDocumentNotFinished();
		documentStack.push(ReportBlockToken.PAGE_HEADER_CENTER);
		try {
			doBeginPageHeaderCenter();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** End the center page header definition. */
	public void endPageHeaderCenter() throws ReportGenerationException {
		checkStackState(true, ReportBlockToken.PAGE_HEADER_CENTER);
		try {
			doEndPageHeaderCenter();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** Begin the right page header definition. */
	public void beginPageHeaderRight() throws ReportGenerationException {
		checkDocumentNotFinished();
		documentStack.push(ReportBlockToken.PAGE_HEADER_RIGHT);
		try {
			doBeginPageHeaderRight();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** End the right page header definition. */
	public void endPageHeaderRight() throws ReportGenerationException {
		checkStackState(true, ReportBlockToken.PAGE_HEADER_RIGHT);
		try {
			doEndPageHeaderRight();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** Init the left page footer definition. */
	public void beginPageFooterLeft() throws ReportGenerationException {
		checkDocumentNotFinished();
		documentStack.push(ReportBlockToken.PAGE_FOOTER_LEFT);
		try {
			doBeginPageFooterLeft();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** End the left page footer definition. */
	public void endPageFooterLeft() throws ReportGenerationException {
		checkStackState(true, ReportBlockToken.PAGE_FOOTER_LEFT);
		try {
			doEndPageFooterLeft();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** Init the center page footer definition. */
	public void beginPageFooterCenter() throws ReportGenerationException {
		checkDocumentNotFinished();
		documentStack.push(ReportBlockToken.PAGE_FOOTER_CENTER);
		try {
			doBeginPageFooterCenter();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** End the center page footer definition. */
	public void endPageFooterCenter() throws ReportGenerationException {
		checkStackState(true, ReportBlockToken.PAGE_FOOTER_CENTER);
		try {
			doEndPageFooterCenter();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** Init the right page footer definition. */
	public void beginPageFooterRight() throws ReportGenerationException {
		checkDocumentNotFinished();
		documentStack.push(ReportBlockToken.PAGE_FOOTER_RIGHT);
		try {
			doBeginPageFooterRight();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** End the right page footer definition. */
	public void endPageFooterRight() throws ReportGenerationException {
		checkStackState(true, ReportBlockToken.PAGE_FOOTER_RIGHT);
		try {
			doEndPageFooterRight();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** Add an image defined by a byte array of its format (usually it's safe to consider that all ReportGenerator's
	 * who support images will support PNG and JPG images formats here).<br>
	 * <br>
	 * <b>Note:</b> When using TeXReportGenerator, you must avoid using this function inside headers or footers.
	 * @param img image represented by a byte array.
	 * @param imgName image name: must be unique per document and without spaces.
	 * @param width width (in pixels) to display the image.  */
	public void addImage(byte[] img, String imgName, int width) throws ReportGenerationException {
		checkDocumentNotFinished();

		if ((img == null) || (img.length == 0)) {
			throw new IllegalArgumentException("The image should not be empty.");
		}
		if ((StringUtils.isEmpty(imgName)) || (imgName.contains(" "))) {
			throw new IllegalArgumentException("The image name is mandatory and shouldn't contain spaces.");
		}
		if (imageNames.contains(imgName)) {
			throw new IllegalArgumentException("The image name should be unique per report.");
		}
		
		try {
			doAddImage(img, imgName, width);
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
		imageNames.add(imgName);
	}
	
	/** @return verify if is a inner .jar file link or not. */
	private boolean isInnerJarFile(String link) {
		return link.startsWith("jar:");
	}

	/** Add an image to the document by its link. If the link is for a jar or URL and {@link #supportImagesOfJarFilesOrURLs}
	 * is <code>false</code>, it's recommended to insert it - if not at header or footer - via 
	 * {@link #addImage(byte[], String, int)}, or to extract it to a file and insert via the path as link.
	 * @param imgLink path to the image. 
	 * @param width width (in pixels) to display the image. */
	public void addLinkedImage(String imgLink, int width) throws ReportGenerationException {
		checkDocumentNotFinished();
		if (StringUtils.isEmpty(imgLink)) {
			throw new IllegalArgumentException("Image link should not be empty.");
		}
		
		if ((isInnerJarFile(imgLink)) && (!supportImagesOfJarFilesOrURLs())) {
			throw new IllegalArgumentException(String.format("This generator doesn't support images inner a jar. %s",
					"You should externally save it and use the link, or add them with the data already read with addImage function."));
		}
		
		try {
			doAddLinkedImage(imgLink, width);
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** Add <code>n</code> blank lines to the report.
	 * @param n Number of blank lines to add. */
	public void addLineSpaces(int n) throws ReportGenerationException {
		checkDocumentNotFinished();
		if (n <= 0) {
			throw new IllegalArgumentException("The number of blank lines must be greater than 0.");
		}
		
		try {
			doCreateLineSpaces(n);
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** Add a line break to the document. Not all documents support line breaks inner tables, so use with parsimony.
	 * <br>
	 * <b>Note:</b> Current font size will affect the height (and so line space) of the line break. */
	public void addLineBreak() throws ReportGenerationException {
		checkDocumentNotFinished();
		
		try {
			doLineBreak();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** Add (or better, append) a contiguous text to the report, replacing <code>null</code> and empty values with '-'.
	 * @param text {@link String} with the text to append or <code>null</code>. */
	public void addTextWithNullSupport(String text) throws ReportGenerationException {
		checkDocumentNotFinished();
		if (StringUtils.isBlank(text)) {
			addText("-");
		} else {
			addText(text);
		}
	}

	/** Add (or better, append) a contiguous text to the report.<br>
	 * <br>
	 * <b>Note:</b> The text should not be empty. For potential empty texts, 
	 * use {@link #addTextWithNullSupport(String)}.
	 * @param text {@link String} with the text to append. */
	public void addText(String text) throws ReportGenerationException {
		checkDocumentNotFinished();
		if (StringUtils.isEmpty(text)) {
			throw new IllegalArgumentException("Can't add an empty text.");
		}
		
		try {
			doAddText(text);
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** Add text to the report with a line break at end.
	 * @param line {@link String} with text to add. */
	public void addTextLine(String line) throws ReportGenerationException {
		addText(line);
		addLineSpaces(1);
	}

	/** Add an horizontal line (-----) separator to the report. */
	public void addSeparatorLine() throws ReportGenerationException {
		checkDocumentNotFinished();
		
		try {
			doAddSeparatorLine();
		} catch (Exception e) {
			throw new ReportGenerationException(e);
		}
	}

	/** Check if document is inited and not yet finished. */
	protected void checkDocumentNotFinished() {
		if ((!documentBeginned) || (documentEnded)) {
			throw new IllegalStateException("The document wasn't inited or is already finished.");
		}
	}
	
	/** Check if no footer or header block is currently opened. 
	 * @throws IllegalStateException if the top isn't one of those expected or <code>null</code>. */
	protected void checkNotOpenedFooterOrHeader() {
		if (documentStack.size() == 0) {
			/* No elements, no footer or header. */
			return;
		}
		
		Iterator<ReportBlockToken> iterator = documentStack.iterator();
		while (iterator.hasNext()) {
			ReportBlockToken token = iterator.next();
			if ((ReportBlockToken.PAGE_FOOTER_CENTER.equals(token)) || (ReportBlockToken.PAGE_FOOTER_LEFT.equals(token)) ||
				(ReportBlockToken.PAGE_FOOTER_RIGHT.equals(token)) || ((ReportBlockToken.PAGE_HEADER_CENTER.equals(token))) ||
				(ReportBlockToken.PAGE_HEADER_LEFT.equals(token)) || (ReportBlockToken.PAGE_FOOTER_RIGHT.equals(token))) {
				throw new IllegalStateException(String.format("Not valid inner a '%s'.", token.toString()));
			}
		}
	}

	/** Verify the top of currently opened commands, expecting at last one type.
	 * @param pop if will pop (<code>true</code>) or peek (<code>false</code>) the stack top to check.
	 * @param expected valid expected types.
	 * @throws IllegalStateException if the top isn't one of those expected or <code>null</code>. 
	 * @throws EmptyStackException if the command stack is empty. */
	protected void checkStackState(boolean pop, ReportBlockToken ... expected) {
		ReportBlockToken current;

		/* Retrieve stack top */
		if (pop) {
			current = documentStack.pop();
		} else {
			current = documentStack.peek();
		}

		/* Check if not null */
		if (current == null) {
			throw new IllegalStateException("Stack's top is null.");
		}

		/* Check if is one of those expected */
		boolean found = false;
		for (ReportBlockToken possibleExpected : expected) {
			if (current.equals(possibleExpected)) {
				found = true;
				break;
			}
		}

		if (!found) {
			throw new IllegalStateException(String.format("Stack's top '%s' isn't expected.", current.toString()));
		}
	}
}
