/*
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at http://www.apache.org/licenses/LICENSE-2.0 Unless required by
 * applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS
 * OF ANY KIND, either express or implied. See the License for the specific
 * language governing permissions and limitations under the License.
 */

package net.sf.jkniv.sqlegance.parser;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Execute parsing at parameters values from XML files, extrating your values.
 * 
 * @author alisson gomes
 * @since 0.0.2
 */
public final class ParameterParser
{
    // find the pattern #{id}
    private static final String REGEX_HASH_SYMBOL     = "#\\{[\\w]+\\}";
    // find the pattern :id
    private static final String REGEX_TWODOTS_SYMBOL  = ":[\\w]+";
    // find the pattern ?
    private static final String REGEX_QUESTION_SYMBOL = "\\?";
    
    public static final Pattern patternHash           = Pattern.compile(REGEX_HASH_SYMBOL, Pattern.CASE_INSENSITIVE);
    public static final Pattern patternTowDots        = Pattern.compile(REGEX_TWODOTS_SYMBOL, Pattern.CASE_INSENSITIVE);
    public static final Pattern patternQuestion       = Pattern.compile(REGEX_QUESTION_SYMBOL, Pattern.CASE_INSENSITIVE);
    
    private ParameterParser()
    {
        // hidden constructor form utility classes
    }
    
    /**
     * This is a null-safe method that extract the parameters from a sentence.
     * 
     * @param sentence
     *            The SQL sentence (is null-safe).
     * @return Return an array of parameter names find at sentence. The array is
     *         zero length if cannot find parameters or sentence is null.
     */
    public static String[] extract(String sentence)
    {
        if (sentence == null)
            return new String[0];
        List<String> params = new ArrayList<String>();
        Matcher matcherHash = patternHash.matcher(sentence);
        Matcher matcherTwoDots = patternTowDots.matcher(sentence);
        Matcher matcherQuestion = patternQuestion.matcher(sentence);
        boolean findPlaceHoldMarker = false; // don't have question is true
        int i = 0;
        while (matcherHash.find())
        {
            params.add(i, sentence.subSequence(matcherHash.start() + 2, matcherHash.end() - 1).toString());
            i++;
            findPlaceHoldMarker = true;
        }
        while (matcherTwoDots.find())
        {
            if (findPlaceHoldMarker && !params.isEmpty())
            {
                throw new RuntimeException(
                        "Cannot mix parameters placehold '#{param}', ':param' and '?' symbol, use just one don't mix. The symbol at position: "
                                + matcherQuestion.start() + " from [" + sentence + "] is invalid!");
            }
            params.add(i, sentence.subSequence(matcherTwoDots.start() + 1, matcherTwoDots.end()).toString());
            i++;
        }
        findPlaceHoldMarker = !params.isEmpty();
        while (matcherQuestion.find())
        {
            if (findPlaceHoldMarker && !params.isEmpty())
            {
                throw new RuntimeException(
                        "Cannot mix parameters placehold '#{param}', ':param' and '?' symbol, use just one don't mix. The symbol at position: "
                                + matcherQuestion.start() + " from [" + sentence + "] is invalid!");
            }
            params.add(i, sentence.subSequence(matcherQuestion.start(), matcherQuestion.end()).toString());
            i++;
        }
        return params.toArray(new String[0]);
    }
}
