/*
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at http://www.apache.org/licenses/LICENSE-2.0 Unless required by
 * applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS
 * OF ANY KIND, either express or implied. See the License for the specific
 * language governing permissions and limitations under the License.
 */
package net.sf.jkniv.sqlegance.builder.xml.dynamic;

import java.util.List;

/**
 * Represent the <code>set</code> sentence from <code>update</code> SQL.
 * 
 * @author alisson gomes
 * @since 0.0.2
 */
public class SetTag implements ITextTag
{
    public static final String TAG_NAME = "set";
    private List<? extends ITextTag>     listIfTag;
    private StringBuilder      text;
    
    /**
     * Build a new <code>set</code> tag.
     * 
     * @param listIfTag
     *            list of texts from set tag, dynamic or static texts.
     */
    public SetTag(List<? extends ITextTag> listIfTag)
    {
        text = new StringBuilder();
        this.listIfTag = listIfTag;
    }
    
    /**
     * Evaluate if attribute test is true. The dynamic text is building while
     * evaluate attribute test, for each true sentence the text is appended.
     * 
     * @return true if some test case its true, false otherwise.
     */
    public boolean eval(Object rootObjects)
    {
        boolean evaluation = false;
        text.delete(0, text.length());
        for (int i = 0; i < listIfTag.size(); i++)
        {
            ITextTag tag = listIfTag.get(i);
            if (tag.eval(rootObjects))
            {
                evaluation = true;
                
                String clause = tag.getText();
                if (clause.endsWith(","))
                    clause = clause.substring(0, clause.length() - 1);
                
                if (text.indexOf("set") >= 0)
                {
                    text.append(", " + clause);
                }
                else
                {
                    text.append("set ");
                    text.append(clause);
                }
            }
        }
        return evaluation;
    }
    
    /**
     * Retrieve the dynamic text from XML element.
     * 
     * @return text from XML element.
     */
    public String getText()
    {
        return text.toString();
    }
    
    /**
     * Indicate if text is dynamic or static.
     * 
     * @return always true is returned, because this object save dynamic text.
     */
    public boolean isDynamic()
    {
        return true;
    }
}
