/*
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at http://www.apache.org/licenses/LICENSE-2.0 Unless required by
 * applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS
 * OF ANY KIND, either express or implied. See the License for the specific
 * language governing permissions and limitations under the License.
 */
package net.sf.jkniv.sqlegance.builder.xml.dynamic;

import ognl.Ognl;
import ognl.OgnlException;

/**
 * Represent the <code>if</code> tag from XML file to put dynamic SQLs.
 * 
 * @author alisson gomes
 * @since 0.0.2
 */
public abstract class ConditionalTag implements ITextTag
{
    public static final String ATTRIBUTE_TEST = "test";
    private Object             ognlExpression;
    private String             text;
    
    /**
     * Parses the given OGNL expression that can be used by Ognl static methods.
     * 
     * @param ognlExpression
     *            the OGNL expression to be parsed
     * @exception MalformedExpression
     *                if the expression is malformed or if there is a
     *                pathological environmental problem.
     */
    public ConditionalTag(String ognlExpression, String text)
    {
        this.text = text;
        if (text == null || "".equals(text))
            throw new MalformedExpression("Error parsing text from tag <if test=\"...\">text</if>, cannot be null or empty.");
        
        try
        {
            this.ognlExpression = Ognl.parseExpression(ognlExpression);
        }
        catch (OgnlException e)
        {
            throw new MalformedExpression("Error parsing expression '" + ognlExpression + "'. Cause: " + e.getMessage(), e);
        }
    }
    
    /**
     * Evaluate the expression from test attribute is true or false.
     * 
     * @return true if expression is true, false otherwise.
     */
    public boolean eval(Object rootObjects)
    {
        return ExpressionEvaluator.eval(ognlExpression, rootObjects);
    }
    
    /**
     * Retrieve the text from XML element.
     * 
     * @return text from XML element.
     */
    public String getText()
    {
        return this.text;
    }
    
    /**
     * Indicate if text is dynamic or static.
     * 
     * @return always true is returned, because this object save dynamic text.
     */
    public boolean isDynamic()
    {
        return true;
    }
    
    @Override
    public String toString() {
    	return "text=" + this.text;
    }
}
