/*
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at http://www.apache.org/licenses/LICENSE-2.0 Unless required by
 * applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS
 * OF ANY KIND, either express or implied. See the License for the specific
 * language governing permissions and limitations under the License.
 */
package net.sf.jkniv.sqlegance.builder.xml;

import java.util.ArrayList;
import java.util.List;

import net.sf.jkniv.sqlegance.builder.xml.dynamic.ITextTag;
import net.sf.jkniv.sqlegance.builder.xml.dynamic.StaticText;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Generic tag to support the common functions from other tags.
 * 
 * @author alisson gomes
 * @since 0.0.2
 */
public abstract class AbstractSqlTag implements ISqlTag
{
    public final Logger        log            = LoggerFactory.getLogger(getClass());
    public static final String ATTRIBUTE_NAME = "id";
    public static final String ATTRIBUTE_TYPE = "type";
    protected String           id;
    protected LanguageType     languageType;
    private List<ITextTag>     textTag;
    
    /**
     * Build a new SQL tag from XML file.
     * 
     * @param id
     *            Name/Identify from tag.
     * @param languageType
     *            type of language from tag.
     */
    public AbstractSqlTag(String id, LanguageType languageType)
    {
        this.textTag = new ArrayList<ITextTag>();
        this.id = id;
        this.languageType = languageType;
    }
    
    /**
     * Retrieve static sql from a node, the dynamic parts is skipped.
     * 
     * @return Sql sentence
     */
    public String getSql()
    {
        StringBuilder sb = new StringBuilder();
        for (ITextTag text : textTag)
        {
            if (!text.isDynamic())
            {
                if (sb.length() > 0)
                    sb.append(" " + text.getText());
                else
                    sb.append(text.getText());
            }
        }
        return sb.toString();
    }
    
    /**
     * Retrieve dynamic sql from a node, the dynamic parts is evaluated at time
     * to create the sql.
     * 
     * @param params
     *            parameters to evaluate dynamic sql, can be a object like
     *            Author, Book, etc or a java.util.Map where yours keys is like
     *            the properties.
     * @return Dynamic sql sentence
     */
    public String getSql(Object params)
    {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < textTag.size(); i++)
        {
            ITextTag tag = textTag.get(i);
            boolean result = tag.eval(params);
            log.trace("eval [" + result + "] " + tag.getText());
            if (result)
            {
                if (sb.length() > 0)
                    sb.append(" " + tag.getText());
                else
                    sb.append(tag.getText());
            }
        }
        return sb.toString();
    }
    
    /**
     * language from SQL sentence.
     * 
     * @return the type of language used to SQL sentence.
     */
    public LanguageType getLanguageType()
    {
        return languageType;
    }
    
    /**
     * Retrieve the identifier name from tag.
     * 
     * @return name from tag
     */
    public String getName()
    {
        return id;
    }
    
    /**
     * add a new text tag.
     */
    public void addTag(ITextTag tag)
    {
        this.textTag.add(tag);
    }
    
    /**
     * add a set of static text from tag elements.
     */
    public void addTag(String text)
    {
        this.textTag.add(new StaticText(text));
    }
    
    /**
     * add a set of text tags (static or dynamic).
     */
    public void addTag(List<ITextTag> tags)
    {
        this.textTag.addAll(tags);
    }
}
