/*
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at http://www.apache.org/licenses/LICENSE-2.0 Unless required by
 * applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS
 * OF ANY KIND, either express or implied. See the License for the specific
 * language governing permissions and limitations under the License.
 */
package net.sf.jkniv.sqlegance.builder;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import net.sf.jkniv.sqlegance.builder.xml.AbstractSqlTag;
import net.sf.jkniv.sqlegance.builder.xml.DeleteTag;
import net.sf.jkniv.sqlegance.builder.xml.ISql;
import net.sf.jkniv.sqlegance.builder.xml.ISqlTag;
import net.sf.jkniv.sqlegance.builder.xml.IncludeTag;
import net.sf.jkniv.sqlegance.builder.xml.InsertTag;
import net.sf.jkniv.sqlegance.builder.xml.LanguageType;
import net.sf.jkniv.sqlegance.builder.xml.ProcedureParameterTag;
import net.sf.jkniv.sqlegance.builder.xml.ProcedureTag;
import net.sf.jkniv.sqlegance.builder.xml.SelectTag;
import net.sf.jkniv.sqlegance.builder.xml.UpdateTag;
import net.sf.jkniv.sqlegance.builder.xml.XMLFileStatus;
import net.sf.jkniv.sqlegance.builder.xml.dynamic.ChooseTag;
import net.sf.jkniv.sqlegance.builder.xml.dynamic.ITextTag;
import net.sf.jkniv.sqlegance.builder.xml.dynamic.IfTag;
import net.sf.jkniv.sqlegance.builder.xml.dynamic.OtherwiseTag;
import net.sf.jkniv.sqlegance.builder.xml.dynamic.SetTag;
import net.sf.jkniv.sqlegance.builder.xml.dynamic.StaticText;
import net.sf.jkniv.sqlegance.builder.xml.dynamic.WhenTag;
import net.sf.jkniv.sqlegance.builder.xml.dynamic.WhereTag;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * <p>
 * The XmlBuilderSql is the central class to read XML files and load the queries
 * at memory. The default initial file with the queries is named "SqlConfig.xml"
 * this file can have included other files like:
 * </p>
 * <p>
 * <code>
 * <include href="SQL1Test.xml"/>
 * <include href="SQL2Test.xml"/>  
 * </code>
 * </p>
 * <p>
 * If you don't like the name "SqlConfig.xml" or have some constraint with this
 * name; you can create other name to file and performed the method
 * <code>XmlBuilderSql.configFile("/newXmlFile.xml")</code>.
 * </p>
 * 
 * @author alisson gomes
 * @since 0.0.2
 */
public final class XmlBuilderSql
{
    private static final Logger            log               = LoggerFactory.getLogger(XmlBuilderSql.class);
    private static final Map<String, ISql>              map               = new HashMap<String, ISql>();
    private static String                  defaultFileConfig = "/SqlConfig.xml";
    private static boolean                 initialized       = false;
    private static List<SqlFile>           listFile          = new ArrayList<SqlFile>();
    private static XPath                   xpath             = XPathFactory.newInstance().newXPath();
    private static final String            ROOT_NODE         = "statements/";
    private static final String            ROOT_NODE_PACKAGE = "statements/package";
    
    private XmlBuilderSql()
    {
    }
    
    static
    {
        try
        {
            //init();
        }
        catch (Exception e)
        {
            System.err.println(e.getMessage());
            e.printStackTrace();
        }
    }
    
    private static void init()
    {
        if (!initialized)
        {
            InputStream is = XmlBuilderSql.class.getResourceAsStream(defaultFileConfig);
            if (is != null)
            {
                Document doc = loadXML(is);
                NodeList filesInc = doc.getElementsByTagName(IncludeTag.TAG_NAME);
                if (filesInc != null)
                {
                    for (int i = 0; i < filesInc.getLength(); i++)
                    {
                        Element e = (Element) filesInc.item(i);
                        SqlFile f = new SqlFile(e.getAttribute(IncludeTag.ATTRIBUTE_HREF));
                        listFile.add(f);
                        Document docInclude = loadXML(XmlBuilderSql.class.getResourceAsStream(f.getName()));
                        processXML(docInclude, f);
                    }
                }
                // for last, process the tags from main file
                SqlFile f = new SqlFile(defaultFileConfig);
                processTagElements(doc, f);
                initialized = true;
            }
        }
    }
    
    /**
     * Retrieve one query according your name.
     * 
     * @param name
     *            Name of the query.
     * @return Return the query object with SQL.
     * @exception IllegalArgumentException
     *                if the parameter name does not refer names of query
     *                configured this exception is throwed.
     */
    public static ISql getQuery(String name)
    {
        ISql sql = null;
        if (!initialized)
        {
            try
            {
                init();
            }
            catch (Exception e)
            {
                e.printStackTrace();
            }
        }
        if (!map.containsKey(name))
            throw new IllegalArgumentException("Query not found [" + name + "]");
        
        sql = map.get(name);
        
        return sql;
    }
    
    public static boolean containsQuery(String name)
    {
        return map.containsKey(name);
    }
    
    /**
     * Re-initialize the mapping XML file, making the framework read again the
     * XML file named "fileConfig".
     * 
     * @param fileConfig
     *            The XML file with SQL sentences.
     * @throws Exception
     */
    public static void configFile(String fileConfig)
    {
        defaultFileConfig = fileConfig;
        initialized = false;
        listFile.clear();
        map.clear();
        init();
    }
    
    /**
     * Load the XML file and make your parse with XML schema.
     * 
     * @param xml
     *            path from XML file in absolute format (start with '/')
     * @return Return XML file as Document object.
     * @throws RuntimeException
     *             is launched in case something wrong occurs; like
     *             ParserConfigurationException, SAXException or IOException.
     * @see Document
     * @see ParserConfigurationException
     * @see SAXException
     * @see IOException
     */
    private static Document loadXML(InputStream xml)
    {
        Document doc = null;
        if (xml == null)
            throw new ConfigurationException("There is XML file '<include href=...' that is not correctly named, check the name "
                    + "from your xml files at '<include href=...' tag. The file path is absolute and must agree that package is.");
        try
        {
            DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();
            // docBuilderFactory.setAttribute("http://java.sun.com/xml/jaxp/properties/schemaLanguage",
            // "http://www.w3.org/2001/XMLSchema");
            // docBuilderFactory.setAttribute("http://java.sun.com/xml/jaxp/properties/schemaSource",
            // "sqlegance-0.1.xsd");
            
            DocumentBuilder docBuilder;
            docBuilder = docBuilderFactory.newDocumentBuilder();
            doc = docBuilder.parse(xml);
        }
        catch (ParserConfigurationException e)
        {
            // TODO re-design exception
            throw new RuntimeException("Error in parser the xml file [" + xml + "]. ParserConfigurationException: " + e.getMessage()
                    + ". Verify if the name from file start with '/' and contains the package, because the path is absolute");
        }
        catch (SAXException e)
        {
            // TODO re-design exception
            throw new RuntimeException("Error in parser the xml file [" + xml + "]. SAXException: " + e.getMessage());
        }
        catch (IOException e)
        {
            // TODO re-design exception
            throw new RuntimeException("Error in parser the xml file [" + xml + "]. IOException: " + e.getMessage());
        }
        return doc;
    }
    
    /**
     * Processes a set of XML files that were not read. If the file have someone
     * xi:include to include some XML file, these files should be read
     * recursively your sentences.
     * 
     * @param doc
     *            Document loaded ready to be processed
     * @param file
     *            object that must be processed.
     */
    private static void processXML(Document doc, SqlFile file)
    {
        log.trace("process xml " + file);
        if (file.getStatus() == XMLFileStatus.PROCESSED)
            return;
        file.processed();
        // Puts all Text nodes in the full depth of the sub-tree underneath this
        // Node: elements, comments, processing instructions, CDATA sections,
        // and entity references
        doc.getDocumentElement().normalize();
        // process the include files recursively
        NodeList filesInc = doc.getElementsByTagName(IncludeTag.TAG_NAME);
        if (filesInc != null)
        {
            for (int i = 0; i < filesInc.getLength(); i++)
            {
                Element e = (Element) filesInc.item(i);
                SqlFile fileIncluded = new SqlFile(e.getAttribute(IncludeTag.ATTRIBUTE_HREF));
                listFile.add(fileIncluded);
                Document docInclude = loadXML(XmlBuilderSql.class.getResourceAsStream(fileIncluded.getName()));
                processXML(docInclude, fileIncluded);
            }
        }
        processTagElements(doc, file);
    }
    
    private static void processTagElements(Document doc, SqlFile f)
    {
        processTagsElements(evaluateXPATH(ROOT_NODE+SelectTag.TAG_NAME, doc), f.getName(), "");
        processTagsElements(evaluateXPATH(ROOT_NODE+InsertTag.TAG_NAME, doc), f.getName(), "");
        processTagsElements(evaluateXPATH(ROOT_NODE+UpdateTag.TAG_NAME, doc), f.getName(), "");
        processTagsElements(evaluateXPATH(ROOT_NODE+DeleteTag.TAG_NAME, doc), f.getName(), "");
        processProcedureElements(evaluateXPATH(ROOT_NODE+ProcedureTag.TAG_NAME, doc), f.getName(), "");
        
        NodeList nodes = evaluateXPATH(ROOT_NODE_PACKAGE, doc);
        if (nodes != null)
        {
            for (int i=0; i<nodes.getLength(); i++)
            {
                Node node = nodes.item(i);
                if (node.getNodeType() == Node.ELEMENT_NODE)
                {
                    Element element = (Element) node;
                    String name = element.getAttribute("name");
                    processTagsElements(evaluateXPATH(SelectTag.TAG_NAME, element), f.getName(), name);
                    processTagsElements(evaluateXPATH(InsertTag.TAG_NAME, element), f.getName(), name);
                    processTagsElements(evaluateXPATH(UpdateTag.TAG_NAME, element), f.getName(), name);
                    processTagsElements(evaluateXPATH(DeleteTag.TAG_NAME, element), f.getName(), name);
                    processProcedureElements(evaluateXPATH(ProcedureTag.TAG_NAME, element), f.getName(), name);
    
                }            
            }
        }
    }
    
    private static void processTagsElements(NodeList nodes, String fileName, String packet)
    {
        for (int i = 0; i < nodes.getLength(); i++)
        {
            Node node = nodes.item(i);
            if (node.getNodeType() == Node.ELEMENT_NODE)
            {
                Element element = (Element) node;
                String id = element.getAttribute(AbstractSqlTag.ATTRIBUTE_NAME);
                LanguageType type = getLanguageType(element.getAttribute(AbstractSqlTag.ATTRIBUTE_TYPE));
                ISqlTag tag = getTag(node.getNodeName(), id, type);
                
                for (int j = 0; j < element.getChildNodes().getLength(); j++)
                {
                    ITextTag textTag = getDynamicNode(element.getChildNodes().item(j));
                    if (textTag != null)
                    {
                        tag.addTag(textTag);
                    }
                }
                if (packet == null || "".equals(packet))
                    putSql(id, node.getNodeName(), fileName, tag);
                else
                    putSql(packet+"."+id, node.getNodeName(), fileName, tag);
            }
        }
    }
    
    private static ITextTag getDynamicNode(Node node)
    {
        String nodeName = node.getNodeName();
        String text = node.getNodeValue();
        ITextTag tag = null;
        // log.info("tag child nodes <"+ nodeName + "> " + text);
        if (IfTag.TAG_NAME.equals(nodeName))
        {
            Element element = (Element) node;
            tag = new IfTag(element.getAttribute(IfTag.ATTRIBUTE_TEST), getTextFromElement(element));
        }
        else if (WhereTag.TAG_NAME.equals(nodeName))
        {
            Element element = (Element) node;
            List<ITextTag> listIfTag = processTagDecision(element);            
            tag = new WhereTag(listIfTag);
        }
        else if (SetTag.TAG_NAME.equals(nodeName))
        {
            Element element = (Element) node;
            List<ITextTag> listIfTag = processTagDecision(element);
            tag = new SetTag(listIfTag);
        }
        else if (ChooseTag.TAG_NAME.equals(nodeName))
        {
            Element element = (Element) node;
            List<WhenTag> listWhenTag = processWhenTag(element);
            tag = new ChooseTag(listWhenTag);
        }
        else
        // #text or #cdata-section
        {
            if (text != null)
            {
                text = text.trim();
                if (!"".equals(text))
                {
                    tag = new StaticText(text);
                }
            }
        }
        return tag;
    }
    
    private static ISqlTag getTag(String nodeName, String id, LanguageType type)
    {
        ISqlTag tag = null;
        
        if (SelectTag.TAG_NAME.equals(nodeName))
            tag = new SelectTag(id, type);
        else if (InsertTag.TAG_NAME.equals(nodeName))
            tag = new InsertTag(id, type);
        else if (UpdateTag.TAG_NAME.equals(nodeName))
            tag = new UpdateTag(id, type);
        else if (DeleteTag.TAG_NAME.equals(nodeName))
            tag = new DeleteTag(id, type);
        else if (ProcedureTag.TAG_NAME.equals(nodeName))
            tag = new ProcedureTag(id, type);
        
        return tag;
    }
    
    private static void processProcedureElements(NodeList procedures, String fileName, String packet)
    {
        for (int s = 0; s < procedures.getLength(); s++)
        {
            Node firstNode = procedures.item(s);
            if (firstNode.getNodeType() == Node.ELEMENT_NODE)
            {
                Element element = (Element) firstNode;
                String id = element.getAttribute(ProcedureTag.ATTRIBUTE_NAME);
                String spName = element.getAttribute(ProcedureTag.ATTRIBUTE_SPNAME);
                ProcedureTag tag = new ProcedureTag(id, LanguageType.STORED);
                tag.setSpName(spName);
                ProcedureParameterTag[] paramsTag = processParameterTag(element);
                tag.setParams(paramsTag);
                putSql(id, ProcedureTag.TAG_NAME, fileName, tag);
            }
        }
    }
        
    /**
     * Evalue all decision tags (if/choose) to guarantee the same execution order that are finds in xml file
     * 
     * @param element
     * @return
     */
    private static List<ITextTag> processTagDecision(Element element)
    {
        List<ITextTag> list = new ArrayList<ITextTag>();
        NodeList nodeList = evaluateXPATH(IfTag.TAG_NAME + " | " + ChooseTag.TAG_NAME, element);

        for (int i = 0; i < nodeList.getLength(); i++)
        {
            Element elementNode = (Element) nodeList.item(i);
            //log.info("tag child nodes <"+ elementNode.getTagName() + "> ");
            if (IfTag.TAG_NAME.equalsIgnoreCase(elementNode.getTagName())) {
            	IfTag tag = new IfTag(elementNode.getAttribute(IfTag.ATTRIBUTE_TEST), getTextFromElement(elementNode));
            	list.add(i, tag);
            }
            else if (ChooseTag.TAG_NAME.equalsIgnoreCase(elementNode.getTagName())) 
            {
            	List<WhenTag> listWhen = processWhenTag(elementNode);
            	list.add(new ChooseTag(listWhen));
            }
        }
        return list;
    }

    /*
     * @param element
     * @return
     *
    private static List<ChooseTag> processChooseTag(Element element)
    {
        List<ChooseTag> listChoose = new ArrayList<ChooseTag>();
        NodeList nodeList = evaluateXPATH(ChooseTag.TAG_NAME, element);
        for (int i = 0; i < nodeList.getLength(); i++)
        {
            Element whnEle = (Element) nodeList.item(i);
        	List<WhenTag> listWhen = processWhenTag(whnEle);
        	listChoose.add(new ChooseTag(listWhen));
        }
        return listChoose;
    }
    */    
    /**
     * TODO documentation
     * 
     * @param element
     * @return
     */
    private static List<WhenTag> processWhenTag(Element element)
    {
        List<WhenTag> list = new ArrayList<WhenTag>();
        NodeList nodeList = evaluateXPATH(WhenTag.TAG_NAME, element);
        for (int i = 0; i < nodeList.getLength(); i++)
        {
            Element whnEle = (Element) nodeList.item(i);
            WhenTag tag = new WhenTag(whnEle.getAttribute(WhenTag.ATTRIBUTE_TEST), getTextFromElement(whnEle));
            list.add(i, tag);
        }
        nodeList = evaluateXPATH(OtherwiseTag.TAG_NAME, element);
        for (int i = 0; i < nodeList.getLength(); i++)
        {
            Element otherEle = (Element) nodeList.item(i);
            OtherwiseTag tag = new OtherwiseTag(getTextFromElement(otherEle));
            list.add(tag);
            if (i > 0)
            	throw new ConfigurationException("There are more one <otherwise> tag inner <choose> tag");
        }
        return list;
    }

    /**
     * Read and load the values of <parameter> tag from <procedure> tag.
     * 
     * @param element
     *            procedure node
     * @return
     */
    private static ProcedureParameterTag[] processParameterTag(Element element)
    {
        NodeList nodeList = evaluateXPATH(ProcedureParameterTag.TAG_NAME, element);
        ProcedureParameterTag[] params = new ProcedureParameterTag[nodeList.getLength()];
        for (int i = 0; i < nodeList.getLength(); i++)
        {
            ProcedureParameterTag tag = null;
            Element ele = (Element) nodeList.item(i);
            String property = ele.getAttribute(ProcedureParameterTag.ATTRIBUTE_PROPERTY);
            String mode = ele.getAttribute(ProcedureParameterTag.ATTRIBUTE_MODE);
            String sqlType = ele.getAttribute(ProcedureParameterTag.ATTRIBUTE_SQLTYPE);
            String typeName = ele.getAttribute(ProcedureParameterTag.ATTRIBUTE_TYPENAME);
            
            if (!"".equals(typeName) && "IN".equals(mode))
                throw new ConfigurationException("The parameter [" + property + "] is wrong. There is a typeName [" + typeName
                        + "] but is like IN mode. TypeName must be used with OUT or INOUT");
            
            if (!"".equals(typeName) && "".equals(sqlType))
                throw new ConfigurationException("The parameter [" + property + "] is wrong. There is a typeName [" + typeName
                        + "] but dont have a sqlType defined");
            
            if ("".equals(sqlType) && "".equals(typeName))
                tag = new ProcedureParameterTag(property, mode);
            else if (!"".equals(typeName) && "".equals(typeName))
                tag = new ProcedureParameterTag(property, mode, sqlType);
            else
                tag = new ProcedureParameterTag(property, mode, sqlType, typeName);
            
            params[i] = tag;
        }
        return params;
    }
    
    /**
     * TODO documentation
     * 
     * @param expressionXpath
     * @param element
     * @return
     */
    private static NodeList evaluateXPATH(String expressionXpath, Element element)
    {
        NodeList nodeList = null;
        try
        {
            nodeList = (NodeList) xpath.evaluate(expressionXpath, element, XPathConstants.NODESET);
        }
        catch (XPathExpressionException e)
        {
            log.info("XPath is wrong: " + expressionXpath);
        }
        return nodeList;
    }
    

    /**
     * TODO documentation
     * 
     * @param expressionXpath
     * @param element
     * @return
     */
    private static NodeList evaluateXPATH(String expressionXpath, Document doc)
    {
        NodeList nodeList = null;
        try
        {
        	XPathExpression exp = xpath.compile(expressionXpath);
            nodeList = (NodeList) exp.evaluate(doc, XPathConstants.NODESET);
        }
        catch (XPathExpressionException e)
        {
            log.info("XPath is wrong: " + expressionXpath);
        }
        return nodeList;
    }
    /**
     * TODO documentation
     * 
     * @param element
     * @return
     */
    private static String getTextFromElement(Element element)
    {
        String text = element.getChildNodes().item(0).getNodeValue().trim();
        
        if ((text == null || "".equals(text)) && element.getChildNodes().item(0).getNextSibling() != null)
        {
            text = element.getChildNodes().item(0).getNextSibling().getNodeValue().trim();
        }
        return text;
    }
    
    /**
     * TODO documentation
     * 
     * @param type
     * @return
     */
    private static LanguageType getLanguageType(String type)
    {
        LanguageType t = null;
        for (LanguageType qt : LanguageType.values())
        {
            if (String.valueOf(type).equalsIgnoreCase(qt.toString()))
            {
                t = qt;
                break;
            }
        }
        if (t == null)
        {
            t = LanguageType.JPQL;
            System.err.println("Type [" + type + "] query no defined (JPQL, HQL, NATIVE, STORED), default JPQL");
        }
        return t;
    }
    
    /**
     * TODO documentation
     * 
     * @param name
     * @param tagName
     * @param fileName
     * @param tag
     */
    private static void putSql(String name, String tagName, String fileName, ISql tag)
    {
        if (map.containsKey(name))
            throw new IllegalArgumentException("Query with name <" + tagName + " name=\"" + name + "\"... is duplicated at file [" + fileName + "]");
        
        map.put(name, tag);
    }
    
    /**
     * Retrieve an unmodifiable list from original file list.
     * 
     * @return
     */
    public static List<SqlFile> getFiles()
    {
        return Collections.unmodifiableList(listFile);
    }
    
    /**
     * Copy the items from list source to destiny list.
     * @param source source from objects
     * @param destiny destiny list from objects.
     */
    private static void copyWhen(List<ITextTag> destiny, List<WhenTag> source) 
    {
    	for (int i=0; i<source.size(); i++)
    	{
    		ITextTag t = source.get(i);
    		destiny.add(t);	
    	}
    }

    /**
     * Copy the items from list source to destiny list.
     * @param source source from objects
     * @param destiny destiny list from objects.
     */
    private static void copyChoose(List<ITextTag> destiny, List<ChooseTag> source) 
    {
    	for (int i=0; i<source.size(); i++)
    	{
    		ITextTag t = source.get(i);
    		destiny.add(t);	
    	}
    }
}
