/*
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at http://www.apache.org/licenses/LICENSE-2.0 Unless required by
 * applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS
 * OF ANY KIND, either express or implied. See the License for the specific
 * language governing permissions and limitations under the License.
 */
package net.sf.jkniv.sqlegance;

import java.util.List;

/**
 * Main interface to persistent a collections-like objects. The implementation
 * of this interface are at the other component named whinstone-*
 * 
 * @author alisson gomes
 * @param <T>
 *            Type of object
 * @since 0.0.2
 */
public interface IRepository<T>
{
    static String PROPERTY_SHOW_CONFIG = "jkniv.repository.show_config";
    
    /**
     * Add a new Object at repository.
     * 
     * @param object
     */
    void add(T object);
    
    /**
     * Remove an existent object at repository.
     * 
     * @param object
     */
    void remove(T object);
    
    /**
     * Execute a query to remove one or many objects from repository.
     * 
     * @param query
     *            Query with parameters
     * @return Return the numbers of objects removed.
     * @throws IllegalArgumentException
     *             when the query is different from delete
     */
    int remove(IQuery query);
    
    /**
     * Up date the object value at repository.
     * 
     * @param object
     */
    void update(T object);
    
    /**
     * Up date by query many objects.
     * 
     * @param query
     *            Query with parameters
     * @return Return the numbers of objects updated.
     * @throws IllegalArgumentException
     *             when the query is different from update sentence
     */
    int update(IQuery query);
    
    /**
     * Get one object instance from repository using a query with name "T.get",
     * where 'T' it's a generic type.
     * 
     * @param object
     *            Properties values of clause where to get the object.
     * @return Return the object that matches with query. A null reference is
     *         returned if the query no match anyone object.
     */
    T get(T object);
    
    /**
     * Get one object instance from repository using a query.
     * 
     * @param query
     *            Query with parameters
     * @return Return the object that matches with query. A null reference is
     *         returned if the query no match anyone object.
     */
    T get(IQuery query);
    
    /**
     * Get a set of objects from repository using a query with name "T.list",
     * where 'T' it's a generic type.
     * 
     * @return Return a set of object that matches with query. A empty list is
     *         returned if the query no match anyone object.
     */
    List<T> list();
    
    /**
     * Get a set of objects from repository using a query.
     * 
     * @param query
     *            Query with parameters
     * @return Return a set of object that matches with query. A empty list is
     *         returned if the query no match anyone object.
     */
    List<T> list(IQuery query);
    
    /**
     * Get a set of objects 'G' from repository using a query.
     * 
     * @param query
     *            Query with parameters
     * @param clazz
     *            Type of object from list of object
     * @return Return a set of object that matches with query. A empty list is
     *         returned if the query no match anyone object.
     */
    <G> List<G> list(IQuery query, Class<G> c);
    
    /*
     * Perform a query and return a scalar value.
     * @param query
     * @return Return a singular value from a query. A null value is returned
     * when the query cannot found any value. Number getValue(IQuery query);
     */
    
    /**
     * Command to execute SQL statements that are in the buffer.
     */
    void flush();
}
