/* 
 * JKNIV, SQLegance keeping queries maintainable.
 * 
 * Copyright (C) 2017, the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software Foundation, Inc., 
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package net.sf.jkniv.sqlegance.builder.xml;

import net.sf.jkniv.sqlegance.Deletable;
import net.sf.jkniv.sqlegance.Insertable;
import net.sf.jkniv.sqlegance.LanguageType;
import net.sf.jkniv.sqlegance.Selectable;
import net.sf.jkniv.sqlegance.Statistical;
import net.sf.jkniv.sqlegance.Updateable;
import net.sf.jkniv.sqlegance.dialect.SqlDialect;
import net.sf.jkniv.sqlegance.statement.ResultSetConcurrency;
import net.sf.jkniv.sqlegance.statement.ResultSetHoldability;
import net.sf.jkniv.sqlegance.statement.ResultSetType;
import net.sf.jkniv.sqlegance.validation.ValidateType;
import net.sf.jkniv.sqlegance.transaction.Isolation;

public class TagFactory
{
    
    public static OneToManyTag newOneToMany(String property, String typeOf, String impl)
    {
        return new OneToManyTag(property, typeOf, impl);
    }
    
    public static Selectable newSelect(String id, LanguageType languageType)
    {
        return new SelectTag(id, languageType);
    }

    public static Selectable newSelect(String id, LanguageType languageType, SqlDialect sqlDialect)
    {
        return new SelectTag(id, languageType, sqlDialect);
    }
    
    /**
     * Build a new <code>select</code> tag from XML file.
     * 
     * @param id
     *            Name/Identify from tag.
     * @param languageType
     *            type of language from tag.
     * @param isolation Retrieves the current transaction isolation level for the query.
     * @param timeout Retrieves the number of seconds the repository will wait for a Query
     * object to execute.
     * @param cache 
     *            Indicate if must keep SQL result in cache           
     * @param resultSetType TODO javadoc
     * @param resultSetConcurrency TODO javadoc
     * @param resultSetHoldability holdability from {@code ResultSet}
     * @param returnType type of object must be returned
     * @param groupBy columns name to group the result of query (columns name separated by comma)
     * @param validateType validation to apply before execute SQL.
     * @param stats sql statistical
     * @return select tag
     */
    public static Selectable newSelect(String id, LanguageType languageType, Isolation isolation, int timeout,
            String cache, ResultSetType resultSetType,
            ResultSetConcurrency resultSetConcurrency, ResultSetHoldability resultSetHoldability, String returnType,
            String groupBy, ValidateType validateType, Statistical stats)
    {
        return new SelectTag(id, languageType, isolation, timeout, cache, resultSetType,
                resultSetConcurrency, resultSetHoldability, returnType, groupBy, validateType, stats);
    }
    
    public static Insertable newInsert(String id, LanguageType languageType)
    {
        return new InsertTag(id, languageType);
    }
    
    public static Insertable newInsert(String id, LanguageType languageType, SqlDialect sqlDialect)
    {
        return new InsertTag(id, languageType, sqlDialect);
    }
    
    
    public static Insertable newInsert(String id, LanguageType languageType, 
            Isolation isolation, int timeout, ValidateType validateType, Statistical stats)
    {
        return new InsertTag(id, languageType, isolation, timeout, validateType, stats);
    }

    /**
     * Build a new <code>where</code> tag.
     * @param type generator strategy
     * @param columns columns names
     * @param props properties name
     * @param text value from tag
     * @return tag for auto generate key
     */
    public static AutoGeneratedKey newAutoGeneratedKey(String type, String columns, String props, String text)
    {
        return new AutoGeneratedKeyTag(type, columns, props, text);
    }
    
    public static Updateable newUpdate(String id, LanguageType languageType)
    {
        return new UpdateTag(id, languageType);
    }

    public static Updateable newUpdate(String id, LanguageType languageType, SqlDialect sqlDialect)
    {
        return new UpdateTag(id, languageType, sqlDialect);
    }
    
    public static Updateable newUpdate(String id, LanguageType languageType, Isolation isolation, 
            int timeout, ValidateType validateType, Statistical stats)
    {
        return new UpdateTag(id, languageType, isolation, timeout, validateType, stats);
    }

    
    public static Deletable newDelete(String id, LanguageType languageType)
    {
        return new DeleteTag(id, languageType);
    }

    public static Deletable newDelete(String id, LanguageType languageType, SqlDialect sqlDialect)
    {
        return new DeleteTag(id, languageType, sqlDialect);
    }

    public static Deletable newDelete(
            String id, LanguageType languageType, Isolation isolation, int timeout,
            ValidateType validateType, Statistical stats)
    {
        return new DeleteTag(id, languageType, isolation, timeout, validateType, stats);
    }
    
}
