/* 
 * JKNIV, SQLegance keeping queries maintainable.
 * 
 * Copyright (C) 2017, the original author or authors.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software Foundation, Inc., 
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package net.sf.jkniv.sqlegance.builder.xml;

import java.util.Collections;
import java.util.List;

import net.sf.jkniv.sqlegance.KeyGeneratorType;
import net.sf.jkniv.sqlegance.builder.xml.dynamic.ITextTag;

/**
 * Represent the <code>autoGeneratedKey</code> configuration from SQL insert element.
 * 
 * @author Alisson Gomes
 * @since 0.6.0
 */
class AutoGeneratedKeyTag implements AutoGeneratedKey
{
    private String             text;
    private KeyGeneratorType   type;
    private String             columns;
    private String             properties;
    
    /**
     * Build a new <code>where</code> tag.
     * @param type generator strategy
     * @param columns columns names
     * @param props properties name
     * @param text text value from tag
     */
    public AutoGeneratedKeyTag(String type, String columns, String props, String text)
    {
        this.text = text;
        this.type = KeyGeneratorType.get(type);
        this.columns = (columns == null ? "" : columns);
        this.properties = (props == null ? "" : props);
    }
    
    /**
     * Static element, return false always.
     * @param rootObjects ongl object to evaluate
     * @return false always.
     */
    @Override
    public boolean eval(Object rootObjects)
    {
        return false;
    }
    
    /**
     * Retrieve the dynamic text from XML element.
     * 
     * @return text from XML element.
     */
    @Override
    public String getText()
    {
        return text.toString();
    }
    
    @Override
    public String getText(Object rootObjects)
    {
        return this.text.toString();
    }
    
    /* (non-Javadoc)
     * @see net.sf.jkniv.sqlegance.builder.xml.AutoGenerateKey#getStrategy()
     */
    @Override
    public KeyGeneratorType getStrategy()
    {
        return type;
    }
    
    /* (non-Javadoc)
     * @see net.sf.jkniv.sqlegance.builder.xml.AutoGenerateKey#isAutoStrategy()
     */
    @Override
    public boolean isAutoStrategy()
    {
        return (type == KeyGeneratorType.AUTO);
    }
    
    /* (non-Javadoc)
     * @see net.sf.jkniv.sqlegance.builder.xml.AutoGenerateKey#isSequenceStrategy()
     */
    @Override
    public boolean isSequenceStrategy()
    {
        return (type == KeyGeneratorType.SEQUENCE);
    }
    
//    public boolean isTableStrategy()
//    {
//        return (type == KeyGeneratorType.TABLE);
//    }
    
    /* (non-Javadoc)
     * @see net.sf.jkniv.sqlegance.builder.xml.AutoGenerateKey#getColumns()
     */
    @Override
    public String getColumns()
    {
        return columns;
    }
    
    /* (non-Javadoc)
     * @see net.sf.jkniv.sqlegance.builder.xml.AutoGenerateKey#getColumnsAsArray()
     */
    @Override
    public String[] getColumnsAsArray()
    {
        return buildArray(this.columns);
    }
    
    /* (non-Javadoc)
     * @see net.sf.jkniv.sqlegance.builder.xml.AutoGenerateKey#getPropertiesAsArray()
     */
    @Override
    public String[] getPropertiesAsArray()
    {
        return buildArray(this.properties);
    }
    
    private String[] buildArray(String valuesWithComma)
    {
        String[] columns_ = {};
        if ("".equals(valuesWithComma))
            return columns_;
        
        if (valuesWithComma.indexOf(",") > 0)
            columns_ = valuesWithComma.split(",");
        else if (valuesWithComma.length() > 0)
            columns_ = new String[]{valuesWithComma};
        
        return columns_;
        
    }

    /* (non-Javadoc)
     * @see net.sf.jkniv.sqlegance.builder.xml.AutoGenerateKey#getProperties()
     */
    @Override
    public String getProperties()
    {
        return properties;
    }
    
    /**
     * Indicate if text is dynamic or static.
     * 
     * @return always true is returned, because this object save dynamic text.
     */
    @Override
    public boolean isDynamic()
    {
        return false;
    }

    @Override
    public boolean isDynamicGroup()
    {
        return false;
    }
    
    @Override
    public List<? extends ITextTag> getTags()
    {
        return Collections.emptyList();
    }
}
