/*
jGuard is a security framework based on top of jaas (java authentication and authorization security).
it is written for web applications, to resolve simply, access control problems.

http://sourceforge.net/projects/jguard/

Copyright (C) 2004  Charles GAY

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


jGuard project home page:
http://sourceforge.net/projects/jguard/

*/
package net.sf.jguard.jee.listeners;

import java.security.Policy;
import java.util.ArrayList;

import javax.servlet.ServletContext;
import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;

import net.sf.jguard.ext.InitHelper;
import net.sf.jguard.core.authentication.AccessContext;
import net.sf.jguard.core.authorization.policy.MultipleAppPolicy;
import net.sf.jguard.ext.SecurityConstants;
import net.sf.jguard.jee.authentication.http.AccessFilter;
import net.sf.jguard.jee.authentication.http.HttpConstants;
import net.sf.jguard.jee.authentication.http.HttpServletAuthenticationBindingsFactory;
import net.sf.jguard.jee.util.ContextUtil;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;



/**
 * handle webapp classloader reference in the
 * JGuardPolicy classloaders repository.
 * @author <a href="mailto:diabolo512@users.sourceforge.net">Charles Gay</a>
 * @author <a href="mailto:vberetti@users.sourceforge.net">Vincent Beretti</a>
 */
public class ContextListener implements ServletContextListener {


	private static final Logger logger = LoggerFactory.getLogger(ContextListener.class);
	

    /**
	 * method called when the webapp shutdown:
	 * this method unregister the webapp in the JGuardPolicy repository.
	 * @see javax.servlet.ServletContextListener#contextDestroyed(javax.servlet.ServletContextEvent)
	 */
	public void contextDestroyed(ServletContextEvent servletContextEvent) {
		logger.debug(" context destroyed ");
		ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
		if(Policy.getPolicy() instanceof MultipleAppPolicy){
		  MultipleAppPolicy policy = (MultipleAppPolicy) Policy.getPolicy();
          policy.unregisterPermissionProvider(contextClassLoader);
		}

		servletContextEvent.getServletContext().removeAttribute(SecurityConstants.CAPTCHA_SERVICE);
		servletContextEvent.getServletContext().removeAttribute(HttpConstants.USERS_IN_SESSION);
	}

	/**
	 * method called when the webapp start.
	 * install jGuard overall Configuration and Policy.
	 * @see javax.servlet.ServletContextListener#contextInitialized(javax.servlet.ServletContextEvent)
	 */
	public void contextInitialized(ServletContextEvent contextEvent) {
		logger.debug("#####  initializing ContextListener ... #####");
		ServletContext context = contextEvent.getServletContext();
		context.setAttribute(HttpConstants.USERS_IN_SESSION,new ArrayList());

		String webappHomePath = ContextUtil.getContextPath(context,"/");
                if(!webappHomePath.endsWith("/")){
                    webappHomePath+="/";
                }
		String applicationName = context.getServletContextName();
        String authorizationConfigurationLocation = getAuthorizationConfigurationLocation(context);
        String authenticationConfigurationLocation = getAuthenticationConfigurationLocation(context);

        AccessContext accessContext = new AccessContext();
        accessContext.setAttribute(AccessFilter.SERVLET_CONTEXT, context);
        String authFactoryClass = HttpServletAuthenticationBindingsFactory.class.getName();

        InitHelper.installAuthorizationAndAuthentication(authorizationConfigurationLocation, webappHomePath, authFactoryClass, accessContext, applicationName,authenticationConfigurationLocation);

		logger.debug("#####  ContextListener initialized  #####");
	}

    /**
     *
     * @param context
     * @return URI string representation of the Authorization configuration location.
     */
    private String getAuthorizationConfigurationLocation(ServletContext context) {
        // use default authorizationConfigurationLocation unless it is explicitly defined
        String authorizationConfigurationLocation = null;
        if (context.getInitParameter(HttpConstants.AUTHORIZATION_CONFIGURATION_LOCATION) != null) {
            authorizationConfigurationLocation = ContextUtil.getContextPath(context, context.getInitParameter(HttpConstants.AUTHORIZATION_CONFIGURATION_LOCATION));
        } else {
            authorizationConfigurationLocation = ContextUtil.getContextPath(context, HttpConstants.DEFAULT_AUTHORIZATION_CONFIGURATION_LOCATION);
        }
        return authorizationConfigurationLocation;
    }

    /**
     *
     * @param context
     * @return URI string representation of the Authentication configuration location.
     */
    private String getAuthenticationConfigurationLocation(ServletContext context) {
        //use default authenticationConfigurationLocation unless it is explicitly defined
        String authenticationConfigurationLocation = null;
        if (context.getInitParameter(HttpConstants.AUTHENTICATION_CONFIGURATION_LOCATION) != null) {
            authenticationConfigurationLocation = ContextUtil.getContextPath(context, context.getInitParameter(HttpConstants.AUTHENTICATION_CONFIGURATION_LOCATION));
        } else {
            authenticationConfigurationLocation = ContextUtil.getContextPath(context, HttpConstants.DEFAULT_AUTHENTICATION_CONFIGURATION_LOCATION);
        }
        return authenticationConfigurationLocation;
    }

	
}
