/*
jGuard is a security framework based on top of jaas (java authentication and authorization security).
it is written for web applications, to resolve simply, access control problems.
version $Name$
http://sourceforge.net/projects/jguard/

Copyright (C) 2004  Charles GAY

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


jGuard project home page:
http://sourceforge.net/projects/jguard/

*/
package net.sf.jguard.jee.authentication.schemes;

import java.io.IOException;
import java.security.Permission;
import java.util.Map;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import net.sf.jguard.core.CoreConstants;
import net.sf.jguard.core.authentication.AccessContext;
import net.sf.jguard.core.authentication.AuthenticationException;
import net.sf.jguard.core.authentication.AuthenticationUtils;
import net.sf.jguard.core.authentication.bindings.AuthenticationBindingsFactory;
import net.sf.jguard.core.authentication.bindings.StatefulAuthenticationBindings;
import net.sf.jguard.core.authentication.schemes.FormSchemeHandler;
import net.sf.jguard.core.authorization.permissions.URLPermission;
import net.sf.jguard.core.authorization.policy.AccessControllerUtils;
import net.sf.jguard.core.filters.PolicyEnforcementPointFilter;
import net.sf.jguard.jee.authentication.http.AccessFilter;
import net.sf.jguard.jee.authentication.http.HttpConstants;
import net.sf.jguard.jee.HttpPermissionFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
* implements an HTTP FORM Authentication scheme based on an HttpServlet API.
* @author <a href="mailto:diabolo512@users.sourceforge.net">Charles Gay</a>
*/
public class HttpServletFormSchemeHandler extends FormSchemeHandler{

    private String logonURI;
    private String logoffURI;
    private String logonProcessURI;
    
    private String loginField;
    private String passwordField;
    private String authenticationSucceedURI;
    private String authenticationFailedURI;
    
    
    private static final Logger logger = LoggerFactory.getLogger(HttpServletFormSchemeHandler.class.getName());
    private URLPermission authenticationSucceedPermission;
    private URLPermission authenticationFailedPermission;
    
    public HttpServletFormSchemeHandler(Map<String,String> parameters,AuthenticationBindingsFactory factory){
        super(parameters,factory);
        initSettings(parameters);
    
    }
    
    
    public void buildChallenge(AccessContext context)throws AuthenticationException{
        HttpServletRequest request = (HttpServletRequest)context.getAttribute(AccessFilter.SERVLET_REQUEST);
	HttpServletResponse response = (HttpServletResponse)context.getAttribute(AccessFilter.SERVLET_RESPONSE);
       if(!response.isCommitted()){
            try {
                response.sendRedirect(response.encodeRedirectURL(request.getContextPath()+logonURI));
            } catch (IOException ex) {
               logger.error(ex.getMessage(),ex);
               throw new AuthenticationException(ex);
            }
        }
    }


    public void authenticationSucceed(AccessContext context)throws AuthenticationException{
            StatefulAuthenticationBindings authenticationBindings = (StatefulAuthenticationBindings) factory.getAuthenticationBindings(context);
            context.setAttribute(PolicyEnforcementPointFilter.REDIRECT, "true");
            HttpServletRequest request = (HttpServletRequest)context.getAttribute(AccessFilter.SERVLET_REQUEST);
	    HttpServletResponse response = (HttpServletResponse)context.getAttribute(AccessFilter.SERVLET_RESPONSE);
		String redirectURI = authenticationSucceedURI;
		URLPermission lastAccessDeniedPermission = (URLPermission)authenticationBindings.getSessionAttribute(CoreConstants.LAST_ACCESS_DENIED_PERMISSION);
		String lastAccessDeniedURI;
		if(lastAccessDeniedPermission==null){
			lastAccessDeniedURI = authenticationSucceedURI;
		}else{
			lastAccessDeniedURI= lastAccessDeniedPermission.getURI();
		}
	    
	    
            AuthenticationUtils authenticationUtils = authenticationBindings.getAuthenticationUtils();
	    
	    //we redirect to the last 'access denied' URI before authentication
	    if(lastAccessDeniedURI!=null &&!"".equals(lastAccessDeniedURI)) {
	    	if(goToLastAccessDeniedUriOnSuccess){
	    		redirectURI = lastAccessDeniedURI;
                }else {
                        if(!AccessControllerUtils.hasPermission(authenticationUtils.getSubject(),authenticationSucceedPermission)){
                            redirectURI = logonURI;
                        }

	    	}
	    }


	    logger.debug(" user is authenticated "," redirect to "+redirectURI);
	    if(!response.isCommitted()){
                try {
                    response.sendRedirect(response.encodeRedirectURL(request.getContextPath()+redirectURI));
                } catch (IOException ex) {
                   logger.error(ex.getMessage(),ex);
                   throw new AuthenticationException(ex);
                }
	    }
	
	}
      

   public void authenticationFailed(AccessContext context)throws AuthenticationException{
            context.setAttribute(PolicyEnforcementPointFilter.REDIRECT, "true");
            HttpServletRequest request = (HttpServletRequest)context.getAttribute(AccessFilter.SERVLET_REQUEST);
	    HttpServletResponse response = (HttpServletResponse)context.getAttribute(AccessFilter.SERVLET_RESPONSE);
	    
		if(response.isCommitted()){
			logger.warn(" response is already committed ");
            return;
		}

	   //an URL for authentication failure event has been set
	   if(authenticationFailedPermission!=null && !((URLPermission)authenticationFailedPermission).getURI().equals("")){
            try {
                response.sendRedirect(response.encodeRedirectURL(request.getContextPath()+((URLPermission)authenticationFailedPermission).getURI()));
                logger.debug("authentication failed redirect to "+((URLPermission)authenticationFailedPermission).getURI());
            } catch (IOException ex) {
               logger.error(ex.getMessage(),ex);
               throw new AuthenticationException(ex);
            }
               logger.debug(" user is not authenticated  and redirected to "+request.getContextPath()+((URLPermission)authenticationFailedPermission).getURI());
	
	}
   }

    @Override
    protected String getLogin(AccessContext context) {
        HttpServletRequest request = (HttpServletRequest)context.getAttribute(AccessFilter.SERVLET_REQUEST);
        String login =request.getParameter(loginField);
        return login;
    }

    @Override
    protected String getPassword(AccessContext context) {
        HttpServletRequest request = (HttpServletRequest)context.getAttribute(AccessFilter.SERVLET_REQUEST);
        String pwd = request.getParameter(passwordField);        
        return pwd;
    }

    protected Permission getlogonProcessPermission() {
     return logonProcessPermission;   
    }

    public Permission getLogoffPermission() {
        return logoffPermission;   
    }
    
     public Permission getLogonPermission() {
        return logonPermission;   
    }

    @Override
    protected Permission getPermission(AccessContext context) {
        return new HttpPermissionFactory().getPermission(context);
    }

    protected void initSettings(Map<String, String> parameters) throws IllegalArgumentException {
        this.logonProcessURI = parameters.get("logonProcessURI");
        this.loginField = parameters.get("loginField");
        this.passwordField = parameters.get("passwordField");
        this.authenticationSucceedURI = parameters.get("authenticationSucceedURI");
        if (authenticationSucceedURI == null && "".equals(authenticationSucceedURI)) {
            throw new IllegalArgumentException("authenticationSucceedURI parameter is null but is required to instantiate HttpServletFormSchemeHandler");
        }
        this.authenticationSucceedPermission = new URLPermission(HttpConstants.AUTHENTICATION_SUCCEED_URI, authenticationSucceedURI);
        this.authenticationFailedURI = parameters.get("authenticationFailedURI");
        if (authenticationFailedURI == null || "".equals(authenticationFailedURI)) {
            throw new IllegalArgumentException("authenticationFailedURI parameter is null but is required to instantiate HttpServletFormSchemeHandler");
        }
        this.authenticationFailedPermission = new URLPermission(HttpConstants.AUTHENTICATION_FAILED_URI, authenticationFailedURI);

        this.logonURI = parameters.get(HttpConstants.LOGON_URI);
        logonPermission = new URLPermission(HttpConstants.LOGON_URI, logonURI);

        this.logoffURI = parameters.get(HttpConstants.LOGOFF_URI);
        this.logoffPermission = new URLPermission(HttpConstants.LOGOFF_URI, logoffURI);

        this.logonProcessURI = parameters.get(HttpConstants.LOGON_PROCESS_URI);
        logonProcessPermission = new URLPermission(HttpConstants.LOGON_PROCESS_URI, logonProcessURI);
    }

    

   
    
}
