/*
jGuard is a security framework based on top of jaas (java authentication and authorization security).
it is written for web applications, to resolve simply, access control problems.
version $Name$
http://sourceforge.net/projects/jguard/

Copyright (C) 2004  Charles GAY

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


jGuard project home page:
http://sourceforge.net/projects/jguard/

*/
package net.sf.jguard.jee.authentication.http;

import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Locale;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import net.sf.jguard.ext.SecurityConstants;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.octo.captcha.module.config.CaptchaModuleConfig;
import com.octo.captcha.service.CaptchaService;
import com.octo.captcha.service.EhcacheManageableCaptchaService;
import com.sun.image.codec.jpeg.JPEGCodec;
import com.sun.image.codec.jpeg.JPEGImageEncoder;

/**
 * 
 * @author <a href="mailto:diabolo512@users.sourceforge.net">Charles Gay</a>
 * @since 1.0
 */
public class CaptchaChallengeBuilder {

	private static final Logger logger = LoggerFactory.getLogger(CaptchaChallengeBuilder.class);
	
	public static void buildCaptchaChallenge(HttpServletRequest request, HttpServletResponse response) throws IOException {
		HttpSession session = request.getSession();
		CaptchaService service = (CaptchaService) session.getServletContext().getAttribute(SecurityConstants.CAPTCHA_SERVICE);
		if (service == null) {
			logger.debug("captcha service should be defined ");
			try {
				service = (CaptchaService)Thread.currentThread().getContextClassLoader().loadClass(CaptchaModuleConfig.getInstance().getServiceClass()).newInstance();
				session.getServletContext().setAttribute(SecurityConstants.CAPTCHA_SERVICE, service);
			} catch (InstantiationException e) {
				logger.error(e.getMessage());
			} catch (IllegalAccessException e) {
				logger.error(e.getMessage());
			} catch (ClassNotFoundException e) {
				logger.error(e.getMessage());
			}
			logger.debug(" CAPTCHA SERVICE=" + service.getClass().getName() + " will be defined");
			
			if (service.getClass().isAssignableFrom(EhcacheManageableCaptchaService.class)) {
				((EhcacheManageableCaptchaService) service).emptyCaptchaStore();
			}
			
		} else {
			logger.debug(" CAPTCHA SERVICE=" + service.getClass().getName());
		}
		byte[] captchaChallengeAsJpeg = buildCaptchaChallenge(session.getId(),request.getLocale(), service);
		
		
		// flush it in the response
		response.setHeader("Cache-Control", "no-store");
		response.setHeader("Pragma", "no-cache");
		response.setDateHeader("Expires", 0);
		response.setContentType("image/jpeg");
		ServletOutputStream responseOutputStream;
		try {
			responseOutputStream = response.getOutputStream();
			responseOutputStream.write(captchaChallengeAsJpeg);
			responseOutputStream.flush();
			responseOutputStream.close();
		} catch (IOException e) {
			logger.error(" captcha cannot be generated", e);
		}
	}
	
	
	/**
	 * build captcha challenge and return it as a byte array.
	 * @param captchaId
	 * @param locale
	 * @param service
	 * @return
	 * @throws IOException
	 */
	private static byte[] buildCaptchaChallenge(String captchaId,Locale locale,CaptchaService service) throws IOException {
		
		byte[] captchaChallengeAsJpeg = null;
		// the output stream to render the captcha image as jpeg into
		ByteArrayOutputStream jpegOutputStream = new ByteArrayOutputStream();
		// get the session id that will identify the generated captcha.
		// the same id must be used to validate the response, the session id is a good candidate!
		logger.debug("sessionID=" + captchaId);
		// call the ImageCaptchaService getChallenge method
		BufferedImage challenge = (BufferedImage) service.getChallengeForID(captchaId, locale);
		logger.debug("challenge=" + service.getQuestionForID(captchaId, locale));
		logger.debug(" service=" + service);

		// a jpeg encoder
		JPEGImageEncoder jpegEncoder = JPEGCodec.createJPEGEncoder(jpegOutputStream);
		jpegEncoder.encode(challenge);
		captchaChallengeAsJpeg = jpegOutputStream.toByteArray();
		return captchaChallengeAsJpeg;
	}

}
