/*
jGuard is a security framework based on top of jaas (java authentication and authorization security).
it is written for web applications, to resolve simply, access control problems.
version $Name$
http://sourceforge.net/projects/jguard/

Copyright (C) 2004  Charles GAY

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


jGuard project home page:
http://sourceforge.net/projects/jguard/

*/
package net.sf.jguard.jee.authentication.callbacks;


import java.io.IOException;

import java.net.UnknownHostException;
import java.util.List;
import java.util.Locale;
import org.slf4j.Logger;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.LanguageCallback;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.servlet.http.HttpServletRequest;
import net.sf.jguard.core.authentication.AccessContext;
import net.sf.jguard.core.authentication.bindings.AuthenticationBindings;
import net.sf.jguard.core.authentication.callbacks.AbstractCallbackHandler;
import net.sf.jguard.core.authentication.callbacks.InetAddressCallback;
import net.sf.jguard.core.authentication.schemes.AuthenticationSchemeHandler;
import net.sf.jguard.jee.authentication.http.AccessFilter;
import org.slf4j.LoggerFactory;
import org.xbill.DNS.DClass;
import org.xbill.DNS.ExtendedResolver;
import org.xbill.DNS.Message;
import org.xbill.DNS.Name;
import org.xbill.DNS.Record;
import org.xbill.DNS.Resolver;
import org.xbill.DNS.ReverseMap;
import org.xbill.DNS.Section;
import org.xbill.DNS.Type;

/**
* implementation for the <b>HttpServlet</b> underlying communication technology.
* @author <a href="mailto:diabolo512@users.sourceforge.net">Charles Gay</a>
*/
public class HttpServletCallbackHandler extends AbstractCallbackHandler{
    
    
    private static final Logger logger = LoggerFactory.getLogger(HttpServletCallbackHandler.class);
    
    public HttpServletCallbackHandler(AuthenticationBindings authBindings,AccessContext context,List<AuthenticationSchemeHandler> authenticationSchemeHandlers){
        super(authBindings,context,authenticationSchemeHandlers);   
    }
    

    /**
     * we fill callbacks not directly related to an AuthenticationSchemeHandler like
     * {@link InetAddressCallback} and {@link LanguageCallback}.
     * @param cbks
     * @throws javax.security.auth.callback.UnsupportedCallbackException
     */
    protected void handleNonSchemeCallbacks(List<Callback> cbks) throws UnsupportedCallbackException {
        HttpServletRequest request = (HttpServletRequest)context.getAttribute(AccessFilter.SERVLET_REQUEST);
        for (Callback cb : cbks) {
            if (cb instanceof InetAddressCallback) {
                String remoteAddress = request.getRemoteAddr();
                String remoteHost = request.getRemoteHost();
                InetAddressCallback inetAddressCallback = (InetAddressCallback) cb;
                inetAddressCallback.setHostAdress(remoteAddress);

                //the server is not configured to return the hostName.
                if (remoteAddress.equals(remoteHost)) {
                    String resolvedHostName = remoteAddress;
                    try {
                        resolvedHostName = reverseDns(remoteAddress);
                    } catch (UnknownHostException uhe) {
                        logger.warn(" host bound to address " + remoteAddress + "cannot be resolved", uhe);
                        throw new UnsupportedCallbackException(cb, uhe.getMessage());
                    } catch (IOException ex) {
                        logger.equals(ex.getMessage());
                        throw new UnsupportedCallbackException(cb, ex.getMessage());
                    }
                    inetAddressCallback.setHostName(resolvedHostName);
                }
            } else if (cb instanceof LanguageCallback) {
                LanguageCallback languageCallback = (LanguageCallback) cb;
                Locale locale = request.getLocale();
                languageCallback.setLocale(locale);
            }
        }
    }
    
    

        /**
         *return the host name related to the IP adress.
         *this method comes from <a href="http://www.oreillynet.com/onjava/blog/2005/11/reverse_dns_lookup_and_java.html">a blog entry about dnsjava</a>.
         *@param hostIp Internet Protocol  adress
         *@return host name related to the hostIp parameter, 
         *or hostIp parameter if no nam eserver is found.
         */
        public static String reverseDns(String hostIp) throws IOException {

                 Resolver res = new ExtendedResolver();
                 Name name = ReverseMap.fromAddress(hostIp);
                 int type = Type.PTR;
                 int dclass = DClass.IN;
                 Record rec = Record.newRecord(name, type, dclass);
                 Message query = Message.newQuery(rec);
                 Message response = res.send(query);

                 Record[] answers = response.getSectionArray(Section.ANSWER);
                 if (answers.length == 0){
                    return hostIp;
                 }else{
                    return answers[0].rdataToString();
                 }
        }

    /**
     * 
     * @return true because client answer in HTTP can be delayed.
     */
    @Override
    protected boolean isAsynchronous() {
        return true;
    }

   
}
