/*
jGuard is a security framework based on top of jaas (java authentication and authorization security).
it is written for web applications, to resolve simply, access control problems.

http://sourceforge.net/projects/jguard/

Copyright (C) 2004  Charles GAY

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


jGuard project home page:
http://sourceforge.net/projects/jguard/

*/
package net.sf.jguard.jee.listeners;

import java.security.Policy;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import javax.security.auth.login.Configuration;
import javax.servlet.ServletContext;
import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;

import net.sf.jguard.core.CoreConstants;
import net.sf.jguard.core.authentication.configuration.JGuardConfiguration;
import net.sf.jguard.core.authorization.policy.AccessControllerUtils;
import net.sf.jguard.core.authorization.policy.MultipleAppPolicy;
import net.sf.jguard.core.authorization.policy.PolicyHelper;
import net.sf.jguard.ext.SecurityConstants;
import net.sf.jguard.ext.authentication.manager.AuthenticationHelper;
import net.sf.jguard.ext.authentication.manager.AuthenticationManager;
import net.sf.jguard.ext.authorization.AuthorizationHelper;
import net.sf.jguard.ext.authorization.manager.AuthorizationManager;
import net.sf.jguard.ext.java5.authentication.jmx.JMXHelper;
import net.sf.jguard.jee.authentication.http.HttpConstants;
import net.sf.jguard.jee.util.WebappUtil;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;



/**
 * handle webapp classloader reference in the
 * JGuardPolicy classloaders repository.
 * @author <a href="mailto:diabolo512@users.sourceforge.net">Charles Gay</a>
 * @author <a href="mailto:vberetti@users.sourceforge.net">Vincent Beretti</a>
 */
public class ContextListener implements ServletContextListener {


	private static final Logger logger = LoggerFactory.getLogger(ContextListener.class);


    /**
	 * method called when the webapp shutdown:
	 * this method unregister the webapp in the JGuardPolicy repository.
	 * @see javax.servlet.ServletContextListener#contextDestroyed(javax.servlet.ServletContextEvent)
	 */
	public void contextDestroyed(ServletContextEvent servletContextEvent) {
		logger.debug(" context destroyed ");
		ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
		if(Policy.getPolicy() instanceof MultipleAppPolicy){
		  MultipleAppPolicy policy = (MultipleAppPolicy) Policy.getPolicy();
          policy.unregisterPermissionProvider(contextClassLoader);
		}

		servletContextEvent.getServletContext().removeAttribute(SecurityConstants.CAPTCHA_SERVICE);
		servletContextEvent.getServletContext().removeAttribute(HttpConstants.USERS_IN_SESSION);
	}

	/**
	 * method called when the webapp start.
	 * install jGuard Configuration and Policy
	 * @see javax.servlet.ServletContextListener#contextInitialized(javax.servlet.ServletContextEvent)
	 */
	public void contextInitialized(ServletContextEvent contextEvent) {
		logger.debug("#####  initializing ContextListener ... #####");
		ServletContext context = contextEvent.getServletContext();
		context.setAttribute(HttpConstants.USERS_IN_SESSION,new ArrayList());

		String webappHomePath = WebappUtil.getWebappHomePath(context,"/");
		String applicationName = context.getServletContextName();

		context.setAttribute(CoreConstants.APPLICATION_NAME,applicationName);

		initAuthentication(context, webappHomePath, applicationName);
		Configuration configuration = (Configuration)context.getAttribute(HttpConstants.JGUARD_CONFIGURATION);
		AccessControllerUtils authUtils = initAuthorization(context, webappHomePath, applicationName);		
		
		//JMX MBeanServer creation
        String enableJMX=(String)context.getInitParameter(SecurityConstants.ENABLE_JMX);
        if(enableJMX != null && "true".equalsIgnoreCase(enableJMX)){
        	Map options = new HashMap();
        	options.put(SecurityConstants.MBEAN_SERVER_FOR_CONNECTOR,context.getInitParameter(SecurityConstants.MBEAN_SERVER_FOR_CONNECTOR));
        	options.put(SecurityConstants.RMI_REGISTRY_HOST,context.getInitParameter(SecurityConstants.RMI_REGISTRY_HOST));
			options.put(SecurityConstants.RMI_REGISTRY_PORT,context.getInitParameter(SecurityConstants.RMI_REGISTRY_PORT));
			options.put(SecurityConstants.MBEAN_SERVER_FOR_CONNECTOR,context.getInitParameter(SecurityConstants.MBEAN_SERVER_FOR_CONNECTOR));
        	JMXHelper.enableJMXSecurity(applicationName,options,configuration,authUtils);
        }

		logger.debug("#####  ContextListener initialized  #####");
	}

	private AccessControllerUtils initAuthorization(ServletContext context, String webappHomePath, String applicationName) {
		// use default authorizationConfigurationLocation unless it is explicitly defined
		String authorizationConfigurationLocation=null;
		if(context.getInitParameter(HttpConstants.AUTHORIZATION_CONFIGURATION_LOCATION)!=null){
			authorizationConfigurationLocation = WebappUtil.getWebappHomePath(context,context.getInitParameter(HttpConstants.AUTHORIZATION_CONFIGURATION_LOCATION));
		}else{
			authorizationConfigurationLocation=WebappUtil.getWebappHomePath(context,HttpConstants.DEFAULT_AUTHORIZATION_CONFIGURATION_LOCATION);
		}
		
	
		Map authorizationSettings = AuthorizationHelper.loadConfiguration(authorizationConfigurationLocation,webappHomePath);
		//handle authorization scope
		String authorizationScope = null;
		if(authorizationSettings.get(SecurityConstants.SCOPE)!=null){
			authorizationScope = (String)authorizationSettings.get(SecurityConstants.SCOPE);
		}else{
			authorizationScope = SecurityConstants.LOCAL_SCOPE;
			context.setAttribute(SecurityConstants.AUTHORIZATION_SCOPE, authorizationScope);
		}
		logger.info(" authorization scope = "+authorizationScope);
		
		
        AuthorizationManager authorizationManager = AuthorizationHelper.initAuthorization(authorizationSettings,applicationName);
        AccessControllerUtils authorizationUtils = null;
        MultipleAppPolicy policy = null;
        if(SecurityConstants.JVM_SCOPE.equalsIgnoreCase(authorizationScope)){
	        PolicyHelper.installPolicyOnJVM(); 
			//Register the new authorization manager with jguard policy provider
			policy = (MultipleAppPolicy)Policy.getPolicy();
			authorizationUtils = new AccessControllerUtils();
        }else{
        	policy = new MultipleAppPolicy();
        	authorizationUtils = new AccessControllerUtils(policy);
        }
        
        policy.registerPermissionProvider(Thread.currentThread().getContextClassLoader(),authorizationManager);
        context.setAttribute(HttpConstants.AUTHZ_UTILS, authorizationUtils);
        context.setAttribute(SecurityConstants.AUTHORIZATION_MANAGER,authorizationManager);
        return authorizationUtils;
	}

	private void initAuthentication(ServletContext context, String webappHomePath, String applicationName) {
		//use default authenticationConfigurationLocation unless it is explicitly defined
		String authenticationConfigurationLocation=null;
		if(context.getInitParameter(HttpConstants.AUTHENTICATION_CONFIGURATION_LOCATION)!=null){
			authenticationConfigurationLocation = WebappUtil.getWebappHomePath(context,context.getInitParameter(HttpConstants.AUTHENTICATION_CONFIGURATION_LOCATION));
		}else{
			authenticationConfigurationLocation=WebappUtil.getWebappHomePath(context,HttpConstants.DEFAULT_AUTHENTICATION_CONFIGURATION_LOCATION);
		}
		
		
		Map authenticationSettings = AuthenticationHelper.loadConfiguration(authenticationConfigurationLocation,webappHomePath);
		//handle authentication scope
		String authenticationScope = null;
		if(authenticationSettings.get(SecurityConstants.SCOPE)!=null){
			authenticationScope = (String)authenticationSettings.get(SecurityConstants.SCOPE);
		}else{
			authenticationScope = SecurityConstants.LOCAL_SCOPE;
			context.setAttribute(SecurityConstants.AUTHENTICATION_SCOPE, authenticationScope);
		}
		logger.info(" authentication scope = "+authenticationScope);

		
		
		JGuardConfiguration jGuardConf = null;
		logger.info(" authentication scope = "+authenticationScope);
		if(SecurityConstants.JVM_SCOPE.equalsIgnoreCase(authenticationScope)){
			jGuardConf = (JGuardConfiguration) Configuration.getConfiguration();
		}else{
			jGuardConf = new JGuardConfiguration();
			context.setAttribute(HttpConstants.JGUARD_CONFIGURATION,jGuardConf);
		}
		
		AuthenticationManager authenticationManager = AuthenticationHelper.initAuthentication(jGuardConf,authenticationSettings,applicationName);
		context.setAttribute(SecurityConstants.AUTHENTICATION_MANAGER,authenticationManager);
	}

	
}
