/*
jGuard is a security framework based on top of jaas (java authentication and authorization security).
it is written for web applications, to resolve simply, access control problems.
version $Name$
http://sourceforge.net/projects/jguard/

Copyright (C) 2004  Charles GAY

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


jGuard project home page:
http://sourceforge.net/projects/jguard/

*/

package net.sf.jguard.jee.authentication.http;

import java.io.IOException;
import java.security.NoSuchAlgorithmException;
import java.security.Permission;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import javax.security.auth.Subject;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import net.sf.jguard.core.CoreConstants;
import net.sf.jguard.core.authentication.AbstractAuthenticationBindings;
import net.sf.jguard.core.authentication.AccessContext;
import net.sf.jguard.core.authentication.AuthenticationBindings;
import net.sf.jguard.core.authentication.credentials.JGuardCredential;
import net.sf.jguard.core.authorization.permissions.URLPermission;
import net.sf.jguard.ext.SecurityConstants;
import net.sf.jguard.ext.authentication.AuthenticationException;
import net.sf.jguard.ext.authentication.manager.AuthenticationManager;
import net.sf.jguard.ext.authentication.manager.AuthenticationManagerFactory;
import net.sf.jguard.ext.registration.SubjectTemplate;
import net.sf.jguard.ext.util.CryptUtils;
import net.sf.jguard.ext.util.XMLUtils;
import net.sf.jguard.jee.authentication.callbacks.HttpServletCallbackHandler;
import net.sf.jguard.jee.authorization.http.HttpAccessControllerUtils;
import net.sf.jguard.jee.authorization.http.HttpPermissionFactory;
import net.sf.jguard.jee.authorization.http.PermissionFactory;

import org.dom4j.Document;
import org.dom4j.Element;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 * @author <a href="mailto:diabolo512@users.sourceforge.net">Charles Gay</a>
 */
public class HttpServletAuthenticationBindings extends AbstractAuthenticationBindings implements AuthenticationBindings{
    
    private static final Logger logger = LoggerFactory.getLogger(HttpServletAuthenticationBindings.class.getName());
    

    private String authScheme = null;
    private URLPermission authenticationFailedPermission = null;
    

    private String indexURI;

    private boolean goToLastAccessDeniedUriOnSuccess;
    private boolean local = false;

    private URLPermission registerProcessPermission;

    private URLPermission registerPermission;
    

    private URLPermission logonProcessPermission;

    private static URLPermission logonPermission;

    private String logonURI;
    private String registerURI;
    private PermissionFactory permissionFactory;

    private URLPermission logoffPermission;
    private URLPermission indexPermission;
    
    /**
     * Creates a new instance of HttpServletAuthenticationBindings
     */
    public HttpServletAuthenticationBindings() {
        permissionFactory = new HttpPermissionFactory();
    }

    public void init(String location,String authenticationScope) {
          Map settings = loadFilterConfiguration(location);
          setSettings(settings);
          
          if(SecurityConstants.JVM_SCOPE.equalsIgnoreCase(authenticationScope)){
               local = false;
          }else{
               local = true;
          }
    }

    
    /**
	 * @param filterSettings Map which contains filter options
	 * @throws ServletException
	 */
	private void setSettings(Map settings) {
        indexURI = (String)settings.get(HttpConstants.INDEX_URI);
        indexPermission = new URLPermission("indexURI",indexURI);
        authenticationFailedPermission = new URLPermission("authenticationFailedURI",(String)settings.get(HttpConstants.AUTHENTICATION_FAILED_URI));
        logonProcessPermission = new URLPermission("logonProcessURI",(String)settings.get(HttpConstants.LOGON_PROCESS_URI));
        String logonURIStr = (String)settings.get(HttpConstants.LOGON_URI);
        logonPermission = new URLPermission("logonURI",logonURIStr);
    
        logoffPermission = new URLPermission("logoffURI",(String)settings.get(HttpConstants.LOGOFF_URI));
        
        registerProcessPermission = new URLPermission(HttpConstants.REGISTER_PROCESS_URI,(String)settings.get(HttpConstants.REGISTER_PROCESS_URI));
        registerURI =(String)settings.get(HttpConstants.REGISTER_URI);
        registerPermission = new URLPermission(HttpConstants.REGISTER_URI,registerURI);
        //grab the authentication scheme authorized
        authScheme = (String)settings.get(HttpConstants.AUTH_SCHEME);
        HttpServletCallbackHandler.setAuthScheme(authScheme);
        Collection authSchemes = AuthSchemesHelper.validateAuthScheme(authScheme);
        if(authSchemes.contains(HttpConstants.FORM_AUTH)){
        	HttpServletCallbackHandler.setLoginField((String)settings.get(HttpConstants.LOGIN_FIELD));
        	HttpServletCallbackHandler.setPasswordField((String)settings.get(HttpConstants.PASSWORD_FIELD));
        }
        
        // grab goToLastAccessDeniedUriOnSuccess value
        String tmpValue = (String)settings.get(HttpConstants.GO_TO_LAST_ACCESS_DENIED_URI_ON_SUCCESS);
        if("false".equalsIgnoreCase(tmpValue) || "no".equalsIgnoreCase(tmpValue))
        	goToLastAccessDeniedUriOnSuccess = false;
        
	}
    public void logonProcess(AccessContext context) {
        HttpServletRequest request = (HttpServletRequest)context.getParameter(AccessFilter.SERVLET_REQUEST);
        HttpServletResponse response = (HttpServletResponse)context.getParameter(AccessFilter.SERVLET_RESPONSE);
        HttpSession session = request.getSession(true);
		Subject subject = HttpAuthenticationUtils.getSubject(session);
		if(HttpConstants.BASIC_AUTH.equalsIgnoreCase(authScheme)){
			   String authorizationHeader = request.getHeader("Authorization");
			   logger.debug("authorizationHeader="+authorizationHeader);

			    //first time user reach the webapp in the session => we automatically authenticate it as a GUEST
			    if(subject == null){
			    	String AuthorizationHeaderValue = HttpServletCallbackHandler.buildBasicAuthHeader(
			    			                          SecurityConstants.GUEST,SecurityConstants.GUEST,request.getCharacterEncoding());
			    	JGuardServletRequestWrapper jguardrequest = new JGuardServletRequestWrapper(request);
			    	jguardrequest.setHeader(HttpServletCallbackHandler.AUTHORIZATION,AuthorizationHeaderValue);
			    	logger.debug(" BASIC authentication subject is null  authenticate phase ");
			    	authenticate(context);
			    }else if((authorizationHeader==null || !authorizationHeader.startsWith("Basic ") )){
		            //in BASIC AUTH, we don't redirect to a login FORM page
		    		//but put in the response header the challenge
			    	logger.debug(" subject is not null but BASIC HEADER is incorrect "+authorizationHeader," jGuard build BASIC challenge  ");
		    		HttpServletCallbackHandler.buildBasicChallenge(response,(String)session.getServletContext().getAttribute(CoreConstants.APPLICATION_NAME));
                            }else{
				logger.debug(" BASIC AUTHENTICATION TYPE   authenticate phase  ");
                                authenticate(context);
                            }
		}else{
			logger.debug(" AUTHENTICATION TYPE ="+authScheme+"  authenticate phase  ");
                        authenticate(context);
		}
    }

    public void logoff(AccessContext context) {
                HttpServletRequest request = (HttpServletRequest)context.getParameter(AccessFilter.SERVLET_REQUEST);
            
                //remove Subject from session
		HttpSession session = request.getSession();

		HttpAuthenticationUtils auth= (HttpAuthenticationUtils)session.getAttribute(HttpConstants.AUTHN_UTILS);
		if(auth!= null){
			auth.logout();
			logger.debug(" user logoff ");
		}

		session.removeAttribute(HttpConstants.AUTHN_UTILS);

		if (logger.isDebugEnabled()) {
				logger.debug("doFilter() -  user logoff ");
		}

        //we invalidate the session to unbound all objects, including subject
	    try{
           session.invalidate();
	    }catch(IllegalStateException ise){
	    	logger.debug(" session is already invalidated ");
	    }
    }

    public boolean authenticate(AccessContext context) {
        HttpServletRequest request = (HttpServletRequest)context.getParameter(AccessFilter.SERVLET_REQUEST);
        HttpServletResponse response = (HttpServletResponse)context.getParameter(AccessFilter.SERVLET_RESPONSE);
        boolean authenticationFailed;
        try {
            authenticationFailed = !HttpAuthenticationUtils.authenticate(request, response, false, local);
        } catch (IOException ex) {
            logger.error(ex.getMessage());
            return false;
        }
	return postAuthenticationProcess(context, authenticationFailed);
    }

    /**
     * decorate the HttpServletRequest with a HttpServletRequestWrapper
     * ({@link JGuardServletRequestWrapper}).
     */
    public void process(AccessContext context) {
        HttpServletRequest request = (HttpServletRequest)context.getParameter(AccessFilter.SERVLET_REQUEST);
        context.putParameter(AccessFilter.SERVLET_REQUEST, new JGuardServletRequestWrapper(request));
    }

    public void accessDenied(AccessContext context){
                HttpServletRequest request = (HttpServletRequest)context.getParameter(AccessFilter.SERVLET_REQUEST);
                HttpServletResponse response = (HttpServletResponse)context.getParameter(AccessFilter.SERVLET_RESPONSE);
                //redirect to accessDeniedURI because user is not authorized
		//to access to this resource
                if (logger.isDebugEnabled()) {
		    logger.debug(" access denied to "+request.getRequestURI());
		}
		
                logger.debug(" access is denied to "+request.getRequestURI()+" accessDeniedURI is not defined  jGuard send 401 http code ");
                response.setStatus(HttpServletResponse.SC_UNAUTHORIZED);
        try {
            response.sendError(HttpServletResponse.SC_UNAUTHORIZED,"access is denied to "+request.getRequestURI());
        } catch (IOException ex) {
            try {
                response.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
            } catch (IOException ex2) {
                throw new RuntimeException(ex2.getMessage());
            }
        }
    }

    public void redirectToLogon(AccessContext context) {
        HttpServletRequest request = (HttpServletRequest)context.getParameter(AccessFilter.SERVLET_REQUEST);
        HttpServletResponse response = (HttpServletResponse)context.getParameter(AccessFilter.SERVLET_RESPONSE);
        try {
            response.sendRedirect(response.encodeRedirectURL(request.getContextPath()+logonURI));
        } catch (IOException ex) {
            throw new RuntimeException(ex.getMessage());
        }
    }

    public Subject getSubject(AccessContext context) {
        HttpServletRequest request = (HttpServletRequest)context.getParameter(AccessFilter.SERVLET_REQUEST);
        HttpSession session = request.getSession(true);
        Subject subject = HttpAuthenticationUtils.getSubject(session);
        return subject;
    }

    public Permission getPermissionRequested(AccessContext context) {
        HttpServletRequest request = (HttpServletRequest)context.getParameter(AccessFilter.SERVLET_REQUEST);
        return permissionFactory.getPermission(request);
    }

    public void setLastAccessDeniedPermission(AccessContext context, Permission permission) {
        if(permission instanceof  URLPermission){
            URLPermission urlPermission = (URLPermission)permission;
            HttpServletRequest request = (HttpServletRequest)context.getParameter(AccessFilter.SERVLET_REQUEST);
            HttpSession session = request.getSession(true);
            session.setAttribute(HttpConstants.LAST_ACCESS_DENIED_URI,urlPermission.getURI());
        }
    }

    public Permission getAuthenticationFailedPermission() {
        return authenticationFailedPermission;
    }

    public Permission getLogonPermission() {
        return logonPermission;
    }

    public Permission getLogonProcessPermission() {
        return logonProcessPermission;
    }
    
    public Permission getIndexPermission() {
        return indexPermission;
    }

    public Permission getLogoffPermission() {
        return logoffPermission;
    }

    public Permission getRegisterProcessPermission(){
    	return registerProcessPermission;
    }
    
    public Permission getRegisterPermission(){
    	return registerPermission;
    }
    private boolean postAuthenticationProcess(AccessContext context, boolean authenticationFailed) {
        HttpServletRequest request = (HttpServletRequest)context.getParameter(AccessFilter.SERVLET_REQUEST);
        HttpServletResponse response = (HttpServletResponse)context.getParameter(AccessFilter.SERVLET_RESPONSE);
        Subject subject = HttpAuthenticationUtils.getSubject(request.getSession(true));
		
		if(authenticationFailed){
			if(response.isCommitted()){
				logger.warn(" response is already committed ");
			}

			if(HttpConstants.BASIC_AUTH.equalsIgnoreCase(authScheme)){
				  HttpServletCallbackHandler.buildBasicChallenge(response,(String)((HttpServletRequest)request).getSession(true).getServletContext().getAttribute(CoreConstants.APPLICATION_NAME));
				  return !authenticationFailed;
			 }
			logger.debug("authentication failed redirect to "+authenticationFailedPermission.getURI());

		   if(authenticationFailedPermission!=null && !authenticationFailedPermission.getURI().equals("")){
                try {
                    response.sendRedirect(response.encodeRedirectURL(request.getContextPath()+authenticationFailedPermission.getURI()));
                } catch (IOException ex) {
                   logger.error(ex.getMessage());
                   return false;
                }
                   logger.debug(" NOT BASIC AUTHENTICATION - user is not authenticated  redirect to "+request.getContextPath()+authenticationFailedPermission.getURI());
			   
		   }else{
                           accessDenied(context);
		   }

		//authentication succeed
		}else{
			HttpSession oldSession = request.getSession(true);
		    String redirectURI = indexURI;
		    String lastAccessDeniedURI = (String)oldSession.getAttribute(HttpConstants.LAST_ACCESS_DENIED_URI);
		    
		    //for security reasons(man-in-the-middle attack for sessionID cookie),
		    //we remove the old session if authentication is successfull
		    //and create a new session
		    //to not lost the subject, we grab in the old session HttpAuthenticationUtils 
		    //and put it in the new session
		    HttpAuthenticationUtils httpAuthenticationUtils = HttpAuthenticationUtils.getHttpAuthenticationUtils(request,local);
		    subject = httpAuthenticationUtils.getSubject();
		    
		    //we test if response is committed, because
		    //when user register, it authenticate twice
		    // => the response is committed at the first step
		    if(!response.isCommitted()){
                            //we remove this variable before invalidate the session 
			    //to prevent the session listener to erase the subject 
			    oldSession.removeAttribute(HttpConstants.AUTHN_UTILS);
			    oldSession.invalidate();
			    
			    HttpSession newSession = request.getSession(true);
			    newSession.setAttribute(HttpConstants.AUTHN_UTILS,httpAuthenticationUtils);
		    }
		    
		    //we redirect to the last 'access denied' URI before authentication
		    if(lastAccessDeniedURI!=null && lastAccessDeniedURI!="") {
		    	if(goToLastAccessDeniedUriOnSuccess)
		    		redirectURI = lastAccessDeniedURI;
		    	else {
				   if(!HttpAccessControllerUtils.hasPermission(request,indexPermission)){
                           redirectURI = logonURI;
                   }

		    	}
		    }
	

		    logger.debug(" user is authenticated "," redirect to "+redirectURI);
		    if(!response.isCommitted()){
                        try {
                            response.sendRedirect(response.encodeRedirectURL(request.getContextPath()+redirectURI));
                        } catch (IOException ex) {
                           logger.error(ex.getMessage());
                           return false;
                        }
		    }
		    
		}
		return !authenticationFailed;
    }
    
       /** load configuration from an XML file.
        * @param configurationLocation
        * @return Map containing filter configuration
        */
	private Map loadFilterConfiguration(String configurationLocation){
		Document doc = XMLUtils.read(configurationLocation);

		Element authentication = doc.getRootElement().element(HttpConstants.FILTER);
		Map filterSettings = new HashMap();
		filterSettings.put(HttpConstants.INDEX_URI,authentication.element(HttpConstants.INDEX_URI).getTextTrim());
		filterSettings.put(HttpConstants.AUTHENTICATION_FAILED_URI,authentication.element(HttpConstants.AUTHENTICATION_FAILED_URI).getTextTrim());
		if(authentication.element(HttpConstants.REGISTER_PROCESS_URI)!=null){
			filterSettings.put(HttpConstants.REGISTER_PROCESS_URI,authentication.element(HttpConstants.REGISTER_PROCESS_URI).getTextTrim());
		}
		if(authentication.element(HttpConstants.REGISTER_URI)!=null){
			filterSettings.put(HttpConstants.REGISTER_URI,authentication.element(HttpConstants.REGISTER_URI).getTextTrim());
		}
		filterSettings.put(HttpConstants.LOGON_PROCESS_URI,authentication.element(HttpConstants.LOGON_PROCESS_URI).getTextTrim());
		filterSettings.put(HttpConstants.LOGON_URI,authentication.element(HttpConstants.LOGON_URI).getTextTrim());

		filterSettings.put(HttpConstants.LOGOFF_URI,authentication.element(HttpConstants.LOGOFF_URI).getTextTrim());
                

		filterSettings.put(HttpConstants.AUTH_SCHEME,authentication.element(HttpConstants.AUTH_SCHEME).getTextTrim());
		Element loginElement = authentication.element(HttpConstants.LOGIN_FIELD);
		if (loginElement != null){
			filterSettings.put(HttpConstants.LOGIN_FIELD,loginElement.getTextTrim());
		}
		Element passwordElement = authentication.element(HttpConstants.PASSWORD_FIELD);
		if (passwordElement != null){
			filterSettings.put(HttpConstants.PASSWORD_FIELD,passwordElement.getTextTrim());
		}

		Element goToLastAccessDeniedUriOnSuccessElement = authentication.element(HttpConstants.GO_TO_LAST_ACCESS_DENIED_URI_ON_SUCCESS);
		if (goToLastAccessDeniedUriOnSuccessElement != null){
			filterSettings.put(HttpConstants.GO_TO_LAST_ACCESS_DENIED_URI_ON_SUCCESS,goToLastAccessDeniedUriOnSuccessElement.getTextTrim());
		}
		
		return filterSettings;
	}

	public AccessContext anonymize(AccessContext context) {
		AccessContext accContext;
		accContext = (AccessContext)context.clone();
		HttpServletRequest request = (HttpServletRequest)context.getParameter(AccessFilter.SERVLET_REQUEST);
		accContext.putParameter(AccessFilter.SERVLET_REQUEST, new AnonymizerRequestWrapper(request));
		return accContext;
	}

	/**
	 * register the user against the @link SubjectTemplate.
	 * @param context 
	 * @return true if registration succeed, false otherwise
	 */
	public boolean registerCoreProcess(AccessContext context) {
		HttpServletRequest request = (HttpServletRequest)context.getParameter(AccessFilter.SERVLET_REQUEST);
		boolean success = false;
		AuthenticationManager auth = AuthenticationManagerFactory.getAuthenticationManager();
		SubjectTemplate st = null;
		try {
			st = buildSubjectTemplate(request);
		} catch (AuthenticationException e1) {
			logger.error(" subject template cannot be built ",e1);
            success = false;
			return false;
		}
                String passwordField = HttpServletCallbackHandler.getPasswordField();
                Set credentials = st.getRequiredCredentials();
                Iterator itCreds = credentials.iterator();
                boolean found = false;
                JGuardCredential passwordCredential = null;
                while(itCreds.hasNext() && found==false){
                    JGuardCredential cred = (JGuardCredential) itCreds.next();
                    if(cred.getId().equals(passwordField)){
                        passwordCredential = cred;
                        found = true;
                    }
                    
                }
                
                if(!found){
                    logger.warn("JGuardCredential matching  passwordField not found in the SubjectTemplate");
                    success = false;
                    return success;
                }
                char[] password = ((String)passwordCredential.getValue()).toCharArray();
                
                try {
                    passwordCredential.setValue(new String(CryptUtils.cryptPassword(password)));
                } catch (NoSuchAlgorithmException ex) {
                    logger.warn(ex.getMessage());
                    success = false;
                    return success;
                }
                
		try {
			auth.createUser(st);
			success = true;
		} catch (AuthenticationException e) {
            logger.debug(" registrationProcess failed ");
            success = false;
		}

		return success;
	}
	

	/**
	 * fill in the SubjectTemplate the credentials from HttpServletRequest.
	 * @param req HttpServletRequest
	 * @return SubjectTemplate filled.
	 * @throws AuthenticationException
	 */
	private static SubjectTemplate buildSubjectTemplate(HttpServletRequest req) throws AuthenticationException{
		AuthenticationManager auth = AuthenticationManagerFactory.getAuthenticationManager();
		SubjectTemplate defaultSt = auth.getDefaultSubjectTemplate();
		SubjectTemplate st = new SubjectTemplate();
                st.setName(defaultSt.getName());

                //private required credentials
		Set privateCredRequiredFromDefaultSt =defaultSt.getPrivateRequiredCredentials();
		Set privRequiredCred = grabRegistrationForm(req, st, privateCredRequiredFromDefaultSt);
		st.setPrivateRequiredCredentials(privRequiredCred);

		//public required credentials
		Set publicCredRequiredFromDefaultSt =defaultSt.getPublicRequiredCredentials();
		Set pubRequiredCred = grabRegistrationForm(req, st, publicCredRequiredFromDefaultSt);
		st.setPublicRequiredCredentials(pubRequiredCred);

                //public optional credentials
		Set publicCredOptionalFromDefaultSt =defaultSt.getPublicOptionalCredentials();
		Set pubOptionalCred = grabRegistrationForm(req, st, publicCredOptionalFromDefaultSt);
		st.setPublicOptionalCredentials(pubOptionalCred);

                //public optional credentials
		Set privateCredOptionalFromDefaultSt =defaultSt.getPrivateOptionalCredentials();
		Set privOptionalCred = grabRegistrationForm(req, st, privateCredOptionalFromDefaultSt);
		st.setPrivateOptionalCredentials(privOptionalCred);


		return st;
	}
	
	/**
	 * build a set of credentials by grabbing data from HttpServletRequest.
	 * @param req HttpServletRequest
	 * @param st SubjectTemplate
	 * @param credentialsFromDefaultSt
	 * @return Set of {@link JGuardCredential}
	 */
	private static Set grabRegistrationForm(HttpServletRequest req, SubjectTemplate st, Set credentialsFromDefaultSt) {
		Iterator itCredentials = credentialsFromDefaultSt.iterator();
		Set credSet = new HashSet();
		while(itCredentials.hasNext()){
			JGuardCredential jcredFromDefault = (JGuardCredential)itCredentials.next();

			//test if we've found the credential in the http request
			if(req.getParameter(jcredFromDefault.getId())!= null){
				JGuardCredential jcred = new JGuardCredential();
				jcred.setId(jcredFromDefault.getId());
				try{
     				jcred.setValue(req.getParameter(jcredFromDefault.getId()));
	    			credSet.add(jcred);
				}catch(IllegalArgumentException iae){
					logger.warn(" the property "+jcredFromDefault.getId()+" doesn't exist in the HttpServletRequest ");
					continue;
				}
			}

		}
		return credSet;
	}

	public void registerProcess(AccessContext context) {
		boolean registerSucceed = registerCoreProcess(context);
		HttpServletRequest request = (HttpServletRequest)context.getParameter(AccessFilter.SERVLET_REQUEST);
		HttpServletResponse response = (HttpServletResponse)context.getParameter(AccessFilter.SERVLET_RESPONSE);
		
		 if(!registerSucceed){
         	logger.debug(" registration failed " ," registerProcess phase ");
         	if(!response.isCommitted()){
         		try {
					response.sendRedirect(response.encodeRedirectURL(request.getContextPath()+registerURI));
				} catch (IOException e) {
					logger.warn(" we cannot redirect to "+request.getContextPath()+registerURI+" because "+e.getMessage());
				}
         	}else{
         		logger.warn(" we cannot redirect to "+request.getContextPath()+registerURI+" because response is already commited ");
         	}
         }else if(registerSucceed){
            	logger.debug(" registration succeed " ," registerProcess phase ");
            	//the user is registered and we submit directly its credentials to the authentication phase
            	request.getSession(true).removeAttribute(HttpConstants.AUTHN_UTILS);
            	request.getSession(true).removeAttribute(HttpConstants.LAST_ACCESS_DENIED_URI);
            	authenticateAfterRegistration(context);
         } 
	}

	/**
	 * authenticate client request according the authentication Policy defined in the 'authSchemes' field.
	 * @param req
	 * @param res
	 * @return <code>true</code> if authentication succeeds, <code>false</code> otherwise
	 * @throws IOException
	 */
	private boolean authenticateAfterRegistration(AccessContext context){
        boolean authenticationFailed;
        HttpServletRequest request = (HttpServletRequest)context.getParameter(AccessFilter.SERVLET_REQUEST);
		HttpServletResponse response = (HttpServletResponse)context.getParameter(AccessFilter.SERVLET_RESPONSE);
		try {
			authenticationFailed = !HttpAuthenticationUtils.authenticate(request,response,true,local);
			return postAuthenticationProcess(context,authenticationFailed);
		} catch (IOException e) {
			logger.warn(e.getMessage());
			return false;
		}
	}
	

}
