/*
jGuard is a security framework based on top of jaas (java authentication and authorization security).
it is written for web applications, to resolve simply, access control problems.
version $Name$
http://sourceforge.net/projects/jguard/

Copyright (C) 2004  Charles GAY

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


jGuard project home page:
http://sourceforge.net/projects/jguard/

*/
package net.sf.jguard.jee.authentication.http;


import java.io.IOException;
import java.util.Collection;

import javax.security.auth.Subject;
import javax.security.auth.login.Configuration;
import javax.security.auth.login.LoginException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import javax.servlet.http.HttpSessionActivationListener;
import javax.servlet.http.HttpSessionBindingEvent;
import javax.servlet.http.HttpSessionBindingListener;
import javax.servlet.http.HttpSessionEvent;

import net.sf.jguard.core.CoreConstants;
import net.sf.jguard.ext.authentication.manager.AuthenticationUtils;
import net.sf.jguard.jee.authentication.callbacks.HttpServletCallbackHandler;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


 

/**
 * Authentication utility class stored on the user's session.
 * @author <a href="mailto:diabolo512@users.sourceforge.net">Charles Gay</a>
 * this class was inspired by the article on jaas published at <a href="http://www.mooreds.com/jaas.html">this</a> address.
 *
 */
public class HttpAuthenticationUtils implements HttpSessionActivationListener,HttpSessionBindingListener{
	private static final Logger logger = LoggerFactory.getLogger(HttpAuthenticationUtils.class);
    
    
    private AuthenticationUtils authenticationUtils = null;

    public HttpAuthenticationUtils(){
    	super();
    	authenticationUtils = new AuthenticationUtils();
    }
    
    public HttpAuthenticationUtils(Configuration configuration){
    	super();
    	authenticationUtils = new AuthenticationUtils(configuration);
    }
    
    
    /**
     * encapsulate JAAS login lifecycle.
     * @param request
     * @param response
     * @param afterRegistration <code>true</code> if user registration was made just before; <code>false</code> otherwise.
     * @throws LoginException
     */
     private  void useLoginContext(HttpServletRequest request,HttpServletResponse response,boolean afterRegistration) throws LoginException {
    	HttpSession session = request.getSession(true);

    	String applicationName = (String)session.getServletContext().getAttribute(CoreConstants.APPLICATION_NAME);
        HttpServletCallbackHandler cbh = new HttpServletCallbackHandler(request,response);
        cbh.setAfterRegistration(afterRegistration);
        authenticationUtils.login(applicationName, cbh,request.getLocale());
        
      }



    /**
     * retrieve the subject from <strong>this</strong> object.
     * @return authenticated Subject, otherwise <strong>null</strong>.
     */
    public Subject getSubject(){
    	return authenticationUtils.getSubject();
    }

    

    /**
     * grab the @link HttpAuthenticationUtils from the HttpServletRequest.
     * @param req
     * @return HttpAuthenticationUtils
     */
    public static HttpAuthenticationUtils getHttpAuthenticationUtils(HttpServletRequest req,boolean local){
		HttpSession session = req.getSession(true);
		HttpAuthenticationUtils httpAuthenticationUtils = (HttpAuthenticationUtils) session.getAttribute(HttpConstants.AUTHN_UTILS);

		//auth.getSubject() can return  null if the user in session has been deleted by authenticationManager
		if(httpAuthenticationUtils!= null && httpAuthenticationUtils.getSubject()==null){
			logger.debug(" subject into HttpAuthenticationUtils is null ");
			httpAuthenticationUtils.logout();
			session.removeAttribute(HttpConstants.AUTHN_UTILS);
			httpAuthenticationUtils = null;
		}

		if(httpAuthenticationUtils==null){
			Configuration configuration = (Configuration)req.getSession().getServletContext().getAttribute(HttpConstants.JGUARD_CONFIGURATION);
			if(local){
			httpAuthenticationUtils = new HttpAuthenticationUtils(configuration);
			}else{
				httpAuthenticationUtils = new HttpAuthenticationUtils();
			}
			session.setAttribute(HttpConstants.AUTHN_UTILS,httpAuthenticationUtils);
		}
    	return httpAuthenticationUtils;
    }

    /**
     * authenticate user and put the corresponding Subject in its session if succeed.
     * @param req
     * @param res
     * @param afterRegistration :<code>true</code> if user registration was made just before;
     * <code>false</code> otherwise.
     * @return authenticationresult: <strong>true</strong> if authentication succeed,
     * <strong>false</strong> otherwise.
     * @throws IOException
     */
	public static boolean authenticate(HttpServletRequest req, HttpServletResponse res,boolean afterRegistration,boolean local) throws IOException {

        HttpAuthenticationUtils auth = getHttpAuthenticationUtils(req,local);
        HttpSession session = req.getSession(true);
		boolean authenticationSucceed = true;
		try {
			auth.useLoginContext(req, res,afterRegistration);
		} catch (LoginException e) {
			authenticationSucceed =  false;
			String messageError = null;
			messageError = e.getLocalizedMessage();
			
			//we store in the user' session the reason the authentication failed
			session.setAttribute(HttpConstants.LOGIN_EXCEPTION_MESSAGE,messageError);
			session.setAttribute(HttpConstants.LOGIN_EXCEPTION_CLASS,e.getClass());
		}

		return authenticationSucceed;
	}



	/**
	 * method called by the container when session is serialized.
	 * @param sessionEvent
	 */
	public void sessionWillPassivate(HttpSessionEvent sessionEvent) {
		if(authenticationUtils!=null){
			authenticationUtils.logout();
		}
		HttpAuthenticationUtils authUtils = (HttpAuthenticationUtils)sessionEvent.getSession().getAttribute(HttpConstants.AUTHN_UTILS);
		if(authUtils!=null){
			authUtils.logout();
		}
		sessionEvent.getSession().removeAttribute(HttpConstants.AUTHN_UTILS);
	}


	/**
	 * method called by container when session is deserialized.
	 * @param sessionEvent
	 */
	public void sessionDidActivate(HttpSessionEvent sessionEvent) {


	}


	public void valueBound(HttpSessionBindingEvent bindingEvent) {
		if(HttpConstants.AUTHN_UTILS.equals(bindingEvent.getName())){
			Collection users = (Collection)bindingEvent.getSession().getServletContext().getAttribute(HttpConstants.USERS_IN_SESSION);
			users.add(this);
		}
		
	}



	public void valueUnbound(HttpSessionBindingEvent bindingEvent) {
		if(HttpConstants.AUTHN_UTILS.equals(bindingEvent.getName())){
			Collection users = (Collection)bindingEvent.getSession().getServletContext().getAttribute(HttpConstants.USERS_IN_SESSION);
			if(users!=null && users.contains(this)){
				users.remove(this);
			}
		}
	}

   

	public void logout() {
		authenticationUtils.logout();
	}

	public AuthenticationUtils getAuthenticationUtils() {
		return authenticationUtils;
	}

	/**
	 * return the Subject from the HttpSession, or null if no Subject is present.
	 * @param session
	 * @return
	 */
	public static Subject getSubject(HttpSession session){
		HttpAuthenticationUtils authutils = (HttpAuthenticationUtils)session.getAttribute(HttpConstants.AUTHN_UTILS);
		if(authutils!=null){
			return authutils.getSubject();
		}
		return null;
	}

	
}