/*
jGuard is a security framework based on top of jaas (java authentication and authorization security).
it is written for web applications, to resolve simply, access control problems.
version $Name$
http://sourceforge.net/projects/jguard/

Copyright (C) 2004  Charles GAY

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


jGuard project home page:
http://sourceforge.net/projects/jguard/

*/
package net.sf.jguard.jee.authentication.http;

import java.io.IOException;
import java.security.NoSuchAlgorithmException;
import java.security.Permissions;
import java.security.Policy;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import javax.security.auth.Subject;
import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import net.sf.jguard.core.CoreConstants;
import net.sf.jguard.core.authentication.credentials.JGuardCredential;
import net.sf.jguard.core.authorization.permissions.URLPermission;
import net.sf.jguard.ext.SecurityConstants;
import net.sf.jguard.ext.authentication.AuthenticationException;
import net.sf.jguard.ext.authentication.manager.AuthenticationManager;
import net.sf.jguard.ext.authentication.manager.AuthenticationManagerFactory;
import net.sf.jguard.ext.registration.SubjectTemplate;
import net.sf.jguard.ext.util.CryptUtils;
import net.sf.jguard.ext.util.XMLUtils;
import net.sf.jguard.jee.authentication.callbacks.HttpServletCallbackHandler;
import net.sf.jguard.jee.authorization.http.HttpAccessControllerUtils;
import net.sf.jguard.jee.authorization.http.HttpPermissionFactory;
import net.sf.jguard.jee.authorization.http.PermissionFactory;
import net.sf.jguard.jee.util.WebappUtil;

import org.dom4j.Document;
import org.dom4j.Element;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

 

/**
 * this Filter handle the control access of all request.
 * @author <a href="mailto:tandilero@users.sourceforge.net">Maximiliano Batelli</a>
 * @author <a href="mailto:diabolo512@users.sourceforge.net">Charles Gay</a>
 */
public class AccessFilterOld implements Filter,HttpConstants{


    static public final Logger logger = LoggerFactory.getLogger(AccessFilterOld.class);

    /*   authentication settings     */
    //page to redirect if authentication success
    private static String indexURI;
    //page to redirect if authentication failed
    private static URLPermission authenticationFailedURI;
    //ressource to access to logonForm
    private static URLPermission logonURI;
    private static String logonURIStr;
    //destination from authentication form to authenticate
    private static URLPermission logonProcessURI;
    //uris to logoff
    private static Permissions logoffURIs;
    
    //uri used to grab informations from
    //registration form
    private static URLPermission registerProcessURI;
    //uri used to grab informations from
    //registration form
    private static String registerURI;

    //unique name of the application
    private static String applicationName;


    //authentication schemes authorized
	private static String authScheme;
    
    private static boolean goToLastAccessDeniedUriOnSuccess = true;
    
    private PermissionFactory permissionFactory = null;
    
    //very important parameter which define if configuration is local to the webapp or not
    private boolean local;
    
    
    
	/**
         * Filter initialization.
	 * @see javax.servlet.Filter#init(javax.servlet.FilterConfig)
	 */
	public void init(FilterConfig filterCfg) throws ServletException {
		
	logger.debug("server info = "+filterCfg.getServletContext().getServerInfo());
    	logger.debug("servletContextName="+ filterCfg.getServletContext().getServletContextName());
    	logger.debug("servlet Real Path="+ WebappUtil.getWebappHomePath(filterCfg.getServletContext(),"/"));
    	logger.debug("current Policy="+Policy.getPolicy());

        ServletContext context = filterCfg.getServletContext();
    	applicationName = context.getServletContextName();
    	if(applicationName == null){
    		logger.error(" ServletContext.getServletContextName() return null \n you should fix your web.xml by adding the 'display-name' markup with the name of your webapp ");
    		throw new ServletException(" ServletContext.getServletContextName() return null \n you should fix your web.xml by adding the 'display-name' markup with the name of your webapp ");
    	}

    	String filterConfigurationLocation = WebappUtil.getWebappHomePath(context,filterCfg.getInitParameter(HttpConstants.CONFIGURATION_LOCATION));

	    Map filterSettings = loadFilterConfiguration(filterConfigurationLocation);

	    setFilterSettings(filterSettings);

	    context.setAttribute(CoreConstants.APPLICATION_NAME,applicationName);
	    context.setAttribute(HttpConstants.AUTH_SCHEME,authScheme);
	    context.setAttribute(HttpConstants.USERS_IN_SESSION,new ArrayList());
	    String authenticationScope = (String)context.getAttribute(SecurityConstants.AUTHENTICATION_SCOPE);
	    if(SecurityConstants.JVM_SCOPE.equalsIgnoreCase(authenticationScope)){
	    	local = false;
	    }else{
	    	local = true;
	    }
            
            permissionFactory = new HttpPermissionFactory();
	}

	/**
	 * @param filterSettings Map which contains filter options
	 * @throws ServletException
	 */
	private void setFilterSettings(Map filterSettings) throws ServletException {
        indexURI = (String)filterSettings.get(HttpConstants.INDEX_URI);
        authenticationFailedURI = new URLPermission("authenticationFailedURI",(String)filterSettings.get(HttpConstants.AUTHENTICATION_FAILED_URI));
        logonProcessURI = new URLPermission("logonProcessURI",(String)filterSettings.get(HttpConstants.LOGON_PROCESS_URI));
        logonURIStr = (String)filterSettings.get(HttpConstants.LOGON_URI);
        logonURI = new URLPermission("logonURI",logonURIStr);
        logoffURIs = new Permissions();
        Iterator itLogoffURI = ((Set)filterSettings.get(HttpConstants.LOGOFF_URIS)).iterator();
        while (itLogoffURI.hasNext()){
            logoffURIs.add(new URLPermission("logoffURI",(String)itLogoffURI.next()));
        }
        registerProcessURI = new URLPermission("registerProcessURI",(String)filterSettings.get(HttpConstants.REGISTER_PROCESS_URI));
        registerURI = (String)filterSettings.get(HttpConstants.REGISTER_URI);

        //grab the authentication scheme authorized
        authScheme = (String)filterSettings.get(HttpConstants.AUTH_SCHEME);
        Collection authSchemes = AuthSchemesHelper.validateAuthScheme(authScheme);
        if(authSchemes.contains(HttpConstants.FORM_AUTH)){
        	HttpServletCallbackHandler.setLoginField((String)filterSettings.get(HttpConstants.LOGIN_FIELD));
        	HttpServletCallbackHandler.setPasswordField((String)filterSettings.get(HttpConstants.PASSWORD_FIELD));
        }
        
        // grab goToLastAccessDeniedUriOnSuccess value
        String tmpValue = (String)filterSettings.get(HttpConstants.GO_TO_LAST_ACCESS_DENIED_URI_ON_SUCCESS);
        if("false".equalsIgnoreCase(tmpValue) || "no".equalsIgnoreCase(tmpValue))
        	goToLastAccessDeniedUriOnSuccess = false;
        
	}


    	/**
         * the method which handle the request.
         * this method was inspired by the article on jaas published at <a href="http://www.mooreds.com/jaas.html">this</a> address.
	 * @see javax.servlet.Filter#doFilter(javax.servlet.ServletRequest, javax.servlet.ServletResponse, javax.servlet.FilterChain)
	 */
	public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        HttpServletRequest req = (HttpServletRequest)request;
        HttpServletResponse res = (HttpServletResponse)response;
        URLPermission urlPermission = (URLPermission) permissionFactory.getPermission(req);
        Subject subject = HttpAuthenticationUtils.getSubject(req.getSession(true));
       
        if(subject == null){
        	logger.debug("LAST_ACCESS_DENIED_URI="+urlPermission.getURI());
        	req.getSession(true).setAttribute(HttpConstants.LAST_ACCESS_DENIED_URI,urlPermission.getURI());
        	logger.debug( " subject is null  logonProcess phase ");
        	logonProcess(new AnonymizerRequestWrapper(req), res);
        }else if(authenticationFailedURI.implies(urlPermission)){
                logger.debug(" access authorized ");
                chain.doFilter(request,response);




        }else if(logonURI.implies(urlPermission)){
        	logger.debug(" logon phase ");
                
                //we build the FORM challenge only with chain.doFilter()
        	HttpServletCallbackHandler.buildFormChallenge(chain, req, res);
        }else if(logonProcessURI.implies(urlPermission)){
        	logger.debug(" logonProcess phase ");
        	logonProcess(req, res);
        }else if(logoffURIs.implies(urlPermission)){
        	logger.debug(" logoff phase ");
                logoff(req, res);
                //authenticated user(subject!=null) is always authorize to logoff
		chain.doFilter(request,response);

        //we check that registration is defined (not defined when an SSO is present for example)
        }else if(registerProcessURI!= null && registerURI != null && registerProcessURI.implies(urlPermission)){

        	logger.debug(" registerProcess phase ");

        	//before registration, we authenticate the user as a guest
        	//=> it "hooks" loginmodules based on credentials because the user is not yer registered
        	//but permits to execute others loginmodules like captcha one and others to avoid
        	//annoyances with robot-based mass registration
        	boolean authenticate = authenticate(new AnonymizerRequestWrapper(req), res);
        	if(authenticate== false){
        		//we don't register the user because it fails loginModules not based on credentials
        		return;
        	}
            boolean registerSucceed = registerProcess(req, res, chain);

            if(!registerSucceed){
            	logger.debug(" registration failed " ," registerProcess phase ");
            	if(!res.isCommitted()){
            	 res.sendRedirect(res.encodeRedirectURL(req.getContextPath()+registerURI));
            	}else{
            		logger.warn(" we cannot redirect to "+req.getContextPath()+registerURI+" because response is already commited ");
            	}
            }else if(registerSucceed){
               	logger.debug(" registration succeed " ," registerProcess phase ");
               	//the user is registered and we submit directly its credentials to the authentication phase
               	req.getSession(true).removeAttribute(HttpConstants.AUTHN_UTILS);
               	req.getSession(true).removeAttribute(HttpConstants.LAST_ACCESS_DENIED_URI);
               	authenticateAfterRegistration(req, res);
            }

        }else{


            //access denied
            if(!HttpAccessControllerUtils.hasPermission(req,urlPermission)){
            	//we store the last access denied URI before authentication
            	//to redirect to this ur after successful authentication
            	req.getSession(true).setAttribute(HttpConstants.LAST_ACCESS_DENIED_URI,urlPermission.getURI());
            	logger.debug(" accessdenied phase ");
                accessDenied(req, res);
            }else{
            	//access granted
            	logger.debug(" authorize phase ");
                authorize(chain, req, res);
            }
        }


	}

	/**
	 * register the user against the @link SubjectTemplate.
	 * @param req
	 * @param res
	 * @param chain
	 * @return true if registration succeed, false otherwise
	 */
	private static boolean registerProcess(HttpServletRequest req, HttpServletResponse res, FilterChain chain) {
		boolean success = false;
		AuthenticationManager auth = AuthenticationManagerFactory.getAuthenticationManager();
		SubjectTemplate st = null;
		try {
			st = buildSubjectTemplate(req);
		} catch (AuthenticationException e1) {
			logger.error(" subject template cannot be built ",e1);
                        success = false;
			return false;
		}
                String passwordField = HttpServletCallbackHandler.getPasswordField();
                Set credentials = st.getRequiredCredentials();
                Iterator itCreds = credentials.iterator();
                boolean found = false;
                JGuardCredential passwordCredential = null;
                while(itCreds.hasNext() && found==false){
                    JGuardCredential cred = (JGuardCredential) itCreds.next();
                    if(cred.getId().equals(passwordField)){
                        passwordCredential = cred;
                        found = true;
                    }
                    
                }
                
                if(!found){
                    logger.warn("JGuardCredential matching  passwordField not found in the SubjectTemplate");
                    success = false;
                    return success;
                }
                char[] password = ((String)passwordCredential.getValue()).toCharArray();
                
                try {
                    passwordCredential.setValue(new String(CryptUtils.cryptPassword(password)));
                } catch (NoSuchAlgorithmException ex) {
                    logger.warn(ex.getMessage());
                    success = false;
                    return success;
                }
                
		Subject subject = null;
		try {
			subject = auth.createUser(st);
			success = true;
		} catch (AuthenticationException e) {
                     StringBuffer sb7 = new StringBuffer(" registration failed ");
                     logger.debug(" registrationProcess phase");
                     success = false;
		}

		return success;
	}

	/**
	 * fill in the SubjectTemplate the credentials from HttpServletRequest.
	 * @param req HttpServletRequest
	 * @return SubjectTemplate filled.
	 * @throws AuthenticationException
	 */
	private static SubjectTemplate buildSubjectTemplate(HttpServletRequest req) throws AuthenticationException{
		AuthenticationManager auth = AuthenticationManagerFactory.getAuthenticationManager();
		SubjectTemplate defaultSt = auth.getDefaultSubjectTemplate();
		SubjectTemplate st = new SubjectTemplate();
                st.setName(defaultSt.getName());

                //private required credentials
		Set privateCredRequiredFromDefaultSt =defaultSt.getPrivateRequiredCredentials();
		Set privRequiredCred = grabRegistrationForm(req, st, privateCredRequiredFromDefaultSt);
		st.setPrivateRequiredCredentials(privRequiredCred);

		//public required credentials
		Set publicCredRequiredFromDefaultSt =defaultSt.getPublicRequiredCredentials();
		Set pubRequiredCred = grabRegistrationForm(req, st, publicCredRequiredFromDefaultSt);
		st.setPublicRequiredCredentials(pubRequiredCred);

                //public optional credentials
		Set publicCredOptionalFromDefaultSt =defaultSt.getPublicOptionalCredentials();
		Set pubOptionalCred = grabRegistrationForm(req, st, publicCredOptionalFromDefaultSt);
		st.setPublicOptionalCredentials(pubOptionalCred);

                //public optional credentials
		Set privateCredOptionalFromDefaultSt =defaultSt.getPrivateOptionalCredentials();
		Set privOptionalCred = grabRegistrationForm(req, st, privateCredOptionalFromDefaultSt);
		st.setPrivateOptionalCredentials(privOptionalCred);


		return st;
	}

	/**
	 * build a set of credentials by grabbing data from HttpServletRequest.
	 * @param req HttpServletRequest
	 * @param st SubjectTemplate
	 * @param credentialsFromDefaultSt
	 * @return Set of {@link JGuardCredential}
	 */
	private static Set grabRegistrationForm(HttpServletRequest req, SubjectTemplate st, Set credentialsFromDefaultSt) {
		Iterator itCredentials = credentialsFromDefaultSt.iterator();
		Set credSet = new HashSet();
		while(itCredentials.hasNext()){
			JGuardCredential jcredFromDefault = (JGuardCredential)itCredentials.next();

			//test if we've found the credential in the http request
			if(req.getParameter(jcredFromDefault.getId())!= null){
				JGuardCredential jcred = new JGuardCredential();
				jcred.setId(jcredFromDefault.getId());
				try{
     				jcred.setValue(req.getParameter(jcredFromDefault.getId()));
	    			credSet.add(jcred);
				}catch(IllegalArgumentException iae){
					logger.warn(" the property "+jcredFromDefault.getId()+" doesn't exist in the HttpServletRequest ");
					continue;
				}
			}

		}
		return credSet;
	}

	/**
	 *
	 * @param req
	 * @param res
	 * @throws IOException
	 */
	private void logonProcess(HttpServletRequest req, HttpServletResponse res) throws IOException {
		HttpSession session = req.getSession(true);
		Subject subject = HttpAuthenticationUtils.getSubject(session);
		if(HttpConstants.BASIC_AUTH.equalsIgnoreCase(authScheme)){
			   String authorizationHeader = req.getHeader("Authorization");
			   logger.debug("authorizationHeader="+authorizationHeader);

			    //first time user reach the webapp in the session => we automatically authenticate it as a GUEST
			    if(subject == null){
			    	String AuthorizationHeaderValue = HttpServletCallbackHandler.buildBasicAuthHeader(
			    			                          SecurityConstants.GUEST,SecurityConstants.GUEST,req.getCharacterEncoding());
			    	JGuardServletRequestWrapper jguardReq = new JGuardServletRequestWrapper(req);
			    	jguardReq.setHeader(HttpServletCallbackHandler.AUTHORIZATION,AuthorizationHeaderValue);
			    	logger.debug(" BASIC authentication subject is null  authenticate phase ");
			    	authenticate(jguardReq, res);
			    }else if((authorizationHeader==null || !authorizationHeader.startsWith("Basic ") )){
		            //in BASIC AUTH, we don't redirect to a login FORM page
		    		//but put in the response header the challenge
			    	logger.debug(" subject is not null but BASIC HEADER is incorrect "+authorizationHeader," jGuard build BASIC challenge  ");
		    		HttpServletCallbackHandler.buildBasicChallenge(res,(String)session.getServletContext().getAttribute(CoreConstants.APPLICATION_NAME));
                            }else{
				logger.debug(" BASIC AUTHENTICATION TYPE   authenticate phase  ");
                                authenticate(req, res);
                            }
		}else{
			logger.debug(" AUTHENTICATION TYPE ="+authScheme+"  authenticate phase  ");
                        authenticate(req, res);
		}
	}

	
	/**
	 * send a HttpServletResponse.SC_UNAUTHORIZED HTTP status code (401).
	 * @param req
	 * @param res
	 * @throws IOException
	 */
	private void accessDenied(HttpServletRequest req, HttpServletResponse res) throws IOException {
		//redirect to accessDeniedURI because user is not authorized
		//to access to this resource
                if (logger.isDebugEnabled()) {
		    logger.debug(" access denied to "+req.getRequestURI());
		}
		
                logger.debug(" access is denied to "+req.getRequestURI()+" accessDeniedURI is not defined  jGuard send 401 http code ");
                res.setStatus(HttpServletResponse.SC_UNAUTHORIZED);
                res.sendError(HttpServletResponse.SC_UNAUTHORIZED,"access is denied to "+req.getRequestURI());
	
	}



	/**
	 * propagate security information and continue filter chain.
	 * @param chain
	 * @param req
	 * @param res
	 * @throws IOException
	 * @throws ServletException
	 */
	private void authorize(FilterChain chain, HttpServletRequest req, HttpServletResponse res) throws IOException, ServletException {
		if (logger.isDebugEnabled()) {
			logger.debug("doFilter() -  access authorized to "+ req.getRequestURI());
		}

		// user is authorized
		//we propagate the security informations in the Thread
		if(System.getSecurityManager()!=null){
		        try {
				    propagateSecurity(new JGuardServletRequestWrapper(req),res,chain);
		        } catch (PrivilegedActionException e) {
				    logger.warn(" access denied ",e);
				    if(!res.isCommitted()){
				    res.sendError(HttpServletResponse.SC_UNAUTHORIZED);
				    }
                                    
		         }
		         
		}else{
			//the securityManager is not enabled, so we don't propagate
			//security informations
			try{
			chain.doFilter(new JGuardServletRequestWrapper(req),res);
			}catch(Throwable t){
				logger.error(t.getMessage());
				res.sendError(HttpServletResponse.SC_SERVICE_UNAVAILABLE);
			}
		}
	}

	/**
	 * unauthenticate the user.
	 * @param request
	 * @param response
	 * @param chain
	 * @throws IOException
	 * @throws ServletException
	 */
	private static void logoff(HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException {
		//remove Subject from session
		HttpSession session = request.getSession();

		HttpAuthenticationUtils auth= (HttpAuthenticationUtils)session.getAttribute(HttpConstants.AUTHN_UTILS);
		if(auth!= null){
			auth.logout();
			logger.debug(" user logoff ");
		}

		session.removeAttribute(HttpConstants.AUTHN_UTILS);

		if (logger.isDebugEnabled()) {
				logger.debug("doFilter() -  user logoff ");
		}

        //we invalidate the session to unbound all objects, including subject
	    try{
           session.invalidate();
	    }catch(IllegalStateException ise){
	    	logger.debug(" session is already invalidated ");
	    }

	}

	/**
	 *
	 * propagate security informations into the execution Thread.
	 * @param request
	 * @param response
	 * @param chain
	 * @throws PrivilegedActionException
	 */
	private static void propagateSecurity(final HttpServletRequest request, final HttpServletResponse response, final FilterChain chain) throws PrivilegedActionException{
		        HttpSession session  = request.getSession();
	            Subject subject = HttpAuthenticationUtils.getSubject(session);
                Subject.doAsPrivileged(subject, new PrivilegedExceptionAction() {
                public Object run() throws IOException, ServletException {
                	//we wrap the ServletRequest to 'correct' the j2ee's JAAS handling
                	// according to the j2se way
                	try{
                	chain.doFilter(new JGuardServletRequestWrapper(request),response);
            			}catch(Throwable t){
            				logger.error(t.getMessage());
            				response.sendError(HttpServletResponse.SC_SERVICE_UNAVAILABLE);
            			}
                    // the 'null' tells the SecurityManager to consider this resource access
                    //in an isolated context, ignoring the permissions of code currently
                    //on the execution stack.
                    return null;
                }
                },null);

	}


	/**
	 * authenticate client request according the authentication Policy defined in the 'authSchemes' field.
	 * @param req
	 * @param res
	 * @return <code>true</code> if authentication succeeds, <code>false</code> otherwise
	 * @throws IOException
	 */
	private boolean authenticate(HttpServletRequest req, HttpServletResponse res) throws IOException {
                boolean authenticationFailed = !HttpAuthenticationUtils.authenticate(req,res,false,local);
		return postAuthenticationProcess(req, res, authenticationFailed);
	}

	/**
	 * authenticate client request according the authentication Policy defined in the 'authSchemes' field.
	 * @param req
	 * @param res
	 * @return <code>true</code> if authentication succeeds, <code>false</code> otherwise
	 * @throws IOException
	 */
	private boolean authenticateAfterRegistration(HttpServletRequest req, HttpServletResponse res) throws IOException {
        boolean authenticationFailed = !HttpAuthenticationUtils.authenticate(req,res,true,local);
		return postAuthenticationProcess(req, res,authenticationFailed);
	}

	private boolean postAuthenticationProcess(HttpServletRequest req, HttpServletResponse res, boolean authenticationFailed) throws IOException {
		
		Subject subject = HttpAuthenticationUtils.getSubject(req.getSession(true));
		
		if(authenticationFailed){
			if(res.isCommitted()){
				logger.warn(" response is already committed ");
			}

			if(HttpConstants.BASIC_AUTH.equalsIgnoreCase(authScheme)){
				  HttpServletCallbackHandler.buildBasicChallenge(res,(String)((HttpServletRequest)req).getSession(true).getServletContext().getAttribute(CoreConstants.APPLICATION_NAME));
				  return !authenticationFailed;
			 }
			logger.debug("authentication failed redirect to "+authenticationFailedURI.getURI());

		   if(authenticationFailedURI!=null && !authenticationFailedURI.getURI().equals("")){
				   res.sendRedirect(res.encodeRedirectURL(req.getContextPath()+authenticationFailedURI.getURI()));
				   logger.debug(" NOT BASIC AUTHENTICATION - user is not authenticated  redirect to "+req.getContextPath()+authenticationFailedURI.getURI());
			   
		   }else{
                           accessDenied(req,res);
		   }

		//authentication succeed
		}else{
			HttpSession oldSession = req.getSession(true);
		    String redirectURI = indexURI;
		    String lastAccessDeniedURI = (String)oldSession.getAttribute(HttpConstants.LAST_ACCESS_DENIED_URI);
		    
		    //for security reasons(man-in-the-middle attack for sessionID cookie),
		    //we remove the old session if authentication is successfull
		    //and create a new session
		    //to not lost the subject, we grab in the old session HttpAuthenticationUtils 
		    //and put it in the new session
		    HttpAuthenticationUtils httpAuthenticationUtils = HttpAuthenticationUtils.getHttpAuthenticationUtils(req,local);
		    subject = httpAuthenticationUtils.getSubject();
		    
		    //we test if response is committed, because
		    //when user register, it authenticate twice
		    // => the response is committed at the first step
		    if(!res.isCommitted()){
                            //we remove this variable before invalidate the session 
			    //to prevent the session listener to erase the subject 
			    oldSession.removeAttribute(HttpConstants.AUTHN_UTILS);
			    oldSession.invalidate();
			    
			    HttpSession newSession = req.getSession(true);
			    newSession.setAttribute(HttpConstants.AUTHN_UTILS,httpAuthenticationUtils);
		    }
		    
		    //we redirect to the last 'access denied' URI before authentication
		    if(lastAccessDeniedURI!=null && lastAccessDeniedURI!="") {
		    	if(goToLastAccessDeniedUriOnSuccess)
		    		redirectURI = lastAccessDeniedURI;
		    	else {
				    URLPermission urlPermission = new URLPermission("indexURI",indexURI);
				    if(!HttpAccessControllerUtils.hasPermission(req,urlPermission))
				    	redirectURI = logonURIStr;
		    	}
		    }
	

		    logger.debug(" user is authenticated "," redirect to "+redirectURI);
		    if(!res.isCommitted()){
		      res.sendRedirect(res.encodeRedirectURL(req.getContextPath()+redirectURI));
		    }
		    
		}
		return !authenticationFailed;
	}

    /**
	 * @see javax.servlet.Filter#destroy()
	 */
	public void destroy() {
	  //no operation to do when the webapp shutdown
	}



	 /**
     * load configuration from an XML file.
     * @param configurationLocation
     * @return Map containing filter configuration
     */
	private Map loadFilterConfiguration(String configurationLocation){
		Document doc = XMLUtils.read(configurationLocation);

		Element authentication = doc.getRootElement().element(HttpConstants.FILTER);
		Map filterSettings = new HashMap();
		filterSettings.put(HttpConstants.INDEX_URI,authentication.element(HttpConstants.INDEX_URI).getTextTrim());
		filterSettings.put(HttpConstants.AUTHENTICATION_FAILED_URI,authentication.element(HttpConstants.AUTHENTICATION_FAILED_URI).getTextTrim());
		if(authentication.element(HttpConstants.REGISTER_PROCESS_URI)!=null){
			filterSettings.put(HttpConstants.REGISTER_PROCESS_URI,authentication.element(HttpConstants.REGISTER_PROCESS_URI).getTextTrim());
		}
		if(authentication.element(HttpConstants.REGISTER_URI)!=null){
			filterSettings.put(HttpConstants.REGISTER_URI,authentication.element(HttpConstants.REGISTER_URI).getTextTrim());
		}
		filterSettings.put(HttpConstants.LOGON_PROCESS_URI,authentication.element(HttpConstants.LOGON_PROCESS_URI).getTextTrim());
		filterSettings.put(HttpConstants.LOGON_URI,authentication.element(HttpConstants.LOGON_URI).getTextTrim());

		Set logoffURIsCollection = new HashSet();
		Iterator itLogoffURI = authentication.element(HttpConstants.LOGOFF_URIS).elements(HttpConstants.LOGOFF_URI).iterator();
		while (itLogoffURI.hasNext()){
			logoffURIsCollection.add(((Element)itLogoffURI.next()).getTextTrim());
		}
		filterSettings.put(HttpConstants.LOGOFF_URIS,logoffURIsCollection);

		filterSettings.put(HttpConstants.AUTH_SCHEME,authentication.element(HttpConstants.AUTH_SCHEME).getTextTrim());
		Element loginElement = authentication.element(HttpConstants.LOGIN_FIELD);
		if (loginElement != null){
			filterSettings.put(HttpConstants.LOGIN_FIELD,loginElement.getTextTrim());
		}
		Element passwordElement = authentication.element(HttpConstants.PASSWORD_FIELD);
		if (passwordElement != null){
			filterSettings.put(HttpConstants.PASSWORD_FIELD,passwordElement.getTextTrim());
		}

		Element goToLastAccessDeniedUriOnSuccessElement = authentication.element(HttpConstants.GO_TO_LAST_ACCESS_DENIED_URI_ON_SUCCESS);
		if (goToLastAccessDeniedUriOnSuccessElement != null){
			filterSettings.put(HttpConstants.GO_TO_LAST_ACCESS_DENIED_URI_ON_SUCCESS,goToLastAccessDeniedUriOnSuccessElement.getTextTrim());
		}
		
		return filterSettings;
	}
        
	
}
