/*
jGuard is a security framework based on top of jaas (java authentication and authorization security).
it is written for web applications, to resolve simply, access control problems.
version $Name:  $
http://sourceforge.net/projects/jguard/

Copyright (C) 2004  Charles GAY

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


jGuard project home page:
http://sourceforge.net/projects/jguard/

*/
package net.sf.jguard.jee.authentication.callbacks;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import javax.security.auth.Subject;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import net.sf.jguard.ext.SecurityConstants;
import net.sf.jguard.ext.authentication.callbacks.CertificatesCallback;
import net.sf.jguard.ext.authentication.callbacks.InetAdressCallback;
import net.sf.jguard.ext.authentication.callbacks.JCaptchaCallback;
import net.sf.jguard.ext.authentication.certificates.CertificateConverter;
import net.sf.jguard.jee.authentication.http.HttpAuthenticationUtils;
import net.sf.jguard.jee.authentication.http.HttpConstants;

import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.octo.captcha.service.CaptchaService;
import org.xbill.DNS.DClass;
import org.xbill.DNS.ExtendedResolver;
import org.xbill.DNS.Message;
import org.xbill.DNS.Name;
import org.xbill.DNS.Record;
import org.xbill.DNS.Resolver;
import org.xbill.DNS.ReverseMap;
import org.xbill.DNS.Section;
import org.xbill.DNS.Type;

/**
 * handle grabbing credentials from an HTTP request.
 * @author <a href="mailto:diabolo512@users.sourceforge.net ">Charles Gay</a>
 */
public class HttpServletCallbackHandler implements CallbackHandler{

	private static final String JAVAX_SERVLET_REQUEST_X509CERTIFICATE = "javax.servlet.request.X509Certificate";
	private static final String DIGEST_REALM = "Digest realm=\"";
	public static final String AUTHORIZATION = "Authorization";
	private static final String BASIC_REALM = "Basic realm=\"";
	private static final String NO_CACHE_AUTHORIZATION = "no-cache=\"Authorization\"";
	private static final String CACHE_CONTROL = "Cache-Control";
	private static final String WWW_AUTHENTICATE = "WWW-Authenticate";
	private static final String BASIC = "Basic ";
	private static final String ISO_8859_1 = "ISO-8859-1";
	/** Logger for this class */
	private static final Logger logger = LoggerFactory.getLogger(HttpServletCallbackHandler.class);
	private HttpServletRequest httpRequest;
	private HttpServletResponse httpResponse;
	private static String authScheme=HttpConstants.FORM_AUTH;
	private static String loginField="login";
	private static String passwordField="password";
	private boolean afterRegistration;


	/**
	 * constructor required by javadoc of the CallbackHandler interface.
	 */
	public HttpServletCallbackHandler(){

	}

	/**
	 * constructor.
	 * @param request
	 * @param response
	 * @param authScheme
	 */
	public HttpServletCallbackHandler(HttpServletRequest request,HttpServletResponse response){
		this.httpRequest = request;
		this.httpResponse = response;
	}


	/**
	 * extract from the HttpServletRequest client's credentials.
	 * if those are not recognised, we put the challenge in the
	 * HttpServletResponse.
	 */
	public void handle(Callback[] callbacks) throws IOException, UnsupportedCallbackException {
                
                boolean httpRelatedAuthScheme = false;
                for (int i = 0; i< callbacks.length;i++){
                    Callback callback = callbacks[i];
                    if(callback instanceof InetAdressCallback){
                        InetAdressCallback inetAdressCallback = (InetAdressCallback)callback;
                        inetAdressCallback.setHostAdress(httpRequest.getRemoteAddr());
                        
                        //the server is not cofnigured to return the hostName.
                        if(httpRequest.getRemoteAddr().equals(httpRequest.getRemoteHost())){
                            String hostName = reverseDns(httpRequest.getRemoteAddr());
                            inetAdressCallback.setHostName(hostName);
                        }else{
                            //the server is configured to return the hostName.
                            inetAdressCallback.setHostName(httpRequest.getRemoteHost());
                        }
                    }
                }
                
                //authentication schemes part
                logger.debug("authSchemeItem="+authScheme);
                String[] schemes = authScheme.split(",");
                List authSchemes = Arrays.asList(schemes);
                Iterator itAutSchemes = authSchemes.iterator();
                while(itAutSchemes.hasNext()){
                        String scheme = (String)itAutSchemes.next();
                        //FORM, BASIC, and DIGEST are mutual exclusive
                        if(!httpRelatedAuthScheme && HttpConstants.FORM_AUTH.equalsIgnoreCase(scheme)){
                                grabFormCredentials(this.httpRequest,callbacks);
                                httpRelatedAuthScheme = true;
                        }else if(!httpRelatedAuthScheme && HttpConstants.BASIC_AUTH.equalsIgnoreCase(scheme)){
                                grabBasicCredentials(this.httpRequest,callbacks);
                                httpRelatedAuthScheme = true;
                        }else if(!httpRelatedAuthScheme && HttpConstants.DIGEST_AUTH.equalsIgnoreCase(scheme)){
                                grabDigestCredentials(this.httpRequest,callbacks);
                                httpRelatedAuthScheme = true;
                        }
                        //CLIENT_CERT can be used with another authentication mechanism
                        //defined above
                        if(HttpConstants.CLIENT_CERT_AUTH.equalsIgnoreCase(scheme)){
                                boolean certificatesFound = grabClientCertCredentials(this.httpRequest,callbacks);
                                if(!certificatesFound){
                                        logger.info(" X509 certificates are not found ");
                                }
                        }
                }

	}

	public HttpServletRequest getHttpRequest() {
		return httpRequest;
	}


	public void setHttpRequest(HttpServletRequest httpRequest) {
		this.httpRequest = httpRequest;
	}

	public HttpServletResponse getHttpResponse() {
		return httpResponse;
	}

	public void setHttpResponse(HttpServletResponse httpResponse) {
		this.httpResponse = httpResponse;
	}

	/**
	 * construct a header value to simulate a Basic authentication with the provided credentials.
	 * @param login
	 * @param password
	 * @param encoding
	 * @return header
	 */
	public static String buildBasicAuthHeader(String login,String password,String encoding){
		if(encoding==null){
			encoding=HttpServletCallbackHandler.ISO_8859_1;
		}
		StringBuffer decodedString = new StringBuffer();
		decodedString.append(login);
		decodedString.append(" : ");
		decodedString.append(password);
		String encodedString;
		try {
			encodedString = new String(Base64.encode(decodedString.toString().getBytes(encoding)));
		} catch (UnsupportedEncodingException e) {
			encodedString = new String(Base64.encode(decodedString.toString().getBytes()));
		}
		StringBuffer header = new StringBuffer();
		header.append(HttpServletCallbackHandler.BASIC);
		header.append(encodedString);
		header.append("==");
		return header.toString();
	}

	/**
	 * send to the client the BASIC challenge into the response, according to the RFC 2617.
	 * @param response reponse send to the Client
	 * @param realmName realm owned by the server => specify what kind of credential the user should provide
	 */
	public static void buildBasicChallenge(HttpServletResponse response,String realmName){
		StringBuffer responseValue= new StringBuffer();
		responseValue.append(HttpServletCallbackHandler.BASIC_REALM);
		responseValue.append(realmName);
		responseValue.append("\"");
		response.setHeader(HttpServletCallbackHandler.WWW_AUTHENTICATE,responseValue.toString());
		response.setHeader(HttpServletCallbackHandler.CACHE_CONTROL, HttpServletCallbackHandler.NO_CACHE_AUTHORIZATION);
		response.setStatus(HttpServletResponse.SC_UNAUTHORIZED);
	}
	
	
	/**
	 * parse into the HttpServletRequest the user and password field,
	 * and authenticate the user with these credentials using the <b>NON SECURE</> BASIC method.
	 * @param request request send by the client.
	 * @param callbacks
	 * @return authentication's result. <i>true</i> for authentication success, <i>false</i> otherwise.
	 */
	private boolean grabBasicCredentials(HttpServletRequest request,Callback[] callbacks){
		boolean result = false;
        String login="";
        String password="";
		//user and password are encoded in Base64
		String encodedLoginAndPwd = request.getHeader(HttpServletCallbackHandler.AUTHORIZATION);

		if(encodedLoginAndPwd==null ||encodedLoginAndPwd.equals("")){
			login =SecurityConstants.GUEST;
	        password =SecurityConstants.GUEST;

		}else{
			encodedLoginAndPwd = encodedLoginAndPwd.substring(6).trim();
			String decodedLoginAndPassword = null;

				String encoding =  request.getCharacterEncoding();
				if(encoding==null){
					encoding=HttpServletCallbackHandler.ISO_8859_1;
				}
				logger.debug(encoding);

				try {
					decodedLoginAndPassword = new String(Base64.decode(encodedLoginAndPwd.getBytes()),encoding);
				} catch (UnsupportedEncodingException e) {
					e.printStackTrace();
					logger.error(" encoding "+encoding+" is not supported by the platform ");
				}

			String[] parts = decodedLoginAndPassword.split(":");
			if(parts.length == 2 ){
				login = parts[0].trim();
				password = parts[1].trim();

				result = true;
			}
			if((login=="" && password=="")||(parts.length==0)){
				login =SecurityConstants.GUEST;
		        password =SecurityConstants.GUEST;
			}

		}

		fillBasicCredentials(callbacks,login,password);
		return result;
	}

    /**
     * grab user credentials from request in the 'form' authentication metod.
     * @param request request send by the client
     * @param callbacks
     * @return authentication result : <b>true</b> when authentication succeed,<b>false</b> when authentication fails.
     */
	private  boolean  grabFormCredentials(HttpServletRequest request,Callback[] callbacks){
		boolean result = false;
        HttpSession session = request.getSession();

		for(int i=0;i<callbacks.length;i++){
        	if(callbacks[i] instanceof NameCallback){
        		NameCallback nc = (NameCallback)callbacks[i];
        		String login =httpRequest.getParameter(loginField);
        		nc.setName(login);
        	}else if(callbacks[i] instanceof PasswordCallback){
        		PasswordCallback pc = (PasswordCallback)callbacks[i];
        		String strPwd = httpRequest.getParameter(passwordField);
                 if(strPwd!= null &&strPwd!=""){
        		  pc.setPassword(strPwd.toCharArray());
                 }else{
                  pc.setPassword(null);
                 }
        	}else if(callbacks[i] instanceof JCaptchaCallback){
        		JCaptchaCallback pc = (JCaptchaCallback)callbacks[i];
        		pc.setCaptchaAnswer(httpRequest.getParameter(SecurityConstants.CAPTCHA_ANSWER));
        		pc.setCaptchaService((CaptchaService)session.getServletContext().getAttribute(SecurityConstants.CAPTCHA_SERVICE));
        		Subject subject = ((HttpAuthenticationUtils)session.getAttribute(HttpConstants.AUTHN_UTILS)).getSubject();
        		if(subject==null ||this.isAfterRegistration()){
        		    pc.setSkipJCaptchaChallenge(true);
        		}

        		pc.setSessionID(session.getId());
        	}
        }
        result = true;


		return result;
	}

	 /**
	 * grab user credentials from request in the 'digest' authentication metod.
     * @param request request send by the client
     * @param callbacks
     * @return authentication result : <b>true</b> when authentication succeed,<b>false</b> when authentication fails.
     */
	private  boolean  grabDigestCredentials(HttpServletRequest request,Callback[] callbacks){
        boolean result = false;
		String login = "";
        String password = "";
        //all users must be authenticated
        //unless when the user send a wrong  login or/and password
        //=> he is redirected to the logonPage
        if(login==null || password == null ){
           login =SecurityConstants.GUEST;
           password =SecurityConstants.GUEST;
        }else{
            //TODO implements digest authentication
           result = true;
        }
		return result;
	}

	/**
	 * grab user credentials from request in the 'clientCert' authentication metod.
	 * @param request
	 * @param callbacks
	 * @return <code>true</code> if successfull, <code>false</code> otherwise
	 */
	private boolean grabClientCertCredentials(HttpServletRequest request,Callback[] callbacks) {
		if(!request.isSecure()){
			logger.warn(" certificate-based authentication MUST be do in secure mode ");
			logger.warn(" but connection is do with the non secured protocol "+request.getScheme());
			return false;
		}

		X509Certificate[] certificates = null;
		javax.security.cert.X509Certificate[] oldCerts = null;
		Object[] objects = (Object[]) request.getAttribute(HttpServletCallbackHandler.JAVAX_SERVLET_REQUEST_X509CERTIFICATE);
		
		if(objects == null || objects.length==0){
			return false;
		}
		
		if(objects instanceof X509Certificate[]) {
			certificates= (X509Certificate[]) objects;
		//convert old X509 certificates into new X509 certificates
		}else if(objects instanceof javax.security.cert.X509Certificate[]) {
			oldCerts = (javax.security.cert.X509Certificate[])objects;
			List newCerts = null;
			for(int i =0;i<oldCerts.length;i++){
			 newCerts = Arrays.asList(certificates);
			 newCerts.add(CertificateConverter.convertOldToNew(oldCerts[i]));
			}
			certificates = (X509Certificate[]) newCerts.toArray();
		}else{
			logger.warn(" X509certificates are needed but not provided by the client ");
			return false;
		}
		fillCertCredentials(callbacks,certificates);

		return true;
	}

	private void fillBasicCredentials(Callback[] callbacks, String login, String password) {
		for(int i=0;i<callbacks.length;i++){
        	if(callbacks[i] instanceof NameCallback){
        		NameCallback nc = (NameCallback)callbacks[i];
        		nc.setName(login);

        	}else if(callbacks[i] instanceof PasswordCallback){
        		PasswordCallback pc = (PasswordCallback)callbacks[i];
        		pc.setPassword(password.toCharArray());
        	}else if (callbacks[i] instanceof JCaptchaCallback){
				JCaptchaCallback jc = (JCaptchaCallback)callbacks[i];
				//we skip JCaptcha because we cannot provide 
				//CAPTCHA challenge through BASIC authentication
				jc.setSkipJCaptchaChallenge(true);
        	}
        }
	}


	private void fillCertCredentials(Callback[] callbacks,X509Certificate[] certificates) {
		for(int i=0;i<callbacks.length;i++){
        	if(callbacks[i] instanceof CertificatesCallback){
        		CertificatesCallback cc = (CertificatesCallback)callbacks[i];
        		cc.setCertificates(certificates);
        		break;
        	}
        }
	}
	
	public static void buildFormChallenge(FilterChain chain,ServletRequest req,ServletResponse res) throws IOException, ServletException{
		chain.doFilter(req,res);
	}

	/**
	 * send to the client the DIGEST challenge into the response, according to the RFC 2617.
	 * @param response reponse send to the Client
	 * @param token realm owned by the server => specify what kind of credential the user should provide
	 */
	public static void buildDigestChallenge(HttpServletRequest request,HttpServletResponse response, String realm) {
			//TODO buildDigestChallenge method is not complete
			StringBuffer responseValue= new StringBuffer();
			//what about domain which defines the protection space?
			
			//realm
			responseValue.append(HttpServletCallbackHandler.DIGEST_REALM);
			responseValue.append(realm);
			responseValue.append("\"");
			responseValue.append(",");
			//quality of protection qop
			responseValue.append("qop=\"");
			responseValue.append(getQop());
			responseValue.append("\"");
			responseValue.append(",");
			
			responseValue.append("nonce=\"");
			responseValue.append(getNonce(request));
			responseValue.append("\"");
			responseValue.append(",");
			//opaque
			responseValue.append("opaque=");
			responseValue.append("\"");
			responseValue.append(getOpaque());
			responseValue.append("\"");
			//algorithm
			responseValue.append("algorithm=");
			responseValue.append("\"");
			responseValue.append(getAlgorithm());
			responseValue.append("\"");
			//stale
			responseValue.append("stale=");
			responseValue.append("\"");
			responseValue.append(getStale());
			responseValue.append("\"");
			response.setHeader(HttpServletCallbackHandler.WWW_AUTHENTICATE,responseValue.toString());
			response.setStatus(HttpServletResponse.SC_UNAUTHORIZED);
	}
	
	/**
	 * A flag, indicating that the previous request from the client was
     rejected because the nonce value was stale. If stale is TRUE
     (case-insensitive), the client may wish to simply retry the request
     with a new encrypted response, without reprompting the user for a
     new username and password. The server should only set stale to TRUE
     if it receives a request for which the nonce is invalid but with a
     valid digest for that nonce (indicating that the client knows the
     correct username/password). If stale is FALSE, or anything other
     than TRUE, or the stale directive is not present, the username
     and/or password are invalid, and new values must be obtained
	 * @return
	 */
	private static String getStale() {
		return "false";
	}

	/**
	 * This directive is optional, but is made so only for backward
	     compatibility with RFC 2069 [6]; it SHOULD be used by all
	     implementations compliant with this version of the Digest scheme.
	     If present, it is a quoted string of one or more tokens indicating
	     the "quality of protection" values supported by the server.  The
	     value "auth" indicates authentication; the value "auth-int"
	     indicates authentication with integrity protection; see the
	     descriptions below for calculating the response directive value for
		the application of this choice. Unrecognized options MUST be
		ignored.
	 * @return
	 */
	private static String getQop() {
		return "auth,auth-int";
	}

	/**
	 * A string of data, specified by the server, which should be returned
	     by the client unchanged in the Authorization header of subsequent
	     requests with URIs in the same protection space. It is recommended
	     that this string be base64 or hexadecimal data.
	 * @return
	 */
	private static String getOpaque() {
		return "5ccc069c403ebaf9f0171e9517f40e41";
	}

	/**
	 * 
     A string indicating a pair of algorithms used to produce the digest
     and a checksum. If this is not present it is assumed to be "MD5".
     If the algorithm is not understood, the challenge should be ignored
     (and a different one used, if there is more than one).

     In this document the string obtained by applying the digest
     algorithm to the data "data" with secret "secret" will be denoted
     by KD(secret, data), and the string obtained by applying the
     checksum algorithm to the data "data" will be denoted H(data). The
     notation unq(X) means the value of the quoted-string X without the
     surrounding quotes.

     For the "MD5" and "MD5-sess" algorithms

         H(data) = MD5(data)

     and

         KD(secret, data) = H(concat(secret, ":", data))

     i.e., the digest is the MD5 of the secret concatenated with a colon
     concatenated with the data. The "MD5-sess" algorithm is intended to
     allow efficient 3rd party authentication servers; for the
     difference in usage, see the description in section 3.2.2.2.
	 * @return
	 */
	private static String getAlgorithm() {
		return "MD5";
	}

	/**
	 * //nonce
			
	   A server-specified data string which should be uniquely generated
	     each time a 401 response is made. It is recommended that this
	     string be base64 or hexadecimal data. Specifically, since the
	     string is passed in the header lines as a quoted string, the
	     double-quote character is not allowed.

	     The contents of the nonce are implementation dependent. The quality
	     of the implementation depends on a good choice. A nonce might, for
	     example, be constructed as the base 64 encoding of

	         time-stamp H(time-stamp ":" ETag ":" private-key)

	     where time-stamp is a server-generated time or other non-repeating
	     value, ETag is the value of the HTTP ETag header associated with
	     the requested entity, and private-key is data known only to the
	     server.  With a nonce of this form a server would recalculate the
	     hash portion after receiving the client authentication header and
	     reject the request if it did not match the nonce from that header
	     or if the time-stamp value is not recent enough. In this way the
	     server can limit the time of the nonce's validity. The inclusion of
	     the ETag prevents a replay request for an updated version of the
	     resource.  (Note: including the IP address of the client in the
	     nonce would appear to offer the server the ability to limit the
	     reuse of the nonce to the same client that originally got it.
	     However, that would break proxy farms, where requests from a single
	     user often go through different proxies in the farm. Also, IP
	     address spoofing is not that hard.)

	     An implementation might choose not to accept a previously used
	     nonce or a previously used digest, in order to protect against a
	     replay attack. Or, an implementation might choose to use one-time
	     nonces or digests for POST or PUT requests and a time-stamp for GET
	     requests.  For more details on the issues involved see section 4.
	     of this document.  The nonce is opaque to the client.
	 * @param request
	 * @return
	 */
	private static String getNonce(HttpServletRequest request){
		return "dcd98b7102dd2f0e8b11d0f600bfb0c093";
	}


	/**
	 * gets the HttpRequest password field
	 * @return password field
	 */
	public static String getPasswordField() {
		return passwordField;
	}

	public static void setPasswordField(String passwordField) {
		if(passwordField!=null){
			HttpServletCallbackHandler.passwordField = passwordField;
		}
	}

	/**
	 * gets the HttpRequest login field
	 * @return login field
	 */
	public static String getLoginField() {
		return loginField;
	}

	public static void setLoginField(String loginField) {
		if(loginField!=null){
			HttpServletCallbackHandler.loginField = loginField;
		}
	}

	public void setAfterRegistration(boolean afterRegistration) {
       this.afterRegistration = afterRegistration;
	}

	public boolean isAfterRegistration() {
		return afterRegistration;
	}

        /**
         *return the host name related to the IP adress.
         *this method comes from <a href="http://www.oreillynet.com/onjava/blog/2005/11/reverse_dns_lookup_and_java.html">a blog entry about dnsjava</a>.
         *@param hostIp Internet Protocol  adress
         *@return host name related to the hostIp parameter, 
         *or hostIp parameter if no nam eserver is found.
         */
        private String reverseDns(String hostIp) throws IOException {
                 Record opt = null;
                 Resolver res = new ExtendedResolver();

                 Name name = ReverseMap.fromAddress(hostIp);
                 int type = Type.PTR;
                 int dclass = DClass.IN;
                 Record rec = Record.newRecord(name, type, dclass);
                 Message query = Message.newQuery(rec);
                 Message response = res.send(query);

                 Record[] answers = response.getSectionArray(Section.ANSWER);
                 if (answers.length == 0){
                    return hostIp;
                 }else{
                    return answers[0].rdataToString();
                 }
        }

    public String getAuthScheme() {
        return authScheme;
    }

    public static void setAuthScheme(String authScheme) {
        authScheme = authScheme;
    }
}
