/*
jGuard is a security framework based on top of jaas (java authentication and authorization security).
it is written for web applications, to resolve simply, access control problems.
version $Name$
http://sourceforge.net/projects/jguard/

Copyright (C) 2004  Charles GAY

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


jGuard project home page:
http://sourceforge.net/projects/jguard/

*/

package net.sf.jguard.jee.provisioning;

import java.io.IOException;
import java.security.NoSuchAlgorithmException;
import java.security.Permission;
import java.security.Permissions;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import net.sf.jguard.core.CoreConstants;
import net.sf.jguard.core.authentication.AccessContext;
import net.sf.jguard.core.authentication.AuthenticationException;
import net.sf.jguard.core.authentication.credentials.JGuardCredential;
import net.sf.jguard.core.authentication.manager.AuthenticationManager;
import net.sf.jguard.core.authentication.manager.AuthenticationManagerFactory;
import net.sf.jguard.core.authorization.permissions.URLPermission;
import net.sf.jguard.core.provisioning.ProvisioningServicePoint;
import net.sf.jguard.core.provisioning.SubjectTemplate;
import net.sf.jguard.ext.util.CryptUtils;
import net.sf.jguard.ext.util.XMLUtils;
import net.sf.jguard.jee.authentication.callbacks.HttpServletCallbackHandler;
import net.sf.jguard.jee.authentication.http.AccessFilter;
import net.sf.jguard.jee.authentication.http.AnonymizerRequestWrapper;
import net.sf.jguard.jee.authentication.http.HttpConstants;

import org.dom4j.Document;
import org.dom4j.Element;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 * @author <a href="mailto:diabolo512@users.sourceforge.net">Charles Gay</a>
 */
public class HttpServletProvisioningServicePoint implements ProvisioningServicePoint{
    
    private static final Logger logger = LoggerFactory.getLogger(HttpServletProvisioningServicePoint.class.getName());
    private String registerURI;
    private URLPermission registerProcessPermission;
    private URLPermission registerPermission;
    
    
    /**
     * Creates a new instance of HttpServletProvisioningServicePoint
     */
    public HttpServletProvisioningServicePoint() {
    }

    
	public void init(String location) {
		Map settings = loadFilterConfiguration(location);
        setSettings(settings);
		
	}
    
    public Permission getRegisterPermission() {
          return registerPermission;
    }

    public Permission getRegisterProcessPermission() {
        return registerProcessPermission;
    }

    public boolean registerProcess(AccessContext context) {
        boolean registerSucceed = registerCoreProcess(context);
	HttpServletRequest request = (HttpServletRequest)context.getAttribute(AccessFilter.SERVLET_REQUEST);
	HttpServletResponse response = (HttpServletResponse)context.getAttribute(AccessFilter.SERVLET_RESPONSE);
	boolean result = false;	
        if(!registerSucceed){
         	logger.debug(" registration failed " ," registerProcess phase ");
                
         	if(!response.isCommitted()){
         		try {
					response.sendRedirect(response.encodeRedirectURL(request.getContextPath()+registerURI));
				} catch (IOException e) {
					logger.warn(" we cannot redirect to "+request.getContextPath()+registerURI+" because "+e.getMessage());
				}
         	}else{
         		logger.warn(" we cannot redirect to "+request.getContextPath()+registerURI+" because response is already commited ");
         	}
                result = false;
         }else if(registerSucceed){
            	logger.debug(" registration succeed " ," registerProcess phase ");
            	//the user is registered and we submit directly its credentials to the authentication phase
            	request.getSession(true).removeAttribute(CoreConstants.AUTHN_UTILS);
            	request.getSession(true).removeAttribute(CoreConstants.LAST_ACCESS_DENIED_PERMISSION);
                result = true;
            	
         } 
        return result;
    }
    
    /**
	 * register the user against the @link SubjectTemplate.
	 * @param context 
	 * @return true if registration succeed, false otherwise
	 */
	public boolean registerCoreProcess(AccessContext context) {
		HttpServletRequest request = (HttpServletRequest)context.getAttribute(AccessFilter.SERVLET_REQUEST);
		boolean success = false;
		AuthenticationManager auth = AuthenticationManagerFactory.getAuthenticationManager();
		SubjectTemplate st = null;
		try {
			st = buildSubjectTemplate(request);
		} catch (AuthenticationException e1) {
			logger.error(" subject template cannot be built ",e1);
                        success = false;
			return false;
		}
                String passwordField = HttpServletCallbackHandler.getPasswordField();
                Set credentials = st.getRequiredCredentials();
                Iterator itCreds = credentials.iterator();
                boolean found = false;
                JGuardCredential passwordCredential = null;
                while(itCreds.hasNext() && found==false){
                    JGuardCredential cred = (JGuardCredential) itCreds.next();
                    if(cred.getName().equals(passwordField)){
                        passwordCredential = cred;
                        found = true;
                    }
                    
                }
                
                if(!found){
                    logger.warn("JGuardCredential matching  passwordField not found in the SubjectTemplate");
                    success = false;
                    return success;
                }
                char[] password = (passwordCredential.getValue().toString()).toCharArray();
                
                try {
                    passwordCredential.setValue(new String(CryptUtils.cryptPassword(password)));
                } catch (NoSuchAlgorithmException ex) {
                    logger.warn(ex.getMessage());
                    success = false;
                    return success;
                }
                
		try {
			auth.createUser(st,auth.getDefaultOrganization());
			success = true;
		} catch (AuthenticationException e) {
            logger.debug(" registrationProcess failed ");
            success = false;
		}

		return success;
	}


    public AccessContext anonymize(AccessContext context) {
	AccessContext accContext;
	accContext = (AccessContext)context.clone();
	HttpServletRequest request = (HttpServletRequest)context.getAttribute(AccessFilter.SERVLET_REQUEST);
	accContext.setAttribute(AccessFilter.SERVLET_REQUEST, new AnonymizerRequestWrapper(request));
	return accContext;
    }
    
    /**
     * fill in the SubjectTemplate the credentials from HttpServletRequest.
     * @param req HttpServletRequest
     * @return SubjectTemplate filled.
     * @throws AuthenticationException
     */
    private static SubjectTemplate buildSubjectTemplate(HttpServletRequest req) throws AuthenticationException{
            AuthenticationManager auth = AuthenticationManagerFactory.getAuthenticationManager();
            SubjectTemplate defaultSt = auth.getDefaultOrganization().getSubjectTemplate();
            SubjectTemplate st = new SubjectTemplate();
            

            //private required credentials
            Set privateCredRequiredFromDefaultSt =defaultSt.getPrivateRequiredCredentials();
            Set privRequiredCred = grabRegistrationForm(req, st, privateCredRequiredFromDefaultSt);
            st.setPrivateRequiredCredentials(privRequiredCred);

            //public required credentials
            Set publicCredRequiredFromDefaultSt =defaultSt.getPublicRequiredCredentials();
            Set pubRequiredCred = grabRegistrationForm(req, st, publicCredRequiredFromDefaultSt);
            st.setPublicRequiredCredentials(pubRequiredCred);

            //public optional credentials
            Set publicCredOptionalFromDefaultSt =defaultSt.getPublicOptionalCredentials();
            Set pubOptionalCred = grabRegistrationForm(req, st, publicCredOptionalFromDefaultSt);
            st.setPublicOptionalCredentials(pubOptionalCred);

            //public optional credentials
            Set privateCredOptionalFromDefaultSt =defaultSt.getPrivateOptionalCredentials();
            Set privOptionalCred = grabRegistrationForm(req, st, privateCredOptionalFromDefaultSt);
            st.setPrivateOptionalCredentials(privOptionalCred);


            return st;
    }
    
    /**
	 * build a set of credentials by grabbing data from HttpServletRequest.
	 * @param req HttpServletRequest
	 * @param st SubjectTemplate
	 * @param credentialsFromDefaultSt
	 * @return Set of {@link JGuardCredential}
	 */
	private static Set grabRegistrationForm(HttpServletRequest req, SubjectTemplate st, Set credentialsFromDefaultSt) {
		Iterator itCredentials = credentialsFromDefaultSt.iterator();
		Set credSet = new HashSet();
		while(itCredentials.hasNext()){
			JGuardCredential jcredFromDefault = (JGuardCredential)itCredentials.next();

			//test if we've found the credential in the http request
			if(req.getParameter(jcredFromDefault.getName())!= null){
				JGuardCredential jcred = new JGuardCredential();
				jcred.setName(jcredFromDefault.getName());
				try{
     				jcred.setValue(req.getParameter(jcredFromDefault.getName()));
	    			credSet.add(jcred);
				}catch(IllegalArgumentException iae){
					logger.warn(" the property "+jcredFromDefault.getName()+" doesn't exist in the HttpServletRequest ");
					continue;
				}
			}

		}
		return credSet;
	}

	 /** load configuration from an XML file.
     * @param configurationLocation
     * @return Map containing filter configuration
     */
	private Map loadFilterConfiguration(String configurationLocation){
		Document doc = XMLUtils.read(configurationLocation);

		Element authentication = doc.getRootElement();
		Map filterSettings = new HashMap();
		if(authentication.element(HttpConstants.REGISTER_PROCESS_URI)!=null){
			filterSettings.put(HttpConstants.REGISTER_PROCESS_URI,authentication.element(HttpConstants.REGISTER_PROCESS_URI).getTextTrim());
		}
		if(authentication.element(HttpConstants.REGISTER_URI)!=null){
			filterSettings.put(HttpConstants.REGISTER_URI,authentication.element(HttpConstants.REGISTER_URI).getTextTrim());
		}

		filterSettings.put(HttpConstants.AUTH_SCHEME,authentication.element(HttpConstants.AUTH_SCHEME).getTextTrim());
		Element loginElement = authentication.element(HttpConstants.LOGIN_FIELD);
		if (loginElement != null){
			filterSettings.put(HttpConstants.LOGIN_FIELD,loginElement.getTextTrim());
		}
		Element passwordElement = authentication.element(HttpConstants.PASSWORD_FIELD);
		if (passwordElement != null){
			filterSettings.put(HttpConstants.PASSWORD_FIELD,passwordElement.getTextTrim());
		}

		
		return filterSettings;
	}
	
	
	/**
	 * @param filterSettings Map which contains filter options
	 * @throws ServletException
	 */
	private void setSettings(Map settings) {
        
        registerProcessPermission = new URLPermission(HttpConstants.REGISTER_PROCESS_URI,(String)settings.get(HttpConstants.REGISTER_PROCESS_URI));
        registerURI =(String)settings.get(HttpConstants.REGISTER_URI);
        registerPermission = new URLPermission(HttpConstants.REGISTER_URI,registerURI);
        
	}


public Permissions getGrantedPermissions() {
		
		Permissions alwaysGrantedPermissions = new Permissions();
		alwaysGrantedPermissions.add(getRegisterPermission());
		alwaysGrantedPermissions.add(getRegisterProcessPermission());
		return alwaysGrantedPermissions;
	}

}
