/*
jGuard is a security framework based on top of jaas (java authentication and authorization security).
it is written for web applications, to resolve simply, access control problems.
version $Name$
http://sourceforge.net/projects/jguard/

Copyright (C) 2004  Charles GAY

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


jGuard project home page:
http://sourceforge.net/projects/jguard/

*/

package net.sf.jguard.jee.authentication.http;

import java.io.IOException;
import java.util.Map;

import javax.security.auth.callback.CallbackHandler;
import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import net.sf.jguard.core.CoreConstants;
import net.sf.jguard.core.PolicyEnforcementPoint;
import net.sf.jguard.core.authentication.AccessContext;
import net.sf.jguard.core.authentication.AuthenticationUtils;
import net.sf.jguard.core.authentication.Stateful;
import net.sf.jguard.core.authorization.AuthorizationBindings;
import net.sf.jguard.core.authorization.permissions.URLPermission;
import net.sf.jguard.core.filters.PolicyEnforcementPointFilter;
import net.sf.jguard.jee.authentication.callbacks.HttpServletCallbackHandler;
import net.sf.jguard.jee.authorization.http.HttpAccessControllerUtils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 * @author <a href="mailto:diabolo512@users.sourceforge.net">Charles Gay</a>
 */
public class HttpServletAuthenticationBindings extends AbstractJEEAuthenticationBindings implements Stateful{
    
    private static final Logger logger = LoggerFactory.getLogger(HttpServletAuthenticationBindings.class.getName());
    private String indexURI;
    private String logonURI;
    
    /**
     * Creates a new instance of HttpServletAuthenticationBindings
     */
    public HttpServletAuthenticationBindings(AuthorizationBindings authZBindings) {
    	super(authZBindings);
    }

  
    /**
     * decorate the HttpServletRequest with a {@link JGuardServletRequestWrapper}.
     */
    public void process(AccessContext context) {
    	HttpServletRequest request = (HttpServletRequest)context.getAttribute(AccessFilter.SERVLET_REQUEST);
        context.setAttribute(AccessFilter.SERVLET_REQUEST, new JGuardServletRequestWrapper(request));
    }

     
    /**
     * 
     * @param context 
     * @return true if authentication failure must propagate, false otherwise
     */
    public boolean authenticationFailed(AccessContext context) {
            context.setAttribute(PolicyEnforcementPointFilter.REDIRECT, "true");
            HttpServletRequest request = (HttpServletRequest)context.getAttribute(AccessFilter.SERVLET_REQUEST);
	    HttpServletResponse response = (HttpServletResponse)context.getAttribute(AccessFilter.SERVLET_RESPONSE);
	    
		if(response.isCommitted()){
			logger.warn(" response is already committed ");
		}

		if(HttpConstants.BASIC_AUTH.equalsIgnoreCase(authScheme)){
			  HttpServletCallbackHandler.buildBasicChallenge(response,(String)(request).getSession(true).getServletContext().getAttribute(CoreConstants.APPLICATION_NAME));
			  return false;
		 }
		

	   //an URL for authentication failure event has been set
	   if(authenticationFailedPermission!=null && !((URLPermission)authenticationFailedPermission).getURI().equals("")){
            try {
                response.sendRedirect(response.encodeRedirectURL(request.getContextPath()+((URLPermission)authenticationFailedPermission).getURI()));
                logger.debug("authentication failed redirect to "+((URLPermission)authenticationFailedPermission).getURI());
            } catch (IOException ex) {
               logger.error(ex.getMessage(),ex);
               return false;
            }
               logger.debug(" NOT BASIC AUTHENTICATION - user is not authenticated  redirect to "+request.getContextPath()+((URLPermission)authenticationFailedPermission).getURI());
		   
	   }else{
              authZbindings.accessDenied(context);
	   }
	   return true;
	}

        /**
         * @param context
         * @return true of <i>authenticationSucceed</i> redirect succeed (double succeed ;-) )
         * false otherwise.
         */
	public boolean authenticationSucceed(AccessContext context) {
            context.setAttribute(PolicyEnforcementPointFilter.REDIRECT, "true");
            HttpServletRequest request = (HttpServletRequest)context.getAttribute(AccessFilter.SERVLET_REQUEST);
	    HttpServletResponse response = (HttpServletResponse)context.getAttribute(AccessFilter.SERVLET_RESPONSE);
		String redirectURI = indexURI;
		URLPermission lastAccessDeniedPermission = (URLPermission)getSessionAttribute(context,CoreConstants.LAST_ACCESS_DENIED_PERMISSION);
		String lastAccessDeniedURI;
		if(lastAccessDeniedPermission==null){
			lastAccessDeniedURI = logonURI;
		}else{
			lastAccessDeniedURI= lastAccessDeniedPermission.getURI();
		}
	    
	    //for security reasons(man-in-the-middle attack for sessionID cookie),
	    //we remove the old session if authentication is successfull
	    //and create a new session
	    //to not lost the subject, we grab in the old session HttpAuthenticationUtils 
	    //and put it in the new session
	    AuthenticationUtils authenticationUtils = getAuthenticationUtils(context);
	    
	    //we test if response is committed, because
	    //when user register, it authenticate twice
	    // => the response is committed at the first step
	    if(!response.isCommitted()){
		    setSessionAttribute(context,CoreConstants.AUTHN_UTILS,authenticationUtils);
	    }
	    
	    //we redirect to the last 'access denied' URI before authentication
	    if(lastAccessDeniedURI!=null &&!"".equals(lastAccessDeniedURI)) {
	    	if(goToLastAccessDeniedUriOnSuccess){
	    		redirectURI = lastAccessDeniedURI;
                }else {
			if(!HttpAccessControllerUtils.hasPermission(request,indexPermission)){
                            redirectURI = logonURI;
                        }

	    	}
	    }


	    logger.debug(" user is authenticated "," redirect to "+redirectURI);
	    if(!response.isCommitted()){
            try {
                response.sendRedirect(response.encodeRedirectURL(request.getContextPath()+redirectURI));
            } catch (IOException ex) {
               logger.error(ex.getMessage(),ex);
               return false;
            }
	    }
		return true;
	}
    

	 	/**
		 * @param filterSettings Map which contains filter options
		 * @throws ServletException
		 */
		protected void init(Map settings) {
	        indexURI = (String)settings.get(HttpConstants.INDEX_URI);
	        indexPermission = new URLPermission("indexURI",indexURI);
	        authenticationFailedPermission = new URLPermission("authenticationFailedURI",(String)settings.get(HttpConstants.AUTHENTICATION_FAILED_URI));
	        logonProcessPermission = new URLPermission("logonProcessURI",(String)settings.get(HttpConstants.LOGON_PROCESS_URI));
	        String logonURIStr = (String)settings.get(HttpConstants.LOGON_URI);
	        logonPermission = new URLPermission("logonURI",logonURIStr);
	    
	        logoffPermission = new URLPermission("logoffURI",(String)settings.get(HttpConstants.LOGOFF_URI));
	        
	        //grab the authentication scheme authorized
	        authScheme = (String)settings.get(HttpConstants.AUTH_SCHEME);
	        HttpServletCallbackHandler.setAuthSchemes(authScheme);
	        authSchemes = AuthSchemesHelper.validateAuthScheme(authScheme);
	        if(authSchemes.contains(HttpConstants.FORM_AUTH)){
	        	HttpServletCallbackHandler.setLoginField((String)settings.get(HttpConstants.LOGIN_FIELD));
	        	HttpServletCallbackHandler.setPasswordField((String)settings.get(HttpConstants.PASSWORD_FIELD));
	        }
	        
	        // grab goToLastAccessDeniedUriOnSuccess value
	        String tmpValue = (String)settings.get(HttpConstants.GO_TO_LAST_ACCESS_DENIED_URI_ON_SUCCESS);
	        if("false".equalsIgnoreCase(tmpValue) || "no".equalsIgnoreCase(tmpValue)){
	        	goToLastAccessDeniedUriOnSuccess = false;
	        }
	        
		}

	public CallbackHandler getCallbackHandler(AccessContext context) {
		HttpServletRequest request = (HttpServletRequest)context.getAttribute(AccessFilter.SERVLET_REQUEST);
	    HttpServletResponse response = (HttpServletResponse)context.getAttribute(AccessFilter.SERVLET_RESPONSE);
		HttpServletCallbackHandler cbh = new HttpServletCallbackHandler(request,response);
		return cbh;
	}
	
	public Object getSessionAttribute(AccessContext context,String key) {
		HttpServletRequest request = (HttpServletRequest)context.getAttribute(AccessFilter.SERVLET_REQUEST);
    	HttpSession session = request.getSession(true);
		return session.getAttribute(key);
	}

	public void setSessionAttribute(AccessContext context,String key, Object value) {
		HttpServletRequest request = (HttpServletRequest)context.getAttribute(AccessFilter.SERVLET_REQUEST);
    	HttpSession session = request.getSession(true);
    	session.setAttribute(key, value);
	}

	public void removeSessionAttribute(AccessContext context, String key) {
		HttpServletRequest request = (HttpServletRequest)context.getAttribute(AccessFilter.SERVLET_REQUEST);
		HttpSession session = request.getSession(true);
		session.removeAttribute(key);
	}
	
	public void removeApplicationAttribute(AccessContext context, String key) {
		HttpServletRequest request = (HttpServletRequest)context.getAttribute(AccessFilter.SERVLET_REQUEST);
		ServletContext servletContext = request.getSession(true).getServletContext();
		servletContext.removeAttribute(key);
	}
	
	public void removeRequestAttribute(AccessContext context, String key) {
		HttpServletRequest request = (HttpServletRequest)context.getAttribute(AccessFilter.SERVLET_REQUEST);
		request.removeAttribute(key);
	}

	public void setRequestAttribute(AccessContext context, String key,Object value) {
		HttpServletRequest request = (HttpServletRequest)context.getAttribute(AccessFilter.SERVLET_REQUEST);
		request.setAttribute(key, value);
	}

	public void setApplicationAttribute(AccessContext context, String key,
			Object value) {
		HttpServletRequest request = (HttpServletRequest)context.getAttribute(AccessFilter.SERVLET_REQUEST);
		HttpSession session = request.getSession(true);
		session.getServletContext().setAttribute(key, value);
	}

	public Object getRequestAttribute(AccessContext context, String key) {
		HttpServletRequest request = (HttpServletRequest)context.getAttribute(AccessFilter.SERVLET_REQUEST);
		return request.getAttribute(key);
	}

	public Object getApplicationAttribute(AccessContext context, String key) {
		HttpServletRequest request = (HttpServletRequest)context.getAttribute(AccessFilter.SERVLET_REQUEST);
		ServletContext servletContext = request.getSession(true).getServletContext();
		return servletContext.getAttribute(key);
	}


	public boolean isStateful() {
		return true;
	}


	public void invalidateSession(AccessContext context) {
		HttpServletRequest request = (HttpServletRequest)context.getAttribute(AccessFilter.SERVLET_REQUEST);
		HttpSession session = request.getSession(false);
		if(session!=null){
			session.invalidate();
		}
	}

}
