/*
jGuard is a security framework based on top of jaas (java authentication and authorization security).
it is written for web applications, to resolve simply, access control problems.
version $Name:  $
http://sourceforge.net/projects/jguard/

Copyright (C) 2004  Charles GAY

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


jGuard project home page:
http://sourceforge.net/projects/jguard/

*/
package net.sf.jguard.core.provisioning;


import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.logging.Logger;

import javax.security.auth.Subject;

import net.sf.jguard.core.authentication.AuthenticationException;
import net.sf.jguard.core.authentication.credentials.JGuardCredential;
import net.sf.jguard.core.organization.Organization;

/**
 * template used to validate user registration and to build the corresponding
 * Subject object.
 * @author <a href="mailto:diabolo512@users.sourceforge.net">Charles Gay</a>
 * @author <a href="mailto:tandilero@users.sourceforge.net">Maximiliano Batelli</a>
 *
 */
public class SubjectTemplate extends EntityTemplate {

    private static final Logger logger = Logger.getLogger(SubjectTemplate.class.getName());
   
    private Set missingPrivateRequiredCred;
    private Set missingPublicRequiredCred;
    private Set unknownPrivateOptionalCred;
    private Set unknownPublicOptionalCred;
    private Set privateRequiredCredentials;
    private Set publicRequiredCredentials;
    private Set publicOptionalCredentials;
    private Set privateOptionalCredentials;

    
    private Long id;
    //principals created during registration
    private Set principals;

	/**
	 * constructor.
	 */
	public SubjectTemplate(){
            super();
            //data used for all the users during registration
            missingPrivateRequiredCred = new HashSet();
            missingPublicRequiredCred = new HashSet();
            unknownPrivateOptionalCred = new HashSet();
            unknownPublicOptionalCred= new HashSet();
            privateOptionalCredentials = new HashSet();
            privateRequiredCredentials = new HashSet();
            publicOptionalCredentials = new HashSet();
            publicRequiredCredentials = new HashSet();
            principals = new HashSet();	
	}

	

	/**
	 * remove unknown credentials and return missing credentials set.
	 * @param user
	 * @return mssing credentials
	 * @throws AuthenticationException
	 */
	public Set validateRequiredCredentialsFromUser(Subject user)throws AuthenticationException{
		Set missingCredentials = new HashSet();
		Set credentialsNotRegistered = null;
		//we remove unknown credentials
		Set userPublicRequiredCredentials = getCredentials(getPublicRequiredCredentials(), user.getPublicCredentials(JGuardCredential.class));
		Set missingPublicCredentials =  validateCredentialSet(getPublicRequiredCredentials(), userPublicRequiredCredentials);
		missingCredentials.addAll(missingPublicCredentials);

		Set userPublicOptionalCredentials = getCredentials(getPublicOptionalCredentials(), user.getPublicCredentials(JGuardCredential.class));
		credentialsNotRegistered = filterCredentialSet(getPublicOptionalCredentials(), userPublicOptionalCredentials);
		if(credentialsNotRegistered.size()>0){
		  logger.warning(" there are some unknown credentials filled by the user during the registration process: ");
		  logger.warning(credentialsNotRegistered.toString());
		}
		

		Set userPrivateRequiredCredentials = getCredentials(getPrivateRequiredCredentials(), user.getPrivateCredentials(JGuardCredential.class));
		Set missingPrivateCredentials = validateCredentialSet(getPrivateRequiredCredentials(), userPrivateRequiredCredentials);
		missingCredentials.addAll(missingPrivateCredentials);
		
		Set userPrivateOptionalCredentials = getCredentials(getPrivateOptionalCredentials(), user.getPrivateCredentials(JGuardCredential.class));
		credentialsNotRegistered = filterCredentialSet(getPrivateOptionalCredentials(), userPrivateOptionalCredentials);
		if(credentialsNotRegistered.size()>0){
		  logger.warning(" there are some unknown credentials filled by the user during the registration process: ");
		  logger.warning(credentialsNotRegistered.toString());
		}
		
		return missingCredentials;
	}


	
	/**
	 * build a Subject from a SubjectTemplate.
	 * @param user SubjectTemplate used to build the Subject object
	 * @return subject built
	 */
	public Subject toSubject(SubjectTemplate user,Organization organization){

		Set principalsForRegisteredUsers = new HashSet();
		principalsForRegisteredUsers.addAll((getPrincipals()));
                
                //add to the user its organization
                principalsForRegisteredUsers.add(organization);

		Set publicCredentials = user.getPublicCredentials();
		Set privateCredentials = user.getPrivateCredentials();

		Subject subject = new Subject(false,principalsForRegisteredUsers,publicCredentials,privateCredentials);
		return subject;
	}
        
        

	/**
	 * build a Subject from a validated SubjectTemplate.
	 * @return subject built
	 */
	public Subject toSubject(Organization organization){
		return toSubject(this,organization);
	}
        
        public final Set getMissingPublicRequiredCred() {
		return missingPublicRequiredCred;
	}

	public final Set getUnknownPrivateOptionalCred() {
		return unknownPrivateOptionalCred;
	}
	public final Set getUnknownPublicOptionalCred() {
		return unknownPublicOptionalCred;
	}
	 /**
	 * implements a deep copy of object
	 * @see java.lang.Object#clone()
	 */
	public Object clone() throws CloneNotSupportedException{

		SubjectTemplate clone = new SubjectTemplate();
		clone.setPrincipals(new HashSet(principals));
		clone.setPrivateOptionalCredentials(JGuardCredential.cloneCredentialsSet(getPrivateOptionalCredentials()));
		clone.setPrivateRequiredCredentials(JGuardCredential.cloneCredentialsSet(getPrivateRequiredCredentials()));
		clone.setPublicOptionalCredentials(JGuardCredential.cloneCredentialsSet(getPublicOptionalCredentials()));
		clone.setPublicRequiredCredentials(JGuardCredential.cloneCredentialsSet(getPublicRequiredCredentials()));
		return clone;
        }

/**
	 * return a read-only SubjectTemplate.
	 * this method is inspired from the
	 * Collections.unmodifiableCollection(Collection c), which
	 * is part of the JDK.
	 * @return read-only SubjectTemplate
	 */
	public SubjectTemplate unmodifiableSubjectTemplate() throws CloneNotSupportedException{
            
        SubjectTemplate readOnly = (SubjectTemplate)this.clone();
        
        //principal stuff
        readOnly.principals = (Collections.unmodifiableSet(principals));
 
        //credential stuff
        readOnly.setPrivateOptionalCredentials((Collections.unmodifiableSet(getPrivateOptionalCredentials())));
        readOnly.setPrivateRequiredCredentials((Collections.unmodifiableSet(getPrivateRequiredCredentials())));
        readOnly.setPublicOptionalCredentials(((Collections.unmodifiableSet(getPublicOptionalCredentials()))));
        readOnly.setPublicRequiredCredentials((Collections.unmodifiableSet(getPublicRequiredCredentials())));

		return readOnly;
	}
         
         
        public final Set getMissingPrivateRequiredCred() {
		return missingPrivateRequiredCred;
	}
         
        public void setPrivateRequiredCredentials(Set privateCredentials) {
                this.privateRequiredCredentials = privateCredentials;
	}

	public void setPublicRequiredCredentials(Set publicCredentials) {
                this.publicRequiredCredentials = publicCredentials;
	}
        
        public Set getPrivateOptionalCredentials() {
		return privateOptionalCredentials;
	}
	public void setPrivateOptionalCredentials(Set privateOptionalCredentials) {
		this.privateOptionalCredentials = privateOptionalCredentials;
	}
	public Set getPublicOptionalCredentials() {
		return publicOptionalCredentials;
	}
	public void setPublicOptionalCredentials(Set publicOptionalCredentials) {
		this.publicOptionalCredentials =  publicOptionalCredentials;
	}
        
         public Set getPublicRequiredCredentials() {
		return publicRequiredCredentials;
	}

	public Set getPrivateRequiredCredentials() {
		return privateRequiredCredentials;
	}
        
        public Set getPublicCredentials(){
		Set publicCredentials = getPublicOptionalCredentials();
		publicCredentials.addAll(getPublicRequiredCredentials());
		return publicCredentials;
	}
        

	public Set getPrivateCredentials(){
		Set privateCredentials = getPrivateOptionalCredentials();
		privateCredentials.addAll(getPrivateRequiredCredentials());
		return privateCredentials;

	}
        
        public Set getRequiredCredentials(){
            Set requiredCredentials = new HashSet(getPublicRequiredCredentials());
            requiredCredentials.addAll(getPrivateRequiredCredentials());
            return requiredCredentials;
	}
                
                /**
	 * validate the EntityTemplate candidate.
	 * @param user the subjectTemplate which is candidate
	 * to be transformed into a Subject object.
	 * @throws AuthenticationException when the user does not content mandatory fields
	 */
	public void validateTemplate(SubjectTemplate candidate)throws RegistrationException{

		if(candidate.getPrivateRequiredCredentials()==null){
			logger.warning("private required credentials set from user is null ");
			candidate.setPrivateRequiredCredentials(new HashSet());
		}
		if(candidate.getPrivateOptionalCredentials()==null){
			logger.warning("private optional credentials set from user is null ");
			candidate.setPrivateOptionalCredentials(new HashSet());
		}
		if(candidate.getPublicRequiredCredentials()==null){
			logger.warning("public required credentials set from user is null ");
			candidate.setPublicRequiredCredentials(new HashSet());
		}
		if(candidate.getPublicOptionalCredentials()==null){
			logger.warning("public optional credentials set from user is null ");
			candidate.setPublicOptionalCredentials(new HashSet());
		}

		missingPrivateRequiredCred =
			validateCredentialSet(getPrivateRequiredCredentials(),candidate.getPrivateRequiredCredentials());
		if(missingPrivateRequiredCred.size()>0){
			throw new RegistrationException("missing private credentials required :"+missingPrivateRequiredCred,new HashSet(),missingPrivateRequiredCred);
		}

		missingPublicRequiredCred =
			validateCredentialSet(getPublicRequiredCredentials(),candidate.getPublicRequiredCredentials());
		if(missingPrivateRequiredCred.size()>0){
			throw new RegistrationException("missing public credentials required :"+missingPublicRequiredCred,missingPublicRequiredCred,new HashSet());
		}

		unknownPrivateOptionalCred =
			filterCredentialSet(getPrivateOptionalCredentials(),candidate.getPrivateOptionalCredentials());
		if(unknownPrivateOptionalCred.size()>0){
                    logger.warning(" user has filled unknown optional private credentials :");
                    logger.warning(unknownPrivateOptionalCred.toString());
		}
		unknownPublicOptionalCred =
			filterCredentialSet(getPublicOptionalCredentials(),candidate.getPublicOptionalCredentials());
		if(unknownPublicOptionalCred.size()>0){
			logger.warning(" user has filled unknown optional public credentials :");
			logger.warning(unknownPublicOptionalCred.toString());
		}
	}

    public Long getId() {
        return id;
    }

    public void setId(Long id) {
        this.id = id;
    }

    
    public Set getPrincipals() {
            return principals;
    }
    /**
     * defined the principals automatically granted to the registered user.
     * @param principals
     */
    public void setPrincipals(Set principals) {
            this.principals = principals;
    }

}
