/*
jGuard is a security framework based on top of jaas (java authentication and authorization security).
it is written for web applications, to resolve simply, access control problems.
version $Name$
http://sourceforge.net/projects/jguard/

Copyright (C) 2004  Charles GAY

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


jGuard project home page:
http://sourceforge.net/projects/jguard/

*/
package net.sf.jguard.core.principals;
import java.io.Serializable;
import java.security.Principal;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

import javax.security.auth.Subject;

import net.sf.jguard.core.authentication.credentials.JGuardCredential;
import net.sf.jguard.core.organization.Organization;

/**
 * UserPrincipal is used to resolve ABAC permissions.
 * @author <a href="mailto:vberetti@users.sourceforge.net">Vincent Beretti</a>
 * @author <a href="mailto:diabolo512@users.sourceforge.net">Charles Gay</a>
 */
public class UserPrincipal implements Principal,Serializable{

	
	private static final String NO_NAME_FOUND = "NO NAME FOUND";
	private static final long serialVersionUID = 9075426017744650798L;
	private Logger logger = Logger.getLogger(UserPrincipal.class.getName());
	private String name = "NO NAME DEFINED";
	private Subject subject;

	public UserPrincipal(Subject subject){
		this.subject = subject;
	}
        

	public void setName(String name) {
		this.name = name;
		
	}

	public Map getPrivateCredentials() {
		
		Set privateCredentials = subject.getPrivateCredentials();
		Map pCredentials = transformCredentialSetIntoMap(privateCredentials);
		return pCredentials;
	}

	private Map transformCredentialSetIntoMap(Set credentials) {
		Map pCredentials = new HashMap();
		Iterator privateCIterator = credentials.iterator();
		while(privateCIterator.hasNext()){
			Object credential = privateCIterator.next();
			if(credential instanceof JGuardCredential){
				JGuardCredential jcred = (JGuardCredential)credential;
				if(!pCredentials.containsKey(jcred.getName())){
					Collection values = new HashSet();
					values.add(jcred.getValue());
					pCredentials.put(jcred.getName(),values);	
				}else{
					Collection valuesStored = (Collection) pCredentials.get(jcred.getName());
					valuesStored.add(jcred.getValue());
				}
				
			}
		}
		return pCredentials;
	}

	public Map getPublicCredentials() {
		Set publicCredentials = subject.getPublicCredentials();
		Map pCredentials = transformCredentialSetIntoMap(publicCredentials);
		return pCredentials;
	}

	/**
	 * @return the value of a credential present in the public credentials ' set
	 * of the Subject, if its 'id' is <i>"name"</i>. 
	 * @see net.sf.jguard.core.authentication.credentials.JGuardCredential
	 */
	public String getName() {
		//we cannot add a more significant method to avoid infinite loop
        return NO_NAME_FOUND;
	}
	
	/**
	 * compare two SubjectAsPrincipal objects(compare their Subject).
	 * @param subjAsPrincipal
	 * @return true if the contained Subject is equals to the one contained
	 * in the SubjectAsPrincipal instance as parameter;otherwise, false.
	 */
	public boolean equals (Object object){
		UserPrincipal userPrincipal;
		if (object instanceof UserPrincipal) {
			userPrincipal = (UserPrincipal) object;
			if(getPrincipals().equals(userPrincipal.getPrincipals())){
				return true;
			}
			// we cannot include credentials in this method to avoid class circularity error 
		}
			return false;
		
	}
	
	/**
	 * return principals present in the subject except userPrincipals
	 * to avoid infinite loop if we look into principals recursively.
	 * @return 
	 */
	protected Map getPrincipals(){
		
		//we filter userprincipal
		Set principals = subject.getPrincipals();
		Set filteredSet = new HashSet();
		Iterator it = principals.iterator();
		while(it.hasNext()){
			Principal principal = (Principal)it.next();
			if (!(principal instanceof UserPrincipal)) {
				filteredSet.add(principal);
			}
		}
		
		//we transform set into map for jexl
		Map ppals = new HashMap();

		Iterator itFiletedPrincipals = filteredSet.iterator();
		while (itFiletedPrincipals.hasNext()){
			Principal principal = (Principal)itFiletedPrincipals.next();
			ppals.put(principal.getName(), principal);
		}

		return ppals;
	}	
	
	
	/**
	 * return {@link RolePrincipal} present in subject. 
	 * @return 
	 */
	public Map getRoles(){
		return getSpecificPrincipals(RolePrincipal.class);
	}	
        
        /**
	 * return {@link RolePrincipal} present in subject. 
	 * @return 
	 */
	public Organization getOrganization(){
		 Set organizationSet = subject.getPrincipals(Organization.class);
                 if(organizationSet.size()!=1){
                     throw new IllegalStateException(" a UserPrincipal object can contains only one organization. if no one is set, the default 'system' organization is used ");
                 }
                 return (Organization) organizationSet.iterator().next();
	}
	
	
	private Map getSpecificPrincipals(Class principalSubclass){
		Set principals = subject.getPrincipals(principalSubclass);
		
		//we transform set into map for jexl
		Map ppals = new HashMap();

		Iterator itPrincipals = principals.iterator();
		while (itPrincipals.hasNext()){
			Principal principal = (Principal)itPrincipals.next();
			ppals.put(principal.getName(), principal);
		}
		return ppals;
	}

	public int compareTo(Object o) {
		UserPrincipal principal = (UserPrincipal)o;
        if (this.equals(o)){
            return 0;
        }

        return this.getName().compareTo(principal.getName());

	}
	
	public String toString(){
		StringBuffer sb = new StringBuffer();
		sb.append("UserPrincipal ");
		sb.append(name);
		sb.append(this.hashcode());
		return sb.toString();
	}
	
	public int hashcode(){
		return getRoles().hashCode()
                +getOrganization().hashCode()
		+getPublicCredentials().hashCode()
		+getPrivateCredentials().hashCode()+45;
	}
}



