/*
jGuard is a security framework based on top of jaas (java authentication and authorization security).
it is written for web applications, to resolve simply, access control problems.
version $Name$
http://sourceforge.net/projects/jguard/

Copyright (C) 2004  Charles GAY

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


jGuard project home page:
http://sourceforge.net/projects/jguard/

*/
package net.sf.jguard.core.principals;

import java.security.Permission;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import net.sf.jguard.core.authorization.permissions.JGPermissionCollection;
import net.sf.jguard.core.organization.Organization;


/**
 * This Principal represents the notion of a role.
 * it is the link between Authentication and Authorization parts.
 * @author <a href="mailto:diabolo512@users.sourceforge.net">Charles Gay</a>
 * @author <a href="mailto:vinipitta@users.sourceforge.net">VinÃ­cius Pitta Lima de Araujo</a>
 * @author <a href="mailto:tandilero@users.sourceforge.net">Maximiliano Batelli</a>
 */
public class RolePrincipal implements BasePrincipal{


    /**
     * serial version id.
     */
    private static final long serialVersionUID = 3761412993065431095L;

    private String localName="";

    //indicate to which application this role apply
    private String applicationName = DEFAULT_APPLICATION_NAME;
    //permissions bound to a domain owned by this Principal
    private Set permissionsFromDomains;
    //all the permissions (bound or not bound to a domain owned by this Principal)
    private Set permissions;
    //permisisons not bound to a Domain owned by this Principal.
    private Set orphanedPermissions;
    //domains owned to this principal
    private Set domains;
    private boolean active = true;
    private String definition = "true";
    private Organization organization = null;
    private Long id;
    
    /**
     * All principals that this role inherites from. This property is use to implement the General Hierarchy
     * proposed by the NIST.
     */
    private Set descendants;

    /**
     * empty constructor.
     */
    public RolePrincipal(){
        permissions = new HashSet();
        domains = new HashSet();
        orphanedPermissions = new HashSet();
        permissionsFromDomains = new HashSet();
        descendants = new HashSet();
       
    }

    /**
     * constructor.
     * @param name
     */
    public RolePrincipal(String name){
        permissions = new HashSet();
        this.setName(name);
        domains = new HashSet();
        orphanedPermissions = new HashSet();
        permissionsFromDomains = new HashSet();
        descendants = new HashSet();
       
    }


    /**
     * constructor.
     * @param localName
     * @param applicationName
     */
    public RolePrincipal(String localName,String  applicationName){
        permissions = new HashSet();
        this.localName = localName;
        this.applicationName = applicationName;
        domains = new HashSet();
        orphanedPermissions = new HashSet();
        permissionsFromDomains = new HashSet();
        descendants = new HashSet();
        
    }
    
    /**
     * constructor.
     * @param localName
     * @param applicationName
     */
    public RolePrincipal(String localName,String  applicationName,Organization organizationOwner){
        permissions = new HashSet();
        this.localName = localName;
        this.applicationName = applicationName;
        domains = new HashSet();
        orphanedPermissions = new HashSet();
        permissionsFromDomains = new HashSet();
        descendants = new HashSet();
        organization = organizationOwner;
    }
    
    /**
     * override the java.lang.Object 's <i>clone</i> method.
     * @return new JGuardPricipal with the same internal references (permissions and Domains).
     */
    public Object clone(){
       RolePrincipal clone  = new RolePrincipal(this.localName,this.applicationName);
       clone.setDomains(new HashSet(this.domains));
       clone.setPermissions(new HashSet(this.permissions));
       clone.setDescendants(new HashSet(this.descendants));
        return clone;

    }
	/**
         * return  the unique name formed by <i>localname#applicationName.</i>
	 * @see java.security.Principal#getName()
	 */
	public String getName() {
		return getName(localName, applicationName);
	}
	
	
	/**
	 * @see java.security.Principal#getName()
	 */
	public static String getName(String localName) {
		return getName(localName, "*");
	}
	/**
	 * @see java.security.Principal#getName()
	 */
	public static String getName(String localName,String applicationName) {
		StringBuffer sb = new StringBuffer(20);
		sb.append(applicationName);
		sb.append("#");
		sb.append(localName);
		return sb.toString();
	}

    /**
     * compare an object to this RolePrincipal.
     * override the Object's <i>equals</i> method.
     * @param another object to compare
     * @return true if another is equals to this RolePrincipal
     */
    public boolean equals(Object another){

        if(another instanceof RolePrincipal){
            RolePrincipal principal = (RolePrincipal)another;
            if(principal.getName().equals(this.getName())){
                return true;
            }
        }

        return false;
    }

    /**
     * override the Object's <i>toString</i> method.
     * return String representation
     */
    public String toString(){
        StringBuffer sb = new StringBuffer();
        sb.append(" principal class name =");
        sb.append(getClass().getName());
        sb.append("\n");
        sb.append(" principal localName =");
        sb.append(localName);
        sb.append("\n");
        sb.append(" principal application name =");
        sb.append(applicationName);
        sb.append("\n");
        sb.append(" organization owner =");
        sb.append(getOrganization());
        sb.append("\n");
        sb.append(" principal domains =");
        sb.append(domains);
        sb.append("\n");
        sb.append(" principal permissions =");
        sb.append(permissions);
        sb.append("\n");
        sb.append(" principal descendants =");
        sb.append(descendants);
        sb.append("\n");
        return sb.toString();
    }

    /**
     * override Object's <i>hashcode</i> method.
     */
    public int hashCode(){
        int i =0;
         i+=localName.hashCode()+applicationName.hashCode();
         if(organization!=null){
             i+=organization.hashCode();
         }
         return i;
    }
	/**
	 * @param string
	 */
	public void setName(String string) {
		String[] tokens = string.split("#");
		if(tokens.length==1){
			applicationName = "*";
			localName = tokens[0];
		}else if(tokens.length==2){
			applicationName = tokens[0];
			localName = tokens[1];
			
		}else{
			throw new IllegalArgumentException(" name is composed of applicationName#localName");
		}
		
		
		
		
	}

	/**
	 * return all permissions owned by this Principal plus
	 * permissions inherited from descendants.
	 * @return permissions
	 */
	public Set getAllPermissions() {
		//all permissions owned by this principal
		Set allPermissions = new HashSet();
        allPermissions.addAll(permissions);

        //get inherited permissions
        for (Iterator it = descendants.iterator(); it.hasNext(); ) {
            allPermissions.addAll(((RolePrincipal) it.next()).getAllPermissions());
        }

		return allPermissions;
	}

    /**
     * return the permissions bounded to a domain plus orphanedPermisions.
     * @return
     */
    public Set getPermissions(){
        return permissions;
    }

    /**
     * return the permissions not bound to a domain owned by this RolePrincipal.
     * @return
     */
    public Set getOrphanedPermissions(){
        return orphanedPermissions;
    }

	/**
	 * @param perms
	 */
	public void setPermissions(Set perms) {
        Iterator itPermissionsSet = perms.iterator();
        while(itPermissionsSet.hasNext()){
          Permission perm = (Permission)itPermissionsSet.next();
          addPermission(perm);
        }
	}
    /**
     * add a permission to the RolePrincipal.
     * @param permission
     */
    public void addPermission(Permission permission){
        permissions.add(permission);
        Iterator itDomains = domains.iterator();
        boolean orphan = true;
        while(itDomains.hasNext()){
        	JGPermissionCollection temp = (JGPermissionCollection)itDomains.next();
        	if(temp.containsPermission(permission)){
        		orphan = false;
        		break;
        	}
        }
        if(orphan == true){
            orphanedPermissions.add(permission);
        }else{
            permissionsFromDomains.add(permission);
        }
    }

    /**
     * add an Domain to the RolePrincipal.
     * @param domain
     */
    public void addDomain(JGPermissionCollection domain){
        domains.add(domain);
        Set permissionsDomain = domain.getPermissions();
        //we remove from orphanedPermissions the permissions bound to this Domain
        orphanedPermissions.removeAll(permissionsDomain);
        //we add the permissions owned by domain to permissionsFromDomains and permissions
        permissionsFromDomains.addAll(permissionsDomain);
        permissions.addAll(permissionsDomain);
    }


	/**
	 * @return application name
	 */
	public String getApplicationName() {
		return applicationName;
	}

	/**
     * define application name.
	 * @param string
	 */
	public void setApplicationName(String string) {
		applicationName = string;
	}

    /**
     * @return Returns the domains.
     */
    public Set getDomains() {
        return domains;
    }


    /**
     * @return Returns the permissionsFromDomains.
     */
    public Set getPermissionsFromDomains() {
        return permissionsFromDomains;
    }

    /**
     * @param doms The domains to set.
     */
    public void setDomains(Set doms) {
        Iterator it = doms.iterator();
        while(it.hasNext()){
            JGPermissionCollection dom = (JGPermissionCollection)it.next();
            this.addDomain(dom);
        }

    }



    /**
     * method used to compare two objects.
     * this method is used in Collection to <strong>order</strong> items, and MUST be
     * consistent with the <i>equals</i> method (eache method should return 0/true in the same cases).
     * @param o object to compare
     * @see java.lang.Comparable#compareTo(java.lang.Object)
     * @return 0 if both objects are equals,
     * &lt;0 if 0 is lesser than the RolePrincipal,
     * &gt;0 if 0 is greater than the RolePrincipal
     * @see java.lang.Comparable#compareTo(java.lang.Object)
     */
    public int compareTo(Object o) {
        RolePrincipal principal = (RolePrincipal)o;
        if (this.equals(o)){
            return 0;
        }

        return this.getName().compareTo(principal.getName());
    }

    public void removeDomain(JGPermissionCollection domain){
        permissionsFromDomains.removeAll(domain.getPermissions());
        permissions.removeAll(domain.getPermissions());
        domains.remove(domain);

    }

	public Set getDescendants() {
		return descendants;
	}

	public void setDescendants(Set descendants) {
		this.descendants = descendants;
	}

	public boolean isActive() {
		return active;
	}

	public void setActive(boolean active) {
		this.active = active;
	}

	/**
	 * @return Returns definition.
	 */
	public String getDefinition() {
		return definition;
	}

	/**
	 * @param definition The definition to set.
	 */
	public void setDefinition(String definition) {
		this.definition = definition;
	}

	public String getLocalName() {
		return localName;
	}

	public void setLocalName(String localName) {
		this.localName = localName;
	}

    public Organization getOrganization() {
        return organization;
    }

    public void setOrganization(Organization organization) {
        this.organization = organization;
    }

    public Long getId() {
        return id;
    }

    public void setId(Long id) {
        this.id = id;
    }

   
}
