/*
jGuard is a security framework based on top of jaas (java authentication and authorization security).
it is written for web applications, to resolve simply, access control problems.
version $Name$
http://sourceforge.net/projects/jguard/

Copyright (C) 2004  Charles GAY

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


jGuard project home page:
http://sourceforge.net/projects/jguard/

*/
package net.sf.jguard.core.filters;

import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.security.Permission;
import java.security.PrivilegedExceptionAction;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.security.auth.Subject;
import net.sf.jguard.core.CoreConstants;
import net.sf.jguard.core.Filter;
import net.sf.jguard.core.FilterChain;
import net.sf.jguard.core.authentication.AccessContext;
import net.sf.jguard.core.authentication.AuthenticationBindings;
import net.sf.jguard.core.authentication.AuthenticationServicePoint;
import net.sf.jguard.core.authentication.AuthenticationUtils;
import net.sf.jguard.core.authentication.Stateful;
import net.sf.jguard.core.authorization.AuthorizationBindings;
import net.sf.jguard.core.authorization.PolicyDecisionPoint;
import net.sf.jguard.core.provisioning.ProvisioningServicePoint;

/**
*
* @author <a href="mailto:diabolo512@users.sourceforge.net">Charles Gay</a>
*/
public class PolicyEnforcementPointFilter implements Filter{
    
	private AuthenticationBindings authenticationBindings = null;
	private PolicyDecisionPoint policyDecisionPoint = null;
        private ProvisioningServicePoint provisioningServicePoint = null;
	private static final Logger logger = Logger.getLogger(PolicyEnforcementPointFilter.class.getName());
	
	private Permission logonProcessPermission = null;
	private Permission logoffPermission = null;
	private String authenticationScope;
	private String applicationName;
        public final static String REDIRECT ="redirect";
    
 public PolicyEnforcementPointFilter(Map options){

    	String authenticationBindingsImpl = (String)options.get(CoreConstants.AUTHENTICATION_BINDINGS);
    	String filterConfigurationLocation = (String)options.get(CoreConstants.CONFIGURATION_LOCATION);
    	authenticationScope = (String)options.get(CoreConstants.AUTHENTICATION_SCOPE);
    	applicationName = (String)options.get(CoreConstants.APPLICATION_NAME);
    	String authorizationBindingsImpl = (String)options.get(CoreConstants.AUTHORIZATION_BINDINGS);
    	//initialize PolicyDecisionPoint
	 this.policyDecisionPoint = initPolicyDecisionPoint(authorizationBindingsImpl);
    	
            //initialize authentication Bindings
            this.authenticationBindings = initAuthenticationBindings(policyDecisionPoint.getAuthorizationBindings(),authenticationBindingsImpl, filterConfigurationLocation, authenticationScope);
            logonProcessPermission = authenticationBindings.getLogonProcessPermission();
	    logoffPermission = authenticationBindings.getLogoffPermission();
    	
	  //initialize Provisioning Service Point
	    String provisioningServicePointImpl = (String)options.get(CoreConstants.PROVISIONING_SERVICE_POINT);
	    if(provisioningServicePointImpl==null || "".equals(provisioningServicePointImpl)){
	    	logger.info("provisioningServicePoint is not set ");
	    }else{
	    	this.provisioningServicePoint = initProvisioningServicePoint(provisioningServicePointImpl, filterConfigurationLocation);
	    	policyDecisionPoint.addAlwaysGrantedPermissionsToPolicy(provisioningServicePoint.getGrantedPermissions());
	    }
	    policyDecisionPoint.addAlwaysGrantedPermissionsToPolicy(authenticationBindings.getGrantedPermissions());
            
	    
	}
 
    
    
    public PolicyDecisionPoint getPolicyDecisionPoint() {
		return policyDecisionPoint;
	}

    public ProvisioningServicePoint getProvisioningServicePoint() {
            return provisioningServicePoint;
    }
	
    private PolicyDecisionPoint initPolicyDecisionPoint(String authorizationBindingsImpl) {
               logger.finest("initializing PolicyDecisionPoint");
               logger.finest("authorizationBindingsImpl="+authorizationBindingsImpl);
               AuthorizationBindings authorizationBindings = null;
               try {
                       authorizationBindings = (AuthorizationBindings) Thread.currentThread().getContextClassLoader().loadClass(authorizationBindingsImpl).newInstance();

               } catch(InstantiationException iex) {
                    logger.log(Level.SEVERE,iex.getMessage(),iex);
                   throw new IllegalArgumentException(iex.getMessage());
               } catch (ClassNotFoundException cne){
                    logger.log(Level.SEVERE,cne.getMessage(),cne);
                   throw new IllegalArgumentException(cne.getMessage());
               }catch(IllegalAccessException iae){
                    logger.log(Level.SEVERE,iae.getMessage(),iae);
                   throw new IllegalArgumentException(iae.getMessage());
               }

               PolicyDecisionPoint pdp = new PolicyDecisionPoint(authorizationBindings);
               return pdp;
    }
			
    private ProvisioningServicePoint initProvisioningServicePoint(String provisioningServicePointImpl,String filterConfigurationLocation){
            logger.finest("initializing ProvisioningServicePoint");
            logger.finest("provisioningServicePointImpl="+provisioningServicePointImpl);
            ProvisioningServicePoint psp = null;
            try {
                    psp = (ProvisioningServicePoint) Thread.currentThread().getContextClassLoader().loadClass(provisioningServicePointImpl).newInstance();
                    psp.init(filterConfigurationLocation);
            } catch(InstantiationException iex) {
                logger.log(Level.SEVERE, iex.getMessage(), iex);
                throw new IllegalArgumentException(iex.getMessage());
            } catch (ClassNotFoundException cne){
                logger.log(Level.SEVERE, cne.getMessage(), cne);
                throw new IllegalArgumentException(cne.getMessage());
            }catch(IllegalAccessException iae){
                    logger.log(Level.SEVERE, iae.getMessage(), iae);
                    throw new IllegalArgumentException(iae.getMessage());

            }

            return psp;
    }
        
    public static AuthenticationBindings initAuthenticationBindings(AuthorizationBindings authZBindings,String authenticationBindingsImpl,String filterConfigurationLocation,String authenticationScope) {
            if(authenticationBindingsImpl==null || "".equals(authenticationBindingsImpl)){
                    throw new IllegalArgumentException("authenticationBindingsImpl is null or empty");
            }
            if(filterConfigurationLocation==null || "".equals(filterConfigurationLocation)){
                    throw new IllegalArgumentException("filterConfigurationLocation is null or empty");
            }

            if(authenticationScope==null || "".equals(authenticationScope)){
                    throw new IllegalArgumentException("authenticationScope is null or empty");
            }

            logger.finest("initializing authenticationBindings");
            logger.finest("authenticationBindingsImpl="+authenticationBindingsImpl);
            logger.finest("filterConfigurationLocation="+filterConfigurationLocation);
            logger.finest("authenticationScope="+authenticationScope);
            AuthenticationBindings authNBindings = null;
            try {
                    Class authenticationBindingsClass = Thread.currentThread().getContextClassLoader().loadClass(authenticationBindingsImpl);
                    Class[] constructorArgumentTypes = new Class[]{AuthorizationBindings.class};
                    Constructor constructor = authenticationBindingsClass.getConstructor(constructorArgumentTypes);
                    authNBindings = (AuthenticationBindings)constructor.newInstance(new Object[]{authZBindings});
                    authNBindings.init(filterConfigurationLocation,authenticationScope);

            } catch(InstantiationException iex) {
                logger.log(Level.SEVERE, iex.getMessage(), iex);
                throw new IllegalArgumentException(iex.getMessage());
            } catch (ClassNotFoundException cne){
                logger.log(Level.SEVERE, cne.getMessage(), cne);
                throw new IllegalArgumentException(cne.getMessage());
            }catch(IllegalAccessException iae){
                logger.log(Level.SEVERE, iae.getMessage(), iae);
                throw new IllegalArgumentException(iae.getMessage());
            } catch (SecurityException e) {
                    logger.log(Level.SEVERE, e.getMessage(), e);
                    throw new IllegalArgumentException(e.getMessage());
                    } catch (NoSuchMethodException e) {
                            logger.log(Level.SEVERE, e.getMessage(), e);
                            throw new IllegalArgumentException(e.getMessage());
                    } catch (IllegalArgumentException e) {
                            logger.log(Level.SEVERE, e.getMessage(), e);
                            throw new IllegalArgumentException(e.getMessage());
                    } catch (InvocationTargetException e) {
                            logger.log(Level.SEVERE, e.getMessage(), e);
                            throw new IllegalArgumentException(e.getMessage());
                    }

            return authNBindings;
     }

    /**
     * authenticate client request according the authentication Policy defined in the 'authSchemes' field.
     * @param req
     * @param res
     * @return <code>true</code> if authentication succeeds, <code>false</code> otherwise
     * @throws IOException
     */
    private boolean authenticateAfterRegistration(AccessContext context){
                    authenticationBindings.setRequestAttribute(context,CoreConstants.REGISTRATION_DONE, Boolean.valueOf(true));
                    return AuthenticationServicePoint.authenticate(context,applicationName,authenticationBindings);

    }

	

    private void logoff(AccessContext contxt) {
        logger.finest(" logoff phase ");
        Stateful statefulAuthenticationBindings = (Stateful) authenticationBindings;
        //remove Subject from session
        AuthenticationUtils auth = (net.sf.jguard.core.authentication.AuthenticationUtils) statefulAuthenticationBindings.getSessionAttribute(contxt,CoreConstants.AUTHN_UTILS);
        if (auth != null) {
            auth.logout();
            logger.finest(" user logoff ");
        }

        statefulAuthenticationBindings.removeSessionAttribute(contxt, net.sf.jguard.core.CoreConstants.AUTHN_UTILS);

        logger.finest("doFilter() -  user logoff ");

        //we invalidate the session to unbound all objects, including subject
        try {
            statefulAuthenticationBindings.invalidateSession(contxt);
        } catch (java.lang.IllegalStateException ise) {
            logger.log(Level.SEVERE, " session is already invalidated ", ise);
        }
    }


	/**
	 * propagate the call with security information into the Thread, i.e
	 * under the hood in the JVM.
	 * @param contxt
	 * @param subject
	 */
	private void propagateWithSecurity(AccessContext contxt, Subject subject) {
		final AccessContext context = contxt;
		final Subject s = subject;

		//propagate security information into the Thread
		try {
		                Subject.doAsPrivileged(s, new PrivilegedExceptionAction() {
		                public Object run() throws IOException{
		                        //we wrap the ServletRequest to 'correct' the j2ee's JAAS handling
		                        // according to the j2se way
		                        try{
		                                policyDecisionPoint.process(context);
		                        }catch(Throwable t){
		                                        logger.log(Level.SEVERE,t.getMessage(),t);
		                                        throw new RuntimeException(t.getMessage(),t);
		                        }
		                  // the 'null' tells the SecurityManager to consider this resource access
		                  //in an isolated context, ignoring the permissions of code currently
		                  //on the execution stack.
		                  return null;
		                }
		  },null);
		  } catch (Throwable t) {
                      logger.log(Level.SEVERE,t.getMessage(),t);
		  }
	}
	       
       
        public void doFilter(AccessContext contxt, FilterChain chain) {
            	Permission permission = policyDecisionPoint.getAuthorizationBindings().getPermissionRequested(contxt);
                AuthenticationUtils authNUtils = authenticationBindings.getAuthenticationUtils(contxt);
	        Subject subject = authNUtils.getSubject();
	        //authentication
	        if(logonProcessPermission.implies(permission)){
	        	logger.finest(" authentication phase ");
	        	boolean authenticationResult = AuthenticationServicePoint.authenticate(contxt,applicationName, authenticationBindings);
	        	logger.finest(" authentication result ="+authenticationResult);
           //user is always authorize to logoff
	        }else if(authenticationBindings.isStateful() && logoffPermission.implies(permission)){
                        logoff(contxt);
	        }else if(provisioningServicePoint.getRegisterProcessPermission()!= null 
	        		&& provisioningServicePoint.getRegisterProcessPermission().implies(permission)){
	        	
	        	logger.finest(" registerProcess phase ");

	        	//before registration, we authenticate the user as a guest
	        	//=> it "hooks" loginmodules based on credentials because the user is not yet registered
	        	//but permits to execute others loginmodules like a CAPTCHA-based one and others to avoid
	        	//annoyances with robot-based mass registration
	        	AccessContext anonymizedContext = provisioningServicePoint.anonymize(contxt);
	        	boolean authenticate = AuthenticationServicePoint.authenticate(anonymizedContext,applicationName,authenticationBindings);
	        	if(authenticate== false){
	        		//we don't register the user because it fails loginModules not based on credentials
	        		return;
	        	}
	            boolean registrationSuceed = provisioningServicePoint.registerProcess(contxt);
                    if(registrationSuceed){
                        authenticateAfterRegistration(contxt);
                    }
                    
	            //we regrab the Subject because registration has created a new one
	            subject = authenticationBindings.getAuthenticationUtils(contxt).getSubject();
	        }else if(subject == null){
                        logger.finest( " subject is null  authentication phase ");
	        	logger.finest("LAST_ACCESS_DENIED_PERMISSION="+permission);
                        if(authenticationBindings.isStateful()){
	        		((Stateful)authenticationBindings).setSessionAttribute(contxt,CoreConstants.LAST_ACCESS_DENIED_PERMISSION,permission);
	        	}
	        	boolean guestAuthenticationResult = AuthenticationServicePoint.authenticateAsGuest(contxt,applicationName,authenticationBindings);
	        	
	        	
	        }
                
                //'decorate' context with authentication informations if needed
                authenticationBindings.process(contxt);
                boolean redirect = Boolean.valueOf((String)contxt.getAttribute(PolicyEnforcementPointFilter.REDIRECT)).booleanValue();
                if(!redirect){
                    propagateWithSecurity(contxt, subject);
                }
        }

}
