/*
jGuard is a security framework based on top of jaas (java authentication and authorization security).
it is written for web applications, to resolve simply, access control problems.
version $Name$
http://sourceforge.net/projects/jguard/

Copyright (C) 2004  Charles GAY

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


jGuard project home page:
http://sourceforge.net/projects/jguard/

*/
package net.sf.jguard.core.authorization.policy;

import java.io.File;
import java.security.Policy;
import java.util.Enumeration;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * utility class to deal with the {@link java.security.Policy} class.
 * @author <a href="mailto:diabolo512@users.sourceforge.net">Charles Gay</a>
 */
public class PolicyHelper {

	//well-known java policies
	public static final String GNU_JAVA_SECURITY_POLICY_FILE = "gnu.java.security.PolicyFile";
	private static final String COM_SUN_SECURITY_AUTH_POLICY_FILE = "com.sun.security.auth.PolicyFile";
	public static final String SUN_SECURITY_PROVIDER_POLICY_FILE = "sun.security.provider.PolicyFile";

	private static Logger logger = Logger.getLogger(PolicyHelper.class.getName());

	/**
	 * install the jGuardPolicy if the default policy of the platform is not
	 * a jGuardPolicy instance.
	 */
	public static void installPolicyOnJVM() {

		     Policy runtimePolicy = Policy.getPolicy();

	         //the jGuard Policy is not set as the policy provider
	        if(!(runtimePolicy.getClass().getName().equals(MultipleAppPolicy.class.getName()))){

	        	logger.log(Level.INFO,"init() -  JGuardPolicy is not set as the policy provider . the actual policy provider is '"+ runtimePolicy.getClass().getName()+"' which is different of '"+MultipleAppPolicy.class.getName()+"' ");
	        	logger.log(Level.INFO,"init() -  if you want the jGuard policy 'governs' all java applications (one choice among others described in the jGuard documentation),");
	        	logger.log(Level.INFO,"init() -  please correct the 'policy.provider' property (policy.provider=net.sf.jguard.core.JGuardPolicy) in  your 'java.security' file,");
	        	logger.log(Level.INFO,"init() -  located in this directory: "+ System.getProperty("java.home") + File.separator+ "lib"+ File.separator + "security"+ File.separator);

	            try {
	               //we set the old policy to the Sun's Policy implementation
	            	try {
						Class clazz = Class.forName(PolicyHelper.COM_SUN_SECURITY_AUTH_POLICY_FILE);
						//we have tested that the com.sun.security.auth.PolicyFile is reachable
						javax.security.auth.Policy.setPolicy((javax.security.auth.Policy)clazz.newInstance());
					} catch (ClassNotFoundException e) {
						logger.log(Level.WARNING,"com.sun.security.auth.PolicyFile is not reachable.\n we cannot set the old javax.security.auth.Policy implementation to it\n "+e.getMessage());
					}

					//give to the new JGuardPolicy the old Policy instance
	            	Policy.setPolicy(new MultipleAppPolicy(Policy.getPolicy()));

				} catch (InstantiationException e) {
					logger.log(Level.SEVERE,"init() -  Policy Implementation cannot be instantiated : InstantiationException"+e.getMessage(),e);
				} catch (IllegalAccessException e) {
					logger.log(Level.SEVERE,"init() -  Policy Implementation cannot be accessed : IllegalAccessException"+e.getMessage(),e);
				}catch(SecurityException e){
					logger.log(Level.SEVERE,"init() -  Policy Implementation cannot be defined : SecurityException . you haven't got the right to set the java policy"+e.getMessage(),e);
	            }
	        }

	        try{

	        	  logger.log(Level.FINE,"System properties : \n");
				  Properties props = System.getProperties();
				  Enumeration enumeration = props.keys();
				  while(enumeration.hasMoreElements()){
					  String key = (String)enumeration.nextElement();
					  String value = (String)props.get(key);
					  logger.log(Level.FINE,key+"="+value);
				  }

			}catch(SecurityException sex){
				logger.log(Level.WARNING,"you have not the permission to grab system properties ");
			}

	}

	 /**
	    * discover the default policy installed on the running platform.
	    * @return defaultPolicy class
	    */
	 public static Class findDefaultPolicy(){
	     //known default policies class => do you know other java.lang.security.Policy implementations?
	     String[] policies = {PolicyHelper.SUN_SECURITY_PROVIDER_POLICY_FILE,PolicyHelper.GNU_JAVA_SECURITY_POLICY_FILE};
	     Class defaultPolicyClass = null;
	     for(int i = 0; i< policies.length;i++){
	         try {
	             defaultPolicyClass = Class.forName(policies[i]);
	        } catch (ClassNotFoundException e) {
	        	logger.log(Level.FINE,"findDefaultPolicy() - " + policies[i]+ " is not the defaultPolicy class ");
	            continue;
	        }
	        logger.log(Level.FINE,"findDefaultPolicy() - " + policies[i]+ " is the defaultPolicy class ");
	        break;
	     }
	     if(null == defaultPolicyClass){
	    	 logger.log(Level.FINE,"findDefaultPolicy() -  no defaultPolicy class has been found ");
	     }
	     return defaultPolicyClass;
	 }


}
