/*
 jGuard is a security framework based on top of jaas (java authentication and authorization security).
 it is written for web applications, to resolve simply, access control problems.
 version $Name$
 http://sourceforge.net/projects/jguard/

 Copyright (C) 2004  Charles GAY

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


 jGuard project home page:
 http://sourceforge.net/projects/jguard/

 */
package net.sf.jguard.core.authorization.policy;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.security.CodeSource;
import java.security.Permission;
import java.security.PermissionCollection;
import java.security.Permissions;
import java.security.Policy;
import java.security.ProtectionDomain;
import java.util.Enumeration;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;

import net.sf.jguard.core.authorization.manager.PermissionProvider;


/**
 * JGuard Policy abstract implementation.
 * @see net.sf.jguard.core.authorization.policy.AbstractMultipleAppPolicy
 * @see net.sf.jguard.core.authorization.policy.MultipleAppPolicy
 * @see net.sf.jguard.ext.authorization.policy.classic.SingleAppPolicy
 * @author <a href="mailto:diabolo512@users.sourceforge.net">Charles Gay</a>
 * @author <a href="mailto:vberetti@users.sourceforge.net">Vincent Beretti</a>
 */
public abstract class JGuardPolicy extends java.security.Policy {

	private static final String LIB = "lib";
	private static final String SECURITY = "security";
	private static final String J_GUARD_POLICY = "jGuard.policy";
	private static final String JGUARD_POLICY_LOCATION = File.separator + JGuardPolicy.LIB + File.separator + JGuardPolicy.SECURITY + File.separator + JGuardPolicy.J_GUARD_POLICY;
	private static final String DEFAULT_POLICY = "defaultPolicy";
	private static final String JAVA_HOME = "java.home";
	//old Policy instance replaced by JGuardPolicy
	protected static Policy defaultPolicy;
	//old Policy instance Class replaced by JGuardPolicy
	private static Class policyClass;
	private static Logger logger = Logger.getLogger(JGuardPolicy.class.getName());
	protected final static String version = "1.1.0 beta 3";
	
	
	/**
	 * default constructor.
	 */
	public JGuardPolicy(){
		super();
	}
	
	/**
	 * load the default Policy implementation class.
	 */
	protected void loadDefaultPolicy() {
		//the securityManager is not set
		if (System.getSecurityManager() == null) {
			String javaHome = System.getProperty(JGuardPolicy.JAVA_HOME);
			Properties props = new Properties();
			String defPolicy = null;

			try {
				props.load(new FileInputStream(new File(javaHome + JGuardPolicy.JGUARD_POLICY_LOCATION)));
				defPolicy = props.getProperty(JGuardPolicy.DEFAULT_POLICY);
			} catch (FileNotFoundException e) {
				logger.log(Level.CONFIG, "loadDefaultPolicy() -  jGuard.policy is not found " + e.getMessage());
			} catch (IOException e) {
				logger.log(Level.CONFIG, "loadDefaultPolicy() -  jGuard.policy is not reachable " + e.getMessage());
			}

			try {

				if(defPolicy==null){
					logger.log(Level.CONFIG,"loadDefaultPolicy() -  'defaultPolicy' field in the jGuard.Policy file is not defined ");
					logger.log(Level.CONFIG,"loadDefaultPolicy() -  jGuard try to discover the default one ");
					// we search the default policy class
					policyClass = PolicyHelper.findDefaultPolicy();
				} else {
					// we use the defined default policy class
					policyClass = Class.forName(defPolicy);
				}
			} catch (ClassNotFoundException e1) {
				logger.log(Level.CONFIG, "loadDefaultPolicy() - the default policy class cannot be found " + e1.getMessage());
			}

			//the securityManager is set
		}else{
			policyClass = PolicyHelper.findDefaultPolicy();
		}

		try {
			defaultPolicy = (Policy)policyClass.newInstance();
		} catch (InstantiationException e2) {
			logger.log(Level.CONFIG,"loadDefaultPolicy() - the default policy class cannot be instantiated"
					+ e2.getMessage());
		} catch (IllegalAccessException e2) {
			logger.log(Level.CONFIG,"loadDefaultPolicy() - the default policy class cannot be accessed "
					+ e2.getMessage());
		}
	}

	/**
	 * JGuard Policy act as a wrapper for this method.
	 * it delegates to default's Policy implementation defined in Jguard.policy file, this method.
	 * @see java.security.Policy#getPermissions(java.security.CodeSource)
	 * @param codesource
	 * @return all the permissions own by the CodeSource
	 */
	public PermissionCollection getPermissions(CodeSource codesource) {
		PermissionCollection permColl = defaultPolicy.getPermissions(codesource);
		return permColl;
	}

	public abstract PermissionCollection getPermissions(ProtectionDomain protectionDomain);

	public abstract void refresh();

	protected PermissionCollection getPermissions(ProtectionDomain protectionDomain, PermissionProvider permissionProvider) {
		PermissionCollection pc = null;
		if(System.getSecurityManager()!=null){
	        pc = defaultPolicy.getPermissions(protectionDomain);
	    }
	
	    //if this protection domain is protected by jGuard
		if(permissionProvider!=null){
			//retrieve permissions from roles owned by the user which are active
			//and resolve regexp in permissions
	    	PermissionCollection pc2= permissionProvider.getPermissions(protectionDomain);
	    	
	        //the SecurityManager is set,we merge the default permissionCollection and the permissionCollection returned by jGuard
	        if(System.getSecurityManager()!=null){
	            Enumeration enumeration = pc2.elements();
	            while(enumeration.hasMoreElements()){
	        	    pc.add((Permission)enumeration.nextElement());
	            }
	        }else{
	            //there is no SecurityManager set
	            //we return only the permissionCollection obtained by jGuard
	            pc = pc2;
	        }
	    }
	
	    return pc;
	}

	public abstract  void addAlwaysGrantedPermissions(ClassLoader cl,Permissions alwaysGrantedPermissions);
		
	
}
