/*
jGuard is a security framework based on top of jaas (java authentication and authorization security).
it is written for web applications, to resolve simply, access control problems.
version $Name$
http://sourceforge.net/projects/jguard/

Copyright (C) 2004  Charles GAY

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


jGuard project home page:
http://sourceforge.net/projects/jguard/

*/
package net.sf.jguard.core.authorization.policy;

import java.security.AccessControlException;
import java.security.AccessController;
import java.security.Permission;
import java.security.Policy;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.security.auth.Subject;



/**
 * Facade class used to bound control check to either the legacy {@link AccessController} bound to the global JVM security,
 *  or to the {@link LocalAccessController} 'isolated'.
 * @author <a href="mailto:diabolo512@users.sourceforge.net">Charles Gay</a>
 * @author <a href="mailto:vberetti@users.sourceforge.net">Vincent Beretti</a>
 * @see java.security.AccessControlContext
 * @see java.security.ProtectionDomain
 */
public class AccessControllerUtils {
	private static final Logger logger = Logger.getLogger(AccessControllerUtils.class.getName());
	private static LocalAccessController accessController = null;
	
	public AccessControllerUtils(){
		
	}
	
	public AccessControllerUtils(Policy policy){
		if(policy == null){
			throw new IllegalArgumentException(" policy is null ");
		}
		if(accessController == null){
			accessController = new LocalAccessController(policy) ;
		}
	}
	
        
        public  static boolean hasPermission(Subject subj, final Permission p){
            try{
                checkPermission(subj,p);
            }catch(Exception ex){
                return false;
            }
            
            return true;
        }
        
	/**
	 * check if the {@link Subject} has got the permission. 
	 * @param subj
	 * @param p permission to check 
	 * @throws PrivilegedActionException
	 * @throws AccessControlException 
	 */
	public static void checkPermission(Subject subj, final Permission p) throws AccessControlException,PrivilegedActionException {

	      try {
	          Subject.doAs(subj, new PrivilegedExceptionAction() {
                          public Object run() {
                              if(accessController == null){
                              AccessController.checkPermission(p);
                              }else{
                                      accessController.checkPermission(p);
                              }
                              // the 'null' tells the SecurityManager to consider this resource access
                              //in an isolated context, ignoring the permissions of code currently
                              //on the execution stack.
                              return null;
                          }
	              });
	          
	      } catch (AccessControlException ace) {
	    	  if(logger.isLoggable(Level.FINEST)){
	    		  logger.log(Level.FINEST,"AccessControlException ",ace);
	    	  }
	    	  throw ace;
	      } catch (PrivilegedActionException pae){
	    	  if(logger.isLoggable(Level.FINEST)){
	    		  logger.log(Level.FINEST,"PrivilegedActionException ",pae);
	    	  }
	    	  throw pae;
	      }
	      
	      if(logger.isLoggable(Level.FINEST)){
    		  logger.log(Level.FINEST,"user has got the permission ",p);
    	  }
	  }

	/**
	 * {@link LocalAccessController} returned can be <strong>null</strong> in <i>JVM</i> mode.
	 * @return
	 */
	public static LocalAccessController getAccessController() {
		return accessController;
	}
	
	/**
	 * return the {@link java.security.Policy} used to control access.
	 * it is either the current Policy in place in the JVM (<i>JVM scope</i>), or the Policy bound to the
	 * {@link LocalAccessController} bound to this AccessControllerUtils.
	 * in this case,like it is a static method, this Policy is bound to the current classloader.
	 * in a JEE context, each webapp has its own isloated classloader, so its own policy in this 
	 * <i>local scope</i>.
	 * @return Policy
	 */
	public static Policy getPolicy() {
		if(accessController==null){
			return Policy.getPolicy();
		}else{
			return accessController.getPolicy();
		}
	}
	
}
